/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Input for CreateTopic action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTopicRequest extends SnsRequest implements
        ToCopyableBuilder<CreateTopicRequest.Builder, CreateTopicRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateTopicRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(CreateTopicRequest::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateTopicRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DATA_PROTECTION_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataProtectionPolicy").getter(getter(CreateTopicRequest::dataProtectionPolicy))
            .setter(setter(Builder::dataProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataProtectionPolicy").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ATTRIBUTES_FIELD,
            TAGS_FIELD, DATA_PROTECTION_POLICY_FIELD));

    private final String name;

    private final Map<String, String> attributes;

    private final List<Tag> tags;

    private final String dataProtectionPolicy;

    private CreateTopicRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.attributes = builder.attributes;
        this.tags = builder.tags;
        this.dataProtectionPolicy = builder.dataProtectionPolicy;
    }

    /**
     * <p>
     * The name of the topic you want to create.
     * </p>
     * <p>
     * Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores, and
     * hyphens, and must be between 1 and 256 characters long.
     * </p>
     * <p>
     * For a FIFO (first-in-first-out) topic, the name must end with the <code>.fifo</code> suffix.
     * </p>
     * 
     * @return The name of the topic you want to create.</p>
     *         <p>
     *         Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers,
     *         underscores, and hyphens, and must be between 1 and 256 characters long.
     *         </p>
     *         <p>
     *         For a FIFO (first-in-first-out) topic, the name must end with the <code>.fifo</code> suffix.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attributes with their corresponding values.
     * </p>
     * <p>
     * The following lists the names, descriptions, and values of the special request parameters that the
     * <code>CreateTopic</code> action uses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     * endpoints.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FifoTopic</code> – Set to true to create a FIFO topic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner can
     * publish or subscribe to the topic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SignatureVersion</code> – The signature version corresponds to the hashing algorithm used while creating
     * the signature of the notifications, subscription confirmations, or unsubscribe confirmation messages sent by
     * Amazon SNS. By default, <code>SignatureVersion</code> is set to 1.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attribute applies only to <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side encryption</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon SNS
     * or a custom CMK. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key Terms</a>. For
     * more examples, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     * >KeyId</a> in the <i>Key Management Service API Reference</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attributes apply only to <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-fifo-topics.html">FIFO topics</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FifoTopic</code> – When this is set to <code>true</code>, a FIFO topic is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ContentBasedDeduplication</code> – Enables content-based deduplication for FIFO topics.
     * </p>
     * <ul>
     * <li>
     * <p>
     * By default, <code>ContentBasedDeduplication</code> is set to <code>false</code>. If you create a FIFO topic and
     * this attribute is <code>false</code>, you must specify a value for the <code>MessageDeduplicationId</code>
     * parameter for the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a> action.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you set <code>ContentBasedDeduplication</code> to <code>true</code>, Amazon SNS uses a SHA-256 hash to
     * generate the <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     * message).
     * </p>
     * <p>
     * (Optional) To override the generated value, you can specify a value for the <code>MessageDeduplicationId</code>
     * parameter for the <code>Publish</code> action.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A map of attributes with their corresponding values.</p>
     *         <p>
     *         The following lists the names, descriptions, and values of the special request parameters that the
     *         <code>CreateTopic</code> action uses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     *         endpoints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FifoTopic</code> – Set to true to create a FIFO topic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner
     *         can publish or subscribe to the topic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SignatureVersion</code> – The signature version corresponds to the hashing algorithm used while
     *         creating the signature of the notifications, subscription confirmations, or unsubscribe confirmation
     *         messages sent by Amazon SNS. By default, <code>SignatureVersion</code> is set to 1.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attribute applies only to <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side
     *         encryption</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
     *         Amazon SNS or a custom CMK. For more information, see <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key
     *         Terms</a>. For more examples, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     *         >KeyId</a> in the <i>Key Management Service API Reference</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attributes apply only to <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-fifo-topics.html">FIFO topics</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FifoTopic</code> – When this is set to <code>true</code>, a FIFO topic is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ContentBasedDeduplication</code> – Enables content-based deduplication for FIFO topics.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         By default, <code>ContentBasedDeduplication</code> is set to <code>false</code>. If you create a FIFO
     *         topic and this attribute is <code>false</code>, you must specify a value for the
     *         <code>MessageDeduplicationId</code> parameter for the <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a> action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When you set <code>ContentBasedDeduplication</code> to <code>true</code>, Amazon SNS uses a SHA-256 hash
     *         to generate the <code>MessageDeduplicationId</code> using the body of the message (but not the attributes
     *         of the message).
     *         </p>
     *         <p>
     *         (Optional) To override the generated value, you can specify a value for the
     *         <code>MessageDeduplicationId</code> parameter for the <code>Publish</code> action.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags to add to a new topic.
     * </p>
     * <note>
     * <p>
     * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
     * <code>sns:TagResource</code> permissions.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tags to add to a new topic.</p> <note>
     *         <p>
     *         To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
     *         <code>sns:TagResource</code> permissions.
     *         </p>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The body of the policy document you want to use for this topic.
     * </p>
     * <p>
     * You can only add one policy per topic.
     * </p>
     * <p>
     * The policy must be in JSON string format.
     * </p>
     * <p>
     * Length Constraints: Maximum length of 30,720.
     * </p>
     * 
     * @return The body of the policy document you want to use for this topic.</p>
     *         <p>
     *         You can only add one policy per topic.
     *         </p>
     *         <p>
     *         The policy must be in JSON string format.
     *         </p>
     *         <p>
     *         Length Constraints: Maximum length of 30,720.
     */
    public final String dataProtectionPolicy() {
        return dataProtectionPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataProtectionPolicy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTopicRequest)) {
            return false;
        }
        CreateTopicRequest other = (CreateTopicRequest) obj;
        return Objects.equals(name(), other.name()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(dataProtectionPolicy(), other.dataProtectionPolicy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateTopicRequest").add("Name", name())
                .add("Attributes", hasAttributes() ? attributes() : null).add("Tags", hasTags() ? tags() : null)
                .add("DataProtectionPolicy", dataProtectionPolicy()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "DataProtectionPolicy":
            return Optional.ofNullable(clazz.cast(dataProtectionPolicy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateTopicRequest, T> g) {
        return obj -> g.apply((CreateTopicRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SnsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateTopicRequest> {
        /**
         * <p>
         * The name of the topic you want to create.
         * </p>
         * <p>
         * Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores,
         * and hyphens, and must be between 1 and 256 characters long.
         * </p>
         * <p>
         * For a FIFO (first-in-first-out) topic, the name must end with the <code>.fifo</code> suffix.
         * </p>
         * 
         * @param name
         *        The name of the topic you want to create.</p>
         *        <p>
         *        Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers,
         *        underscores, and hyphens, and must be between 1 and 256 characters long.
         *        </p>
         *        <p>
         *        For a FIFO (first-in-first-out) topic, the name must end with the <code>.fifo</code> suffix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A map of attributes with their corresponding values.
         * </p>
         * <p>
         * The following lists the names, descriptions, and values of the special request parameters that the
         * <code>CreateTopic</code> action uses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
         * endpoints.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FifoTopic</code> – Set to true to create a FIFO topic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner can
         * publish or subscribe to the topic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SignatureVersion</code> – The signature version corresponds to the hashing algorithm used while
         * creating the signature of the notifications, subscription confirmations, or unsubscribe confirmation messages
         * sent by Amazon SNS. By default, <code>SignatureVersion</code> is set to 1.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attribute applies only to <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side encryption</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for Amazon
         * SNS or a custom CMK. For more information, see <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key Terms</a>.
         * For more examples, see <a href=
         * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         * >KeyId</a> in the <i>Key Management Service API Reference</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attributes apply only to <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/sns-fifo-topics.html">FIFO topics</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FifoTopic</code> – When this is set to <code>true</code>, a FIFO topic is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ContentBasedDeduplication</code> – Enables content-based deduplication for FIFO topics.
         * </p>
         * <ul>
         * <li>
         * <p>
         * By default, <code>ContentBasedDeduplication</code> is set to <code>false</code>. If you create a FIFO topic
         * and this attribute is <code>false</code>, you must specify a value for the
         * <code>MessageDeduplicationId</code> parameter for the <a
         * href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a> action.
         * </p>
         * </li>
         * <li>
         * <p>
         * When you set <code>ContentBasedDeduplication</code> to <code>true</code>, Amazon SNS uses a SHA-256 hash to
         * generate the <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         * message).
         * </p>
         * <p>
         * (Optional) To override the generated value, you can specify a value for the
         * <code>MessageDeduplicationId</code> parameter for the <code>Publish</code> action.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param attributes
         *        A map of attributes with their corresponding values.</p>
         *        <p>
         *        The following lists the names, descriptions, and values of the special request parameters that the
         *        <code>CreateTopic</code> action uses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to
         *        HTTP/S endpoints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FifoTopic</code> – Set to true to create a FIFO topic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic
         *        owner can publish or subscribe to the topic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SignatureVersion</code> – The signature version corresponds to the hashing algorithm used while
         *        creating the signature of the notifications, subscription confirmations, or unsubscribe confirmation
         *        messages sent by Amazon SNS. By default, <code>SignatureVersion</code> is set to 1.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attribute applies only to <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side
         *        encryption</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KmsMasterKeyId</code> – The ID of an Amazon Web Services managed customer master key (CMK) for
         *        Amazon SNS or a custom CMK. For more information, see <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key
         *        Terms</a>. For more examples, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         *        >KeyId</a> in the <i>Key Management Service API Reference</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attributes apply only to <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-fifo-topics.html">FIFO topics</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FifoTopic</code> – When this is set to <code>true</code>, a FIFO topic is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ContentBasedDeduplication</code> – Enables content-based deduplication for FIFO topics.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        By default, <code>ContentBasedDeduplication</code> is set to <code>false</code>. If you create a FIFO
         *        topic and this attribute is <code>false</code>, you must specify a value for the
         *        <code>MessageDeduplicationId</code> parameter for the <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When you set <code>ContentBasedDeduplication</code> to <code>true</code>, Amazon SNS uses a SHA-256
         *        hash to generate the <code>MessageDeduplicationId</code> using the body of the message (but not the
         *        attributes of the message).
         *        </p>
         *        <p>
         *        (Optional) To override the generated value, you can specify a value for the
         *        <code>MessageDeduplicationId</code> parameter for the <code>Publish</code> action.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note>
         * 
         * @param tags
         *        The list of tags to add to a new topic.</p> <note>
         *        <p>
         *        To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         *        <code>sns:TagResource</code> permissions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note>
         * 
         * @param tags
         *        The list of tags to add to a new topic.</p> <note>
         *        <p>
         *        To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         *        <code>sns:TagResource</code> permissions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sns.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.sns.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.sns.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.sns.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The body of the policy document you want to use for this topic.
         * </p>
         * <p>
         * You can only add one policy per topic.
         * </p>
         * <p>
         * The policy must be in JSON string format.
         * </p>
         * <p>
         * Length Constraints: Maximum length of 30,720.
         * </p>
         * 
         * @param dataProtectionPolicy
         *        The body of the policy document you want to use for this topic.</p>
         *        <p>
         *        You can only add one policy per topic.
         *        </p>
         *        <p>
         *        The policy must be in JSON string format.
         *        </p>
         *        <p>
         *        Length Constraints: Maximum length of 30,720.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataProtectionPolicy(String dataProtectionPolicy);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SnsRequest.BuilderImpl implements Builder {
        private String name;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String dataProtectionPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTopicRequest model) {
            super(model);
            name(model.name);
            attributes(model.attributes);
            tags(model.tags);
            dataProtectionPolicy(model.dataProtectionPolicy);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = TopicAttributesMapCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = TopicAttributesMapCopier.copy(attributes);
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDataProtectionPolicy() {
            return dataProtectionPolicy;
        }

        public final void setDataProtectionPolicy(String dataProtectionPolicy) {
            this.dataProtectionPolicy = dataProtectionPolicy;
        }

        @Override
        public final Builder dataProtectionPolicy(String dataProtectionPolicy) {
            this.dataProtectionPolicy = dataProtectionPolicy;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTopicRequest build() {
            return new CreateTopicRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
