/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.sns.internal.SnsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.sns.model.AddPermissionRequest;
import software.amazon.awssdk.services.sns.model.AddPermissionResponse;
import software.amazon.awssdk.services.sns.model.AuthorizationErrorException;
import software.amazon.awssdk.services.sns.model.BatchEntryIdsNotDistinctException;
import software.amazon.awssdk.services.sns.model.BatchRequestTooLongException;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutRequest;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ConcurrentAccessException;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionRequest;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointResponse;
import software.amazon.awssdk.services.sns.model.CreateSmsSandboxPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.CreateSmsSandboxPhoneNumberResponse;
import software.amazon.awssdk.services.sns.model.CreateTopicRequest;
import software.amazon.awssdk.services.sns.model.CreateTopicResponse;
import software.amazon.awssdk.services.sns.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.sns.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.DeleteSmsSandboxPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.DeleteSmsSandboxPhoneNumberResponse;
import software.amazon.awssdk.services.sns.model.DeleteTopicRequest;
import software.amazon.awssdk.services.sns.model.DeleteTopicResponse;
import software.amazon.awssdk.services.sns.model.EmptyBatchRequestException;
import software.amazon.awssdk.services.sns.model.EndpointDisabledException;
import software.amazon.awssdk.services.sns.model.FilterPolicyLimitExceededException;
import software.amazon.awssdk.services.sns.model.GetDataProtectionPolicyRequest;
import software.amazon.awssdk.services.sns.model.GetDataProtectionPolicyResponse;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSmsSandboxAccountStatusRequest;
import software.amazon.awssdk.services.sns.model.GetSmsSandboxAccountStatusResponse;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.InternalErrorException;
import software.amazon.awssdk.services.sns.model.InvalidBatchEntryIdException;
import software.amazon.awssdk.services.sns.model.InvalidParameterException;
import software.amazon.awssdk.services.sns.model.InvalidParameterValueException;
import software.amazon.awssdk.services.sns.model.InvalidSecurityException;
import software.amazon.awssdk.services.sns.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.sns.model.KmsDisabledException;
import software.amazon.awssdk.services.sns.model.KmsInvalidStateException;
import software.amazon.awssdk.services.sns.model.KmsNotFoundException;
import software.amazon.awssdk.services.sns.model.KmsOptInRequiredException;
import software.amazon.awssdk.services.sns.model.KmsThrottlingException;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.ListOriginationNumbersRequest;
import software.amazon.awssdk.services.sns.model.ListOriginationNumbersResponse;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutRequest;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse;
import software.amazon.awssdk.services.sns.model.ListSmsSandboxPhoneNumbersRequest;
import software.amazon.awssdk.services.sns.model.ListSmsSandboxPhoneNumbersResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sns.model.ListTopicsRequest;
import software.amazon.awssdk.services.sns.model.ListTopicsResponse;
import software.amazon.awssdk.services.sns.model.NotFoundException;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberResponse;
import software.amazon.awssdk.services.sns.model.OptedOutException;
import software.amazon.awssdk.services.sns.model.PlatformApplicationDisabledException;
import software.amazon.awssdk.services.sns.model.PublishBatchRequest;
import software.amazon.awssdk.services.sns.model.PublishBatchResponse;
import software.amazon.awssdk.services.sns.model.PublishRequest;
import software.amazon.awssdk.services.sns.model.PublishResponse;
import software.amazon.awssdk.services.sns.model.PutDataProtectionPolicyRequest;
import software.amazon.awssdk.services.sns.model.PutDataProtectionPolicyResponse;
import software.amazon.awssdk.services.sns.model.RemovePermissionRequest;
import software.amazon.awssdk.services.sns.model.RemovePermissionResponse;
import software.amazon.awssdk.services.sns.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.SnsException;
import software.amazon.awssdk.services.sns.model.StaleTagException;
import software.amazon.awssdk.services.sns.model.SubscribeRequest;
import software.amazon.awssdk.services.sns.model.SubscribeResponse;
import software.amazon.awssdk.services.sns.model.SubscriptionLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagPolicyException;
import software.amazon.awssdk.services.sns.model.TagResourceRequest;
import software.amazon.awssdk.services.sns.model.TagResourceResponse;
import software.amazon.awssdk.services.sns.model.ThrottledException;
import software.amazon.awssdk.services.sns.model.TooManyEntriesInBatchRequestException;
import software.amazon.awssdk.services.sns.model.TopicLimitExceededException;
import software.amazon.awssdk.services.sns.model.UnsubscribeRequest;
import software.amazon.awssdk.services.sns.model.UnsubscribeResponse;
import software.amazon.awssdk.services.sns.model.UntagResourceRequest;
import software.amazon.awssdk.services.sns.model.UntagResourceResponse;
import software.amazon.awssdk.services.sns.model.UserErrorException;
import software.amazon.awssdk.services.sns.model.ValidationException;
import software.amazon.awssdk.services.sns.model.VerificationException;
import software.amazon.awssdk.services.sns.model.VerifySmsSandboxPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.VerifySmsSandboxPhoneNumberResponse;
import software.amazon.awssdk.services.sns.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CheckIfPhoneNumberIsOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ConfirmSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreateSmsSandboxPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreateTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeletePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteSmsSandboxPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetDataProtectionPolicyRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSmsSandboxAccountStatusRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListEndpointsByPlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListOriginationNumbersRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPhoneNumbersOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPlatformApplicationsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSmsSandboxPhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsByTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTopicsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.OptInPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.PublishBatchRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.PublishRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.PutDataProtectionPolicyRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.VerifySmsSandboxPhoneNumberRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SnsAsyncClient}.
 *
 * @see SnsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnsAsyncClient implements SnsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSnsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.QUERY).build();

    private final AsyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SnsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSnsAsyncClient(SnsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init();
    }

    /**
     * <p>
     * Adds a statement to a topic's access control policy, granting access for the specified Amazon Web Services
     * accounts to the specified actions.
     * </p>
     * <note>
     * <p>
     * To remove the ability to change topic permissions, you must deny permissions to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetTopicAttributes</code> actions in your IAM policy.
     * </p>
     * </note>
     *
     * @param addPermissionRequest
     * @return A Java Future containing the result of the AddPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.AddPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddPermissionResponse> addPermission(AddPermissionRequest addPermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addPermissionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddPermission");

            HttpResponseHandler<AddPermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(AddPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AddPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                            .withOperationName("AddPermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addPermissionRequest));
            CompletableFuture<AddPermissionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your
     * Amazon Web Services account. You cannot send SMS messages to a number that is opted out.
     * </p>
     * <p>
     * To resume sending messages, you can opt in the number by using the <code>OptInPhoneNumber</code> action.
     * </p>
     *
     * @param checkIfPhoneNumberIsOptedOutRequest
     *        The input for the <code>CheckIfPhoneNumberIsOptedOut</code> action.
     * @return A Java Future containing the result of the CheckIfPhoneNumberIsOptedOut operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CheckIfPhoneNumberIsOptedOut
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CheckIfPhoneNumberIsOptedOut"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckIfPhoneNumberIsOptedOutResponse> checkIfPhoneNumberIsOptedOut(
            CheckIfPhoneNumberIsOptedOutRequest checkIfPhoneNumberIsOptedOutRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(checkIfPhoneNumberIsOptedOutRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkIfPhoneNumberIsOptedOutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckIfPhoneNumberIsOptedOut");

            HttpResponseHandler<CheckIfPhoneNumberIsOptedOutResponse> responseHandler = protocolFactory
                    .createResponseHandler(CheckIfPhoneNumberIsOptedOutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CheckIfPhoneNumberIsOptedOutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckIfPhoneNumberIsOptedOutRequest, CheckIfPhoneNumberIsOptedOutResponse>()
                            .withOperationName("CheckIfPhoneNumberIsOptedOut").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CheckIfPhoneNumberIsOptedOutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(checkIfPhoneNumberIsOptedOutRequest));
            CompletableFuture<CheckIfPhoneNumberIsOptedOutResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an
     * earlier <code>Subscribe</code> action. If the token is valid, the action creates a new subscription and returns
     * its Amazon Resource Name (ARN). This call requires an AWS signature only when the
     * <code>AuthenticateOnUnsubscribe</code> flag is set to "true".
     * </p>
     *
     * @param confirmSubscriptionRequest
     *        Input for ConfirmSubscription action.
     * @return A Java Future containing the result of the ConfirmSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubscriptionLimitExceededException Indicates that the customer already owns the maximum allowed
     *         number of subscriptions.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your Amazon Web
     *         Services account exceeds the limit. To add more filter polices, submit an Amazon SNS Limit Increase case
     *         in the Amazon Web Services Support Center.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ConfirmSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ConfirmSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ConfirmSubscriptionResponse> confirmSubscription(
            ConfirmSubscriptionRequest confirmSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(confirmSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, confirmSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfirmSubscription");

            HttpResponseHandler<ConfirmSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(ConfirmSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ConfirmSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfirmSubscriptionRequest, ConfirmSubscriptionResponse>()
                            .withOperationName("ConfirmSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ConfirmSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(confirmSubscriptionRequest));
            CompletableFuture<ConfirmSubscriptionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a platform application object for one of the supported push notification services, such as APNS and GCM
     * (Firebase Cloud Messaging), to which devices and mobile apps may register. You must specify
     * <code>PlatformPrincipal</code> and <code>PlatformCredential</code> attributes when using the
     * <code>CreatePlatformApplication</code> action.
     * </p>
     * <p>
     * <code>PlatformPrincipal</code> and <code>PlatformCredential</code> are received from the notification service.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>ADM</code>, <code>PlatformPrincipal</code> is <code>client id</code> and
     * <code>PlatformCredential</code> is <code>client secret</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>Baidu</code>, <code>PlatformPrincipal</code> is <code>API key</code> and
     * <code>PlatformCredential</code> is <code>secret key</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>APNS</code> and <code>APNS_SANDBOX</code> using certificate credentials, <code>PlatformPrincipal</code>
     * is <code>SSL certificate</code> and <code>PlatformCredential</code> is <code>private key</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>APNS</code> and <code>APNS_SANDBOX</code> using token credentials, <code>PlatformPrincipal</code> is
     * <code>signing key ID</code> and <code>PlatformCredential</code> is <code>signing key</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>GCM</code> (Firebase Cloud Messaging), there is no <code>PlatformPrincipal</code> and the
     * <code>PlatformCredential</code> is <code>API key</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>MPNS</code>, <code>PlatformPrincipal</code> is <code>TLS certificate</code> and
     * <code>PlatformCredential</code> is <code>private key</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>WNS</code>, <code>PlatformPrincipal</code> is <code>Package Security Identifier</code> and
     * <code>PlatformCredential</code> is <code>secret key</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can use the returned <code>PlatformApplicationArn</code> as an attribute for the
     * <code>CreatePlatformEndpoint</code> action.
     * </p>
     *
     * @param createPlatformApplicationRequest
     *        Input for CreatePlatformApplication action.
     * @return A Java Future containing the result of the CreatePlatformApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreatePlatformApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlatformApplicationResponse> createPlatformApplication(
            CreatePlatformApplicationRequest createPlatformApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPlatformApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlatformApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlatformApplication");

            HttpResponseHandler<CreatePlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreatePlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreatePlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlatformApplicationRequest, CreatePlatformApplicationResponse>()
                            .withOperationName("CreatePlatformApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPlatformApplicationRequest));
            CompletableFuture<CreatePlatformApplicationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM
     * (Firebase Cloud Messaging) and APNS. <code>CreatePlatformEndpoint</code> requires the
     * <code>PlatformApplicationArn</code> that is returned from <code>CreatePlatformApplication</code>. You can use the
     * returned <code>EndpointArn</code> to send a message to a mobile app or by the <code>Subscribe</code> action for
     * subscription to a topic. The <code>CreatePlatformEndpoint</code> action is idempotent, so if the requester
     * already owns an endpoint with the same device token and attributes, that endpoint's ARN is returned without
     * creating a new endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * When using <code>CreatePlatformEndpoint</code> with Baidu, two attributes must be provided: ChannelId and UserId.
     * The token field must also contain the ChannelId. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePushBaiduEndpoint.html">Creating an Amazon SNS Endpoint
     * for Baidu</a>.
     * </p>
     *
     * @param createPlatformEndpointRequest
     *        Input for CreatePlatformEndpoint action.
     * @return A Java Future containing the result of the CreatePlatformEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreatePlatformEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlatformEndpointResponse> createPlatformEndpoint(
            CreatePlatformEndpointRequest createPlatformEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPlatformEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlatformEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlatformEndpoint");

            HttpResponseHandler<CreatePlatformEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreatePlatformEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreatePlatformEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlatformEndpointRequest, CreatePlatformEndpointResponse>()
                            .withOperationName("CreatePlatformEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePlatformEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPlatformEndpointRequest));
            CompletableFuture<CreatePlatformEndpointResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a destination phone number to an Amazon Web Services account in the SMS sandbox and sends a one-time
     * password (OTP) to that phone number.
     * </p>
     * <p>
     * When you start using Amazon SNS to send SMS messages, your Amazon Web Services account is in the <i>SMS
     * sandbox</i>. The SMS sandbox provides a safe environment for you to try Amazon SNS features without risking your
     * reputation as an SMS sender. While your Amazon Web Services account is in the SMS sandbox, you can use all of the
     * features of Amazon SNS. However, you can send SMS messages only to verified destination phone numbers. For more
     * information, including how to move out of the sandbox to send messages without restrictions, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-sms-sandbox.html">SMS sandbox</a> in the <i>Amazon SNS
     * Developer Guide</i>.
     * </p>
     *
     * @param createSmsSandboxPhoneNumberRequest
     * @return A Java Future containing the result of the CreateSMSSandboxPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>OptedOutException Indicates that the specified phone number opted out of receiving SMS messages from
     *         your Amazon Web Services account. You can't send SMS messages to phone numbers that opt out.</li>
     *         <li>UserErrorException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreateSMSSandboxPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreateSMSSandboxPhoneNumber"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSmsSandboxPhoneNumberResponse> createSMSSandboxPhoneNumber(
            CreateSmsSandboxPhoneNumberRequest createSmsSandboxPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSmsSandboxPhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSmsSandboxPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSMSSandboxPhoneNumber");

            HttpResponseHandler<CreateSmsSandboxPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateSmsSandboxPhoneNumberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateSmsSandboxPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSmsSandboxPhoneNumberRequest, CreateSmsSandboxPhoneNumberResponse>()
                            .withOperationName("CreateSMSSandboxPhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSmsSandboxPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSmsSandboxPhoneNumberRequest));
            CompletableFuture<CreateSmsSandboxPhoneNumberResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a topic to which notifications can be published. Users can create at most 100,000 standard topics (at
     * most 1,000 FIFO topics). For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-create-topic.html">Creating an Amazon SNS topic</a> in the
     * <i>Amazon SNS Developer Guide</i>. This action is idempotent, so if the requester already owns a topic with the
     * specified name, that topic's ARN is returned without creating a new topic.
     * </p>
     *
     * @param createTopicRequest
     *        Input for CreateTopic action.
     * @return A Java Future containing the result of the CreateTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>TopicLimitExceededException Indicates that the customer already owns the maximum allowed number of
     *         topics.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreateTopic
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreateTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTopicResponse> createTopic(CreateTopicRequest createTopicRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTopicRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTopicRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTopic");

            HttpResponseHandler<CreateTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTopicRequest, CreateTopicResponse>()
                            .withOperationName("CreateTopic").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTopicRequest));
            CompletableFuture<CreateTopicResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the endpoint for a device and mobile app from Amazon SNS. This action is idempotent. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile
     * Push Notifications</a>.
     * </p>
     * <p>
     * When you delete an endpoint that is also subscribed to a topic, then you must also unsubscribe the endpoint from
     * the topic.
     * </p>
     *
     * @param deleteEndpointRequest
     *        Input for DeleteEndpoint action.
     * @return A Java Future containing the result of the DeleteEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEndpointResponse> deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEndpointRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEndpoint");

            HttpResponseHandler<DeleteEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEndpointRequest, DeleteEndpointResponse>()
                            .withOperationName("DeleteEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEndpointRequest));
            CompletableFuture<DeleteEndpointResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a platform application object for one of the supported push notification services, such as APNS and GCM
     * (Firebase Cloud Messaging). For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param deletePlatformApplicationRequest
     *        Input for DeletePlatformApplication action.
     * @return A Java Future containing the result of the DeletePlatformApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeletePlatformApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeletePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlatformApplicationResponse> deletePlatformApplication(
            DeletePlatformApplicationRequest deletePlatformApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePlatformApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlatformApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlatformApplication");

            HttpResponseHandler<DeletePlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeletePlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeletePlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlatformApplicationRequest, DeletePlatformApplicationResponse>()
                            .withOperationName("DeletePlatformApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePlatformApplicationRequest));
            CompletableFuture<DeletePlatformApplicationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Web Services account's verified or pending phone number from the SMS sandbox.
     * </p>
     * <p>
     * When you start using Amazon SNS to send SMS messages, your Amazon Web Services account is in the <i>SMS
     * sandbox</i>. The SMS sandbox provides a safe environment for you to try Amazon SNS features without risking your
     * reputation as an SMS sender. While your Amazon Web Services account is in the SMS sandbox, you can use all of the
     * features of Amazon SNS. However, you can send SMS messages only to verified destination phone numbers. For more
     * information, including how to move out of the sandbox to send messages without restrictions, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-sms-sandbox.html">SMS sandbox</a> in the <i>Amazon SNS
     * Developer Guide</i>.
     * </p>
     *
     * @param deleteSmsSandboxPhoneNumberRequest
     * @return A Java Future containing the result of the DeleteSMSSandboxPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>UserErrorException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeleteSMSSandboxPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteSMSSandboxPhoneNumber"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSmsSandboxPhoneNumberResponse> deleteSMSSandboxPhoneNumber(
            DeleteSmsSandboxPhoneNumberRequest deleteSmsSandboxPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSmsSandboxPhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSmsSandboxPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSMSSandboxPhoneNumber");

            HttpResponseHandler<DeleteSmsSandboxPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteSmsSandboxPhoneNumberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteSmsSandboxPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSmsSandboxPhoneNumberRequest, DeleteSmsSandboxPhoneNumberResponse>()
                            .withOperationName("DeleteSMSSandboxPhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSmsSandboxPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSmsSandboxPhoneNumberRequest));
            CompletableFuture<DeleteSmsSandboxPhoneNumberResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a topic and all its subscriptions. Deleting a topic might prevent some messages previously sent to the
     * topic from being delivered to subscribers. This action is idempotent, so deleting a topic that does not exist
     * does not result in an error.
     * </p>
     *
     * @param deleteTopicRequest
     * @return A Java Future containing the result of the DeleteTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeleteTopic
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTopicResponse> deleteTopic(DeleteTopicRequest deleteTopicRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTopicRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTopicRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTopic");

            HttpResponseHandler<DeleteTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTopicRequest, DeleteTopicResponse>()
                            .withOperationName("DeleteTopic").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTopicRequest));
            CompletableFuture<DeleteTopicResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified inline <code>DataProtectionPolicy</code> document that is stored in the specified Amazon
     * SNS topic.
     * </p>
     *
     * @param getDataProtectionPolicyRequest
     * @return A Java Future containing the result of the GetDataProtectionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetDataProtectionPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetDataProtectionPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataProtectionPolicyResponse> getDataProtectionPolicy(
            GetDataProtectionPolicyRequest getDataProtectionPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataProtectionPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataProtectionPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataProtectionPolicy");

            HttpResponseHandler<GetDataProtectionPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetDataProtectionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetDataProtectionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataProtectionPolicyRequest, GetDataProtectionPolicyResponse>()
                            .withOperationName("GetDataProtectionPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataProtectionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataProtectionPolicyRequest));
            CompletableFuture<GetDataProtectionPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM
     * (Firebase Cloud Messaging) and APNS. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param getEndpointAttributesRequest
     *        Input for GetEndpointAttributes action.
     * @return A Java Future containing the result of the GetEndpointAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetEndpointAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEndpointAttributesResponse> getEndpointAttributes(
            GetEndpointAttributesRequest getEndpointAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEndpointAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEndpointAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEndpointAttributes");

            HttpResponseHandler<GetEndpointAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetEndpointAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetEndpointAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEndpointAttributesRequest, GetEndpointAttributesResponse>()
                            .withOperationName("GetEndpointAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEndpointAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEndpointAttributesRequest));
            CompletableFuture<GetEndpointAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the attributes of the platform application object for the supported push notification services, such as
     * APNS and GCM (Firebase Cloud Messaging). For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param getPlatformApplicationAttributesRequest
     *        Input for GetPlatformApplicationAttributes action.
     * @return A Java Future containing the result of the GetPlatformApplicationAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetPlatformApplicationAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlatformApplicationAttributesResponse> getPlatformApplicationAttributes(
            GetPlatformApplicationAttributesRequest getPlatformApplicationAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlatformApplicationAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getPlatformApplicationAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlatformApplicationAttributes");

            HttpResponseHandler<GetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetPlatformApplicationAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetPlatformApplicationAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlatformApplicationAttributesRequest, GetPlatformApplicationAttributesResponse>()
                            .withOperationName("GetPlatformApplicationAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPlatformApplicationAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPlatformApplicationAttributesRequest));
            CompletableFuture<GetPlatformApplicationAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the settings for sending SMS messages from your Amazon Web Services account.
     * </p>
     * <p>
     * These settings are set with the <code>SetSMSAttributes</code> action.
     * </p>
     *
     * @param getSmsAttributesRequest
     *        The input for the <code>GetSMSAttributes</code> request.
     * @return A Java Future containing the result of the GetSMSAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetSMSAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSmsAttributesResponse> getSMSAttributes(GetSmsAttributesRequest getSmsAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSmsAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSmsAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSMSAttributes");

            HttpResponseHandler<GetSmsAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetSmsAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetSmsAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSmsAttributesRequest, GetSmsAttributesResponse>()
                            .withOperationName("GetSMSAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSmsAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSmsAttributesRequest));
            CompletableFuture<GetSmsAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the SMS sandbox status for the calling Amazon Web Services account in the target Amazon Web Services
     * Region.
     * </p>
     * <p>
     * When you start using Amazon SNS to send SMS messages, your Amazon Web Services account is in the <i>SMS
     * sandbox</i>. The SMS sandbox provides a safe environment for you to try Amazon SNS features without risking your
     * reputation as an SMS sender. While your Amazon Web Services account is in the SMS sandbox, you can use all of the
     * features of Amazon SNS. However, you can send SMS messages only to verified destination phone numbers. For more
     * information, including how to move out of the sandbox to send messages without restrictions, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-sms-sandbox.html">SMS sandbox</a> in the <i>Amazon SNS
     * Developer Guide</i>.
     * </p>
     *
     * @param getSmsSandboxAccountStatusRequest
     * @return A Java Future containing the result of the GetSMSSandboxAccountStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetSMSSandboxAccountStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSMSSandboxAccountStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSmsSandboxAccountStatusResponse> getSMSSandboxAccountStatus(
            GetSmsSandboxAccountStatusRequest getSmsSandboxAccountStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSmsSandboxAccountStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSmsSandboxAccountStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSMSSandboxAccountStatus");

            HttpResponseHandler<GetSmsSandboxAccountStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetSmsSandboxAccountStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetSmsSandboxAccountStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSmsSandboxAccountStatusRequest, GetSmsSandboxAccountStatusResponse>()
                            .withOperationName("GetSMSSandboxAccountStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSmsSandboxAccountStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSmsSandboxAccountStatusRequest));
            CompletableFuture<GetSmsSandboxAccountStatusResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all of the properties of a subscription.
     * </p>
     *
     * @param getSubscriptionAttributesRequest
     *        Input for GetSubscriptionAttributes.
     * @return A Java Future containing the result of the GetSubscriptionAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetSubscriptionAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriptionAttributesResponse> getSubscriptionAttributes(
            GetSubscriptionAttributesRequest getSubscriptionAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSubscriptionAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriptionAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriptionAttributes");

            HttpResponseHandler<GetSubscriptionAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetSubscriptionAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetSubscriptionAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriptionAttributesRequest, GetSubscriptionAttributesResponse>()
                            .withOperationName("GetSubscriptionAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSubscriptionAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSubscriptionAttributesRequest));
            CompletableFuture<GetSubscriptionAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of
     * the user.
     * </p>
     *
     * @param getTopicAttributesRequest
     *        Input for GetTopicAttributes action.
     * @return A Java Future containing the result of the GetTopicAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetTopicAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTopicAttributesResponse> getTopicAttributes(GetTopicAttributesRequest getTopicAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTopicAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTopicAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTopicAttributes");

            HttpResponseHandler<GetTopicAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetTopicAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetTopicAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTopicAttributesRequest, GetTopicAttributesResponse>()
                            .withOperationName("GetTopicAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTopicAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTopicAttributesRequest));
            CompletableFuture<GetTopicAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM
     * (Firebase Cloud Messaging) and APNS. The results for <code>ListEndpointsByPlatformApplication</code> are
     * paginated and return a limited list of endpoints, up to 100. If additional records are available after the first
     * page results, then a NextToken string will be returned. To receive the next page, you call
     * <code>ListEndpointsByPlatformApplication</code> again using the NextToken string received from the previous call.
     * When there are no more records to return, NextToken will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listEndpointsByPlatformApplicationRequest
     *        Input for ListEndpointsByPlatformApplication action.
     * @return A Java Future containing the result of the ListEndpointsByPlatformApplication operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListEndpointsByPlatformApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListEndpointsByPlatformApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEndpointsByPlatformApplicationResponse> listEndpointsByPlatformApplication(
            ListEndpointsByPlatformApplicationRequest listEndpointsByPlatformApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEndpointsByPlatformApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listEndpointsByPlatformApplicationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEndpointsByPlatformApplication");

            HttpResponseHandler<ListEndpointsByPlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListEndpointsByPlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListEndpointsByPlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEndpointsByPlatformApplicationRequest, ListEndpointsByPlatformApplicationResponse>()
                            .withOperationName("ListEndpointsByPlatformApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEndpointsByPlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEndpointsByPlatformApplicationRequest));
            CompletableFuture<ListEndpointsByPlatformApplicationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the calling Amazon Web Services account's dedicated origination numbers and their metadata. For more
     * information about origination numbers, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/channels-sms-originating-identities-origination-numbers.html"
     * >Origination numbers</a> in the <i>Amazon SNS Developer Guide</i>.
     * </p>
     *
     * @param listOriginationNumbersRequest
     * @return A Java Future containing the result of the ListOriginationNumbers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ValidationException Indicates that a parameter in the request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListOriginationNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListOriginationNumbers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOriginationNumbersResponse> listOriginationNumbers(
            ListOriginationNumbersRequest listOriginationNumbersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOriginationNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOriginationNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOriginationNumbers");

            HttpResponseHandler<ListOriginationNumbersResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListOriginationNumbersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListOriginationNumbersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOriginationNumbersRequest, ListOriginationNumbersResponse>()
                            .withOperationName("ListOriginationNumbers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOriginationNumbersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOriginationNumbersRequest));
            CompletableFuture<ListOriginationNumbersResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     * </p>
     * <p>
     * The results for <code>ListPhoneNumbersOptedOut</code> are paginated, and each page returns up to 100 phone
     * numbers. If additional phone numbers are available after the first page of results, then a <code>NextToken</code>
     * string will be returned. To receive the next page, you call <code>ListPhoneNumbersOptedOut</code> again using the
     * <code>NextToken</code> string received from the previous call. When there are no more records to return,
     * <code>NextToken</code> will be null.
     * </p>
     *
     * @param listPhoneNumbersOptedOutRequest
     *        The input for the <code>ListPhoneNumbersOptedOut</code> action.
     * @return A Java Future containing the result of the ListPhoneNumbersOptedOut operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListPhoneNumbersOptedOut
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPhoneNumbersOptedOut" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPhoneNumbersOptedOutResponse> listPhoneNumbersOptedOut(
            ListPhoneNumbersOptedOutRequest listPhoneNumbersOptedOutRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPhoneNumbersOptedOutRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumbersOptedOutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumbersOptedOut");

            HttpResponseHandler<ListPhoneNumbersOptedOutResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListPhoneNumbersOptedOutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListPhoneNumbersOptedOutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPhoneNumbersOptedOutRequest, ListPhoneNumbersOptedOutResponse>()
                            .withOperationName("ListPhoneNumbersOptedOut").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPhoneNumbersOptedOutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPhoneNumbersOptedOutRequest));
            CompletableFuture<ListPhoneNumbersOptedOutResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the platform application objects for the supported push notification services, such as APNS and GCM
     * (Firebase Cloud Messaging). The results for <code>ListPlatformApplications</code> are paginated and return a
     * limited list of applications, up to 100. If additional records are available after the first page results, then a
     * NextToken string will be returned. To receive the next page, you call <code>ListPlatformApplications</code> using
     * the NextToken string received from the previous call. When there are no more records to return,
     * <code>NextToken</code> will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 15 transactions per second (TPS).
     * </p>
     *
     * @param listPlatformApplicationsRequest
     *        Input for ListPlatformApplications action.
     * @return A Java Future containing the result of the ListPlatformApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListPlatformApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPlatformApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlatformApplicationsResponse> listPlatformApplications(
            ListPlatformApplicationsRequest listPlatformApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlatformApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlatformApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlatformApplications");

            HttpResponseHandler<ListPlatformApplicationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListPlatformApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListPlatformApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlatformApplicationsRequest, ListPlatformApplicationsResponse>()
                            .withOperationName("ListPlatformApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPlatformApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPlatformApplicationsRequest));
            CompletableFuture<ListPlatformApplicationsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the calling Amazon Web Services account's current verified and pending destination phone numbers in the SMS
     * sandbox.
     * </p>
     * <p>
     * When you start using Amazon SNS to send SMS messages, your Amazon Web Services account is in the <i>SMS
     * sandbox</i>. The SMS sandbox provides a safe environment for you to try Amazon SNS features without risking your
     * reputation as an SMS sender. While your Amazon Web Services account is in the SMS sandbox, you can use all of the
     * features of Amazon SNS. However, you can send SMS messages only to verified destination phone numbers. For more
     * information, including how to move out of the sandbox to send messages without restrictions, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-sms-sandbox.html">SMS sandbox</a> in the <i>Amazon SNS
     * Developer Guide</i>.
     * </p>
     *
     * @param listSmsSandboxPhoneNumbersRequest
     * @return A Java Future containing the result of the ListSMSSandboxPhoneNumbers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSMSSandboxPhoneNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSMSSandboxPhoneNumbers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSmsSandboxPhoneNumbersResponse> listSMSSandboxPhoneNumbers(
            ListSmsSandboxPhoneNumbersRequest listSmsSandboxPhoneNumbersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSmsSandboxPhoneNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSmsSandboxPhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSMSSandboxPhoneNumbers");

            HttpResponseHandler<ListSmsSandboxPhoneNumbersResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListSmsSandboxPhoneNumbersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListSmsSandboxPhoneNumbersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSmsSandboxPhoneNumbersRequest, ListSmsSandboxPhoneNumbersResponse>()
                            .withOperationName("ListSMSSandboxPhoneNumbers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSmsSandboxPhoneNumbersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSmsSandboxPhoneNumbersRequest));
            CompletableFuture<ListSmsSandboxPhoneNumbersResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If
     * there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter
     * in a new <code>ListSubscriptions</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsRequest
     *        Input for ListSubscriptions action.
     * @return A Java Future containing the result of the ListSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscriptionsResponse> listSubscriptions(ListSubscriptionsRequest listSubscriptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSubscriptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubscriptions");

            HttpResponseHandler<ListSubscriptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscriptionsRequest, ListSubscriptionsResponse>()
                            .withOperationName("ListSubscriptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSubscriptionsRequest));
            CompletableFuture<ListSubscriptionsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to
     * 100. If there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code>
     * parameter in a new <code>ListSubscriptionsByTopic</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsByTopicRequest
     *        Input for ListSubscriptionsByTopic action.
     * @return A Java Future containing the result of the ListSubscriptionsByTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptionsByTopic
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptionsByTopic" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscriptionsByTopicResponse> listSubscriptionsByTopic(
            ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSubscriptionsByTopicRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubscriptionsByTopicRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubscriptionsByTopic");

            HttpResponseHandler<ListSubscriptionsByTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListSubscriptionsByTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListSubscriptionsByTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscriptionsByTopicRequest, ListSubscriptionsByTopicResponse>()
                            .withOperationName("ListSubscriptionsByTopic").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSubscriptionsByTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSubscriptionsByTopicRequest));
            CompletableFuture<ListSubscriptionsByTopicResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all tags added to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon Simple
     * Notification Service Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are
     * more topics, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter in a new
     * <code>ListTopics</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listTopicsRequest
     * @return A Java Future containing the result of the ListTopics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListTopics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTopics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTopicsResponse> listTopics(ListTopicsRequest listTopicsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTopicsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTopicsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTopics");

            HttpResponseHandler<ListTopicsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListTopicsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListTopicsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTopicsRequest, ListTopicsResponse>().withOperationName("ListTopics")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTopicsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTopicsRequest));
            CompletableFuture<ListTopicsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this request to opt in a phone number that is opted out, which enables you to resume sending SMS messages to
     * the number.
     * </p>
     * <p>
     * You can opt in a phone number only once every 30 days.
     * </p>
     *
     * @param optInPhoneNumberRequest
     *        Input for the OptInPhoneNumber action.
     * @return A Java Future containing the result of the OptInPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.OptInPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/OptInPhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<OptInPhoneNumberResponse> optInPhoneNumber(OptInPhoneNumberRequest optInPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(optInPhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, optInPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OptInPhoneNumber");

            HttpResponseHandler<OptInPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(OptInPhoneNumberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<OptInPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<OptInPhoneNumberRequest, OptInPhoneNumberResponse>()
                            .withOperationName("OptInPhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new OptInPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(optInPhoneNumberRequest));
            CompletableFuture<OptInPhoneNumberResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a message to an Amazon SNS topic, a text message (SMS message) directly to a phone number, or a message to
     * a mobile platform endpoint (when you specify the <code>TargetArn</code>).
     * </p>
     * <p>
     * If you send a message to a topic, Amazon SNS delivers the message to each endpoint that is subscribed to the
     * topic. The format of the message depends on the notification protocol for each subscribed endpoint.
     * </p>
     * <p>
     * When a <code>messageId</code> is returned, the message is saved and Amazon SNS immediately delivers it to
     * subscribers.
     * </p>
     * <p>
     * To use the <code>Publish</code> action for publishing a message to a mobile endpoint, such as an app on a Kindle
     * device or mobile phone, you must specify the EndpointArn for the TargetArn parameter. The EndpointArn is returned
     * when making a call with the <code>CreatePlatformEndpoint</code> action.
     * </p>
     * <p>
     * For more information about formatting messages, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/mobile-push-send-custommessage.html">Send Custom
     * Platform-Specific Payloads in Messages to Mobile Devices</a>.
     * </p>
     * <important>
     * <p>
     * You can publish messages only to topics and endpoints in the same Amazon Web Services Region.
     * </p>
     * </important>
     *
     * @param publishRequest
     *        Input for Publish action.
     * @return A Java Future containing the result of the Publish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InvalidParameterValueException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>EndpointDisabledException Exception error indicating endpoint disabled.</li>
     *         <li>PlatformApplicationDisabledException Exception error indicating platform application disabled.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>KmsDisabledException The request was rejected because the specified customer master key (CMK) isn't
     *         enabled.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource isn't
     *         valid for this request. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>Key Management Service Developer Guide</i>.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource can't be
     *         found.</li>
     *         <li>KmsOptInRequiredException The Amazon Web Services access key ID needs a subscription for the service.
     *         </li>
     *         <li>KmsThrottlingException The request was denied due to request throttling. For more information about
     *         throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Key Management Service Developer Guide.</i></li>
     *         <li>KmsAccessDeniedException The ciphertext references a key that doesn't exist or that you don't have
     *         access to.</li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>ValidationException Indicates that a parameter in the request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PublishResponse> publish(PublishRequest publishRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(publishRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Publish");

            HttpResponseHandler<PublishResponse> responseHandler = protocolFactory
                    .createResponseHandler(PublishResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PublishResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishRequest, PublishResponse>().withOperationName("Publish")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new PublishRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(publishRequest));
            CompletableFuture<PublishResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Publishes up to ten messages to the specified topic. This is a batch version of <code>Publish</code>. For FIFO
     * topics, multiple messages within a single batch are published in the order they are sent, and messages are
     * deduplicated within the batch and across batches for 5 minutes.
     * </p>
     * <p>
     * The result of publishing each message is reported individually in the response. Because the batch request can
     * result in a combination of successful and unsuccessful actions, you should check for batch errors even when the
     * call returns an HTTP status code of <code>200</code>.
     * </p>
     * <p>
     * The maximum allowed individual message size and the maximum total payload size (the sum of the individual lengths
     * of all of the batched messages) are both 256 KB (262,144 bytes).
     * </p>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * &amp;AttributeName.1=first
     * </p>
     * <p>
     * &amp;AttributeName.2=second
     * </p>
     * <p>
     * If you send a batch message to a topic, Amazon SNS publishes the batch message to each endpoint that is
     * subscribed to the topic. The format of the batch message depends on the notification protocol for each subscribed
     * endpoint.
     * </p>
     * <p>
     * When a <code>messageId</code> is returned, the batch message is saved and Amazon SNS immediately delivers the
     * message to subscribers.
     * </p>
     *
     * @param publishBatchRequest
     * @return A Java Future containing the result of the PublishBatch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InvalidParameterValueException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>EndpointDisabledException Exception error indicating endpoint disabled.</li>
     *         <li>PlatformApplicationDisabledException Exception error indicating platform application disabled.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>BatchEntryIdsNotDistinctException Two or more batch entries in the request have the same
     *         <code>Id</code>.</li>
     *         <li>BatchRequestTooLongException The length of all the batch messages put together is more than the
     *         limit.</li>
     *         <li>EmptyBatchRequestException The batch request doesn't contain any entries.</li>
     *         <li>InvalidBatchEntryIdException The <code>Id</code> of a batch entry in a batch request doesn't abide by
     *         the specification.</li>
     *         <li>TooManyEntriesInBatchRequestException The batch request contains more entries than permissible.</li>
     *         <li>KmsDisabledException The request was rejected because the specified customer master key (CMK) isn't
     *         enabled.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource isn't
     *         valid for this request. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>Key Management Service Developer Guide</i>.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource can't be
     *         found.</li>
     *         <li>KmsOptInRequiredException The Amazon Web Services access key ID needs a subscription for the service.
     *         </li>
     *         <li>KmsThrottlingException The request was denied due to request throttling. For more information about
     *         throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Key Management Service Developer Guide.</i></li>
     *         <li>KmsAccessDeniedException The ciphertext references a key that doesn't exist or that you don't have
     *         access to.</li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>ValidationException Indicates that a parameter in the request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.PublishBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/PublishBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PublishBatchResponse> publishBatch(PublishBatchRequest publishBatchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(publishBatchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PublishBatch");

            HttpResponseHandler<PublishBatchResponse> responseHandler = protocolFactory
                    .createResponseHandler(PublishBatchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PublishBatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishBatchRequest, PublishBatchResponse>()
                            .withOperationName("PublishBatch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PublishBatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(publishBatchRequest));
            CompletableFuture<PublishBatchResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates an inline policy document that is stored in the specified Amazon SNS topic.
     * </p>
     *
     * @param putDataProtectionPolicyRequest
     * @return A Java Future containing the result of the PutDataProtectionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.PutDataProtectionPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/PutDataProtectionPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDataProtectionPolicyResponse> putDataProtectionPolicy(
            PutDataProtectionPolicyRequest putDataProtectionPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putDataProtectionPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDataProtectionPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDataProtectionPolicy");

            HttpResponseHandler<PutDataProtectionPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutDataProtectionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutDataProtectionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDataProtectionPolicyRequest, PutDataProtectionPolicyResponse>()
                            .withOperationName("PutDataProtectionPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutDataProtectionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putDataProtectionPolicyRequest));
            CompletableFuture<PutDataProtectionPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a statement from a topic's access control policy.
     * </p>
     * <note>
     * <p>
     * To remove the ability to change topic permissions, you must deny permissions to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetTopicAttributes</code> actions in your IAM policy.
     * </p>
     * </note>
     *
     * @param removePermissionRequest
     *        Input for RemovePermission action.
     * @return A Java Future containing the result of the RemovePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.RemovePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemovePermissionResponse> removePermission(RemovePermissionRequest removePermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemovePermission");

            HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(RemovePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<RemovePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                            .withOperationName("RemovePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removePermissionRequest));
            CompletableFuture<RemovePermissionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM
     * (Firebase Cloud Messaging) and APNS. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param setEndpointAttributesRequest
     *        Input for SetEndpointAttributes action.
     * @return A Java Future containing the result of the SetEndpointAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetEndpointAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetEndpointAttributesResponse> setEndpointAttributes(
            SetEndpointAttributesRequest setEndpointAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setEndpointAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setEndpointAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetEndpointAttributes");

            HttpResponseHandler<SetEndpointAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetEndpointAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetEndpointAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetEndpointAttributesRequest, SetEndpointAttributesResponse>()
                            .withOperationName("SetEndpointAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetEndpointAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setEndpointAttributesRequest));
            CompletableFuture<SetEndpointAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS
     * and GCM (Firebase Cloud Messaging). For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>. For information on configuring attributes for message delivery status, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-msg-status.html">Using Amazon SNS Application Attributes for
     * Message Delivery Status</a>.
     * </p>
     *
     * @param setPlatformApplicationAttributesRequest
     *        Input for SetPlatformApplicationAttributes action.
     * @return A Java Future containing the result of the SetPlatformApplicationAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetPlatformApplicationAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetPlatformApplicationAttributesResponse> setPlatformApplicationAttributes(
            SetPlatformApplicationAttributesRequest setPlatformApplicationAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setPlatformApplicationAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                setPlatformApplicationAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetPlatformApplicationAttributes");

            HttpResponseHandler<SetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetPlatformApplicationAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetPlatformApplicationAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetPlatformApplicationAttributesRequest, SetPlatformApplicationAttributesResponse>()
                            .withOperationName("SetPlatformApplicationAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetPlatformApplicationAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setPlatformApplicationAttributesRequest));
            CompletableFuture<SetPlatformApplicationAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this request to set the default settings for sending SMS messages and receiving daily SMS usage reports.
     * </p>
     * <p>
     * You can override some of these settings for a single message when you use the <code>Publish</code> action with
     * the <code>MessageAttributes.entry.N</code> parameter. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sms_publish-to-phone.html">Publishing to a mobile phone</a> in
     * the <i>Amazon SNS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this operation, you must grant the Amazon SNS service principal (<code>sns.amazonaws.com</code>)
     * permission to perform the <code>s3:ListBucket</code> action.
     * </p>
     * </note>
     *
     * @param setSmsAttributesRequest
     *        The input for the SetSMSAttributes action.
     * @return A Java Future containing the result of the SetSMSAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetSMSAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetSmsAttributesResponse> setSMSAttributes(SetSmsAttributesRequest setSmsAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setSmsAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setSmsAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetSMSAttributes");

            HttpResponseHandler<SetSmsAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetSmsAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetSmsAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetSmsAttributesRequest, SetSmsAttributesResponse>()
                            .withOperationName("SetSMSAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetSmsAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setSmsAttributesRequest));
            CompletableFuture<SetSmsAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a subscription owner to set an attribute of the subscription to a new value.
     * </p>
     *
     * @param setSubscriptionAttributesRequest
     *        Input for SetSubscriptionAttributes action.
     * @return A Java Future containing the result of the SetSubscriptionAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your Amazon Web
     *         Services account exceeds the limit. To add more filter polices, submit an Amazon SNS Limit Increase case
     *         in the Amazon Web Services Support Center.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetSubscriptionAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SetSubscriptionAttributesResponse> setSubscriptionAttributes(
            SetSubscriptionAttributesRequest setSubscriptionAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setSubscriptionAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setSubscriptionAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetSubscriptionAttributes");

            HttpResponseHandler<SetSubscriptionAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetSubscriptionAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetSubscriptionAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetSubscriptionAttributesRequest, SetSubscriptionAttributesResponse>()
                            .withOperationName("SetSubscriptionAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetSubscriptionAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setSubscriptionAttributesRequest));
            CompletableFuture<SetSubscriptionAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a topic owner to set an attribute of the topic to a new value.
     * </p>
     * <note>
     * <p>
     * To remove the ability to change topic permissions, you must deny permissions to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetTopicAttributes</code> actions in your IAM policy.
     * </p>
     * </note>
     *
     * @param setTopicAttributesRequest
     *        Input for SetTopicAttributes action.
     * @return A Java Future containing the result of the SetTopicAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetTopicAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetTopicAttributesResponse> setTopicAttributes(SetTopicAttributesRequest setTopicAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setTopicAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setTopicAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetTopicAttributes");

            HttpResponseHandler<SetTopicAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetTopicAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetTopicAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetTopicAttributesRequest, SetTopicAttributesResponse>()
                            .withOperationName("SetTopicAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetTopicAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setTopicAttributesRequest));
            CompletableFuture<SetTopicAttributesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Subscribes an endpoint to an Amazon SNS topic. If the endpoint type is HTTP/S or email, or if the endpoint and
     * the topic are not in the same Amazon Web Services account, the endpoint owner must run the
     * <code>ConfirmSubscription</code> action to confirm the subscription.
     * </p>
     * <p>
     * You call the <code>ConfirmSubscription</code> action with the token from the subscription response. Confirmation
     * tokens are valid for three days.
     * </p>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param subscribeRequest
     *        Input for Subscribe action.
     * @return A Java Future containing the result of the Subscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubscriptionLimitExceededException Indicates that the customer already owns the maximum allowed
     *         number of subscriptions.</li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your Amazon Web
     *         Services account exceeds the limit. To add more filter polices, submit an Amazon SNS Limit Increase case
     *         in the Amazon Web Services Support Center.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Subscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Subscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SubscribeResponse> subscribe(SubscribeRequest subscribeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(subscribeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, subscribeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Subscribe");

            HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory
                    .createResponseHandler(SubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>().withOperationName("Subscribe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(subscribeRequest));
            CompletableFuture<SubscribeResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add tags to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     * <p>
     * When you use topic tags, keep the following guidelines in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Adding more than 50 tags to a topic isn't recommended.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags don't have any semantic meaning. Amazon SNS interprets tags as character strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A new tag with a key identical to that of an existing tag overwrites the existing tag.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tagging actions are limited to 10 TPS per Amazon Web Services account, per Amazon Web Services Region. If your
     * application requires a higher throughput, file a <a
     * href="https://console.aws.amazon.com/support/home#/case/create?issueType=technical">technical support
     * request</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the
     * subscription or the topic's owner can unsubscribe, and an Amazon Web Services signature is required. If the
     * <code>Unsubscribe</code> call does not require authentication and the requester is not the subscription owner, a
     * final cancellation message is delivered to the endpoint, so that the endpoint owner can easily resubscribe to the
     * topic if the <code>Unsubscribe</code> request was unintended.
     * </p>
     * <note>
     * <p>
     * Amazon SQS queue subscriptions require authentication for deletion. Only the owner of the subscription, or the
     * owner of the topic can unsubscribe using the required Amazon Web Services signature.
     * </p>
     * </note>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param unsubscribeRequest
     *        Input for Unsubscribe action.
     * @return A Java Future containing the result of the Unsubscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Unsubscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Unsubscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UnsubscribeResponse> unsubscribe(UnsubscribeRequest unsubscribeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unsubscribeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unsubscribeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Unsubscribe");

            HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory
                    .createResponseHandler(UnsubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UnsubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                            .withOperationName("Unsubscribe").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(unsubscribeRequest));
            CompletableFuture<UnsubscribeResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies a destination phone number with a one-time password (OTP) for the calling Amazon Web Services account.
     * </p>
     * <p>
     * When you start using Amazon SNS to send SMS messages, your Amazon Web Services account is in the <i>SMS
     * sandbox</i>. The SMS sandbox provides a safe environment for you to try Amazon SNS features without risking your
     * reputation as an SMS sender. While your Amazon Web Services account is in the SMS sandbox, you can use all of the
     * features of Amazon SNS. However, you can send SMS messages only to verified destination phone numbers. For more
     * information, including how to move out of the sandbox to send messages without restrictions, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-sms-sandbox.html">SMS sandbox</a> in the <i>Amazon SNS
     * Developer Guide</i>.
     * </p>
     *
     * @param verifySmsSandboxPhoneNumberRequest
     * @return A Java Future containing the result of the VerifySMSSandboxPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Can’t perform the action on the specified resource. Make sure that the
     *         resource exists.</li>
     *         <li>VerificationException Indicates that the one-time password (OTP) used for verification is invalid.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.VerifySMSSandboxPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/VerifySMSSandboxPhoneNumber"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VerifySmsSandboxPhoneNumberResponse> verifySMSSandboxPhoneNumber(
            VerifySmsSandboxPhoneNumberRequest verifySmsSandboxPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifySmsSandboxPhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifySmsSandboxPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SNS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifySMSSandboxPhoneNumber");

            HttpResponseHandler<VerifySmsSandboxPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(VerifySmsSandboxPhoneNumberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<VerifySmsSandboxPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifySmsSandboxPhoneNumberRequest, VerifySmsSandboxPhoneNumberResponse>()
                            .withOperationName("VerifySMSSandboxPhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new VerifySmsSandboxPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(verifySmsSandboxPhoneNumberRequest));
            CompletableFuture<VerifySmsSandboxPhoneNumberResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SnsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubscriptionLimitExceeded")
                                .exceptionBuilderSupplier(SubscriptionLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSThrottling")
                                .exceptionBuilderSupplier(KmsThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameter")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct")
                                .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterValueInvalid")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointDisabled")
                                .exceptionBuilderSupplier(EndpointDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterPolicyLimitExceeded")
                                .exceptionBuilderSupplier(FilterPolicyLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidState")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSNotFound")
                                .exceptionBuilderSupplier(KmsNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserError").exceptionBuilderSupplier(UserErrorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSOptInRequired")
                                .exceptionBuilderSupplier(KmsOptInRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StaleTag").exceptionBuilderSupplier(StaleTagException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFound").exceptionBuilderSupplier(NotFoundException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VerificationException")
                                .exceptionBuilderSupplier(VerificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("Throttled").exceptionBuilderSupplier(ThrottledException::builder)
                                .httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyBatchRequest")
                                .exceptionBuilderSupplier(EmptyBatchRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSecurity")
                                .exceptionBuilderSupplier(InvalidSecurityException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptedOut").exceptionBuilderSupplier(OptedOutException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest")
                                .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchRequestTooLong")
                                .exceptionBuilderSupplier(BatchRequestTooLongException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSAccessDenied")
                                .exceptionBuilderSupplier(KmsAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSDisabled")
                                .exceptionBuilderSupplier(KmsDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TopicLimitExceeded")
                                .exceptionBuilderSupplier(TopicLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicy").exceptionBuilderSupplier(TagPolicyException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceeded")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PlatformApplicationDisabled")
                                .exceptionBuilderSupplier(PlatformApplicationDisabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationError")
                                .exceptionBuilderSupplier(AuthorizationErrorException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentAccess")
                                .exceptionBuilderSupplier(ConcurrentAccessException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidBatchEntryId")
                                .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).httpStatusCode(400).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SnsException::builder).build();
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SnsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = SnsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
