/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Input for CreatePlatformEndpoint action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreatePlatformEndpointRequest extends SnsRequest implements
        ToCopyableBuilder<CreatePlatformEndpointRequest.Builder, CreatePlatformEndpointRequest> {
    private static final SdkField<String> PLATFORM_APPLICATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlatformApplicationArn").getter(getter(CreatePlatformEndpointRequest::platformApplicationArn))
            .setter(setter(Builder::platformApplicationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformApplicationArn").build())
            .build();

    private static final SdkField<String> TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Token")
            .getter(getter(CreatePlatformEndpointRequest::token)).setter(setter(Builder::token))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Token").build()).build();

    private static final SdkField<String> CUSTOM_USER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomUserData").getter(getter(CreatePlatformEndpointRequest::customUserData))
            .setter(setter(Builder::customUserData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomUserData").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(CreatePlatformEndpointRequest::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            PLATFORM_APPLICATION_ARN_FIELD, TOKEN_FIELD, CUSTOM_USER_DATA_FIELD, ATTRIBUTES_FIELD));

    private final String platformApplicationArn;

    private final String token;

    private final String customUserData;

    private final Map<String, String> attributes;

    private CreatePlatformEndpointRequest(BuilderImpl builder) {
        super(builder);
        this.platformApplicationArn = builder.platformApplicationArn;
        this.token = builder.token;
        this.customUserData = builder.customUserData;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * <code>PlatformApplicationArn</code> returned from CreatePlatformApplication is used to create a an endpoint.
     * </p>
     * 
     * @return <code>PlatformApplicationArn</code> returned from CreatePlatformApplication is used to create a an
     *         endpoint.
     */
    public final String platformApplicationArn() {
        return platformApplicationArn;
    }

    /**
     * <p>
     * Unique identifier created by the notification service for an app on a device. The specific name for Token will
     * vary, depending on which notification service is being used. For example, when using APNS as the notification
     * service, you need the device token. Alternatively, when using GCM (Firebase Cloud Messaging) or ADM, the device
     * token equivalent is called the registration ID.
     * </p>
     * 
     * @return Unique identifier created by the notification service for an app on a device. The specific name for Token
     *         will vary, depending on which notification service is being used. For example, when using APNS as the
     *         notification service, you need the device token. Alternatively, when using GCM (Firebase Cloud Messaging)
     *         or ADM, the device token equivalent is called the registration ID.
     */
    public final String token() {
        return token;
    }

    /**
     * <p>
     * Arbitrary user data to associate with the endpoint. Amazon SNS does not use this data. The data must be in UTF-8
     * format and less than 2KB.
     * </p>
     * 
     * @return Arbitrary user data to associate with the endpoint. Amazon SNS does not use this data. The data must be
     *         in UTF-8 format and less than 2KB.
     */
    public final String customUserData() {
        return customUserData;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * For a list of attributes, see <a
     * href="https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html">
     * <code>SetEndpointAttributes</code> </a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return For a list of attributes, see <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html">
     *         <code>SetEndpointAttributes</code> </a>.
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(platformApplicationArn());
        hashCode = 31 * hashCode + Objects.hashCode(token());
        hashCode = 31 * hashCode + Objects.hashCode(customUserData());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreatePlatformEndpointRequest)) {
            return false;
        }
        CreatePlatformEndpointRequest other = (CreatePlatformEndpointRequest) obj;
        return Objects.equals(platformApplicationArn(), other.platformApplicationArn()) && Objects.equals(token(), other.token())
                && Objects.equals(customUserData(), other.customUserData()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreatePlatformEndpointRequest").add("PlatformApplicationArn", platformApplicationArn())
                .add("Token", token()).add("CustomUserData", customUserData())
                .add("Attributes", hasAttributes() ? attributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlatformApplicationArn":
            return Optional.ofNullable(clazz.cast(platformApplicationArn()));
        case "Token":
            return Optional.ofNullable(clazz.cast(token()));
        case "CustomUserData":
            return Optional.ofNullable(clazz.cast(customUserData()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreatePlatformEndpointRequest, T> g) {
        return obj -> g.apply((CreatePlatformEndpointRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SnsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreatePlatformEndpointRequest> {
        /**
         * <p>
         * <code>PlatformApplicationArn</code> returned from CreatePlatformApplication is used to create a an endpoint.
         * </p>
         * 
         * @param platformApplicationArn
         *        <code>PlatformApplicationArn</code> returned from CreatePlatformApplication is used to create a an
         *        endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformApplicationArn(String platformApplicationArn);

        /**
         * <p>
         * Unique identifier created by the notification service for an app on a device. The specific name for Token
         * will vary, depending on which notification service is being used. For example, when using APNS as the
         * notification service, you need the device token. Alternatively, when using GCM (Firebase Cloud Messaging) or
         * ADM, the device token equivalent is called the registration ID.
         * </p>
         * 
         * @param token
         *        Unique identifier created by the notification service for an app on a device. The specific name for
         *        Token will vary, depending on which notification service is being used. For example, when using APNS
         *        as the notification service, you need the device token. Alternatively, when using GCM (Firebase Cloud
         *        Messaging) or ADM, the device token equivalent is called the registration ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder token(String token);

        /**
         * <p>
         * Arbitrary user data to associate with the endpoint. Amazon SNS does not use this data. The data must be in
         * UTF-8 format and less than 2KB.
         * </p>
         * 
         * @param customUserData
         *        Arbitrary user data to associate with the endpoint. Amazon SNS does not use this data. The data must
         *        be in UTF-8 format and less than 2KB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customUserData(String customUserData);

        /**
         * <p>
         * For a list of attributes, see <a
         * href="https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html">
         * <code>SetEndpointAttributes</code> </a>.
         * </p>
         * 
         * @param attributes
         *        For a list of attributes, see <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html">
         *        <code>SetEndpointAttributes</code> </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SnsRequest.BuilderImpl implements Builder {
        private String platformApplicationArn;

        private String token;

        private String customUserData;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreatePlatformEndpointRequest model) {
            super(model);
            platformApplicationArn(model.platformApplicationArn);
            token(model.token);
            customUserData(model.customUserData);
            attributes(model.attributes);
        }

        public final String getPlatformApplicationArn() {
            return platformApplicationArn;
        }

        public final void setPlatformApplicationArn(String platformApplicationArn) {
            this.platformApplicationArn = platformApplicationArn;
        }

        @Override
        public final Builder platformApplicationArn(String platformApplicationArn) {
            this.platformApplicationArn = platformApplicationArn;
            return this;
        }

        public final String getToken() {
            return token;
        }

        public final void setToken(String token) {
            this.token = token;
        }

        @Override
        public final Builder token(String token) {
            this.token = token;
            return this;
        }

        public final String getCustomUserData() {
            return customUserData;
        }

        public final void setCustomUserData(String customUserData) {
            this.customUserData = customUserData;
        }

        @Override
        public final Builder customUserData(String customUserData) {
            this.customUserData = customUserData;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = MapStringToStringCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = MapStringToStringCopier.copy(attributes);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreatePlatformEndpointRequest build() {
            return new CreatePlatformEndpointRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
