/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Input for Subscribe action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SubscribeRequest extends SnsRequest implements ToCopyableBuilder<SubscribeRequest.Builder, SubscribeRequest> {
    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SubscribeRequest::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SubscribeRequest::protocol)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SubscribeRequest::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SubscribeRequest::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> RETURN_SUBSCRIPTION_ARN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(SubscribeRequest::returnSubscriptionArn)).setter(setter(Builder::returnSubscriptionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnSubscriptionArn").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOPIC_ARN_FIELD,
            PROTOCOL_FIELD, ENDPOINT_FIELD, ATTRIBUTES_FIELD, RETURN_SUBSCRIPTION_ARN_FIELD));

    private final String topicArn;

    private final String protocol;

    private final String endpoint;

    private final Map<String, String> attributes;

    private final Boolean returnSubscriptionArn;

    private SubscribeRequest(BuilderImpl builder) {
        super(builder);
        this.topicArn = builder.topicArn;
        this.protocol = builder.protocol;
        this.endpoint = builder.endpoint;
        this.attributes = builder.attributes;
        this.returnSubscriptionArn = builder.returnSubscriptionArn;
    }

    /**
     * <p>
     * The ARN of the topic you want to subscribe to.
     * </p>
     * 
     * @return The ARN of the topic you want to subscribe to.
     */
    public String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The protocol you want to use. Supported protocols include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>http</code> – delivery of JSON-encoded message via HTTP POST
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>https</code> – delivery of JSON-encoded message via HTTPS POST
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>email</code> – delivery of message via SMTP
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>email-json</code> – delivery of JSON-encoded message via SMTP
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sms</code> – delivery of message via SMS
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sqs</code> – delivery of JSON-encoded message to an Amazon SQS queue
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application</code> – delivery of JSON-encoded message to an EndpointArn for a mobile app and device.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda</code> – delivery of JSON-encoded message to an AWS Lambda function.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The protocol you want to use. Supported protocols include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>http</code> – delivery of JSON-encoded message via HTTP POST
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>https</code> – delivery of JSON-encoded message via HTTPS POST
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>email</code> – delivery of message via SMTP
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>email-json</code> – delivery of JSON-encoded message via SMTP
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sms</code> – delivery of message via SMS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sqs</code> – delivery of JSON-encoded message to an Amazon SQS queue
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>application</code> – delivery of JSON-encoded message to an EndpointArn for a mobile app and
     *         device.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda</code> – delivery of JSON-encoded message to an AWS Lambda function.
     *         </p>
     *         </li>
     */
    public String protocol() {
        return protocol;
    }

    /**
     * <p>
     * The endpoint that you want to receive notifications. Endpoints vary by protocol:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>http</code> protocol, the endpoint is an URL beginning with "http://"
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>https</code> protocol, the endpoint is a URL beginning with "https://"
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>email</code> protocol, the endpoint is an email address
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>email-json</code> protocol, the endpoint is an email address
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>sms</code> protocol, the endpoint is a phone number of an SMS-enabled device
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>sqs</code> protocol, the endpoint is the ARN of an Amazon SQS queue
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>application</code> protocol, the endpoint is the EndpointArn of a mobile app and device.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>lambda</code> protocol, the endpoint is the ARN of an AWS Lambda function.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The endpoint that you want to receive notifications. Endpoints vary by protocol:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>http</code> protocol, the endpoint is an URL beginning with "http://"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>https</code> protocol, the endpoint is a URL beginning with "https://"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>email</code> protocol, the endpoint is an email address
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>email-json</code> protocol, the endpoint is an email address
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>sms</code> protocol, the endpoint is a phone number of an SMS-enabled device
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>sqs</code> protocol, the endpoint is the ARN of an Amazon SQS queue
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>application</code> protocol, the endpoint is the EndpointArn of a mobile app and device.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>lambda</code> protocol, the endpoint is the ARN of an AWS Lambda function.
     *         </p>
     *         </li>
     */
    public String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * A map of attributes with their corresponding values.
     * </p>
     * <p>
     * The following lists the names, descriptions, and values of the special request parameters that the
     * <code>SetTopicAttributes</code> action uses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     * endpoints.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FilterPolicy</code> – The simple JSON object that lets your subscriber receive only a subset of messages,
     * rather than receiving every message published to the topic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RawMessageDelivery</code> – When set to <code>true</code>, enables raw message delivery to Amazon SQS or
     * HTTP/S endpoints. This eliminates the need for the endpoints to process JSON formatting, which is otherwise
     * created for Amazon SNS metadata.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A map of attributes with their corresponding values.</p>
     *         <p>
     *         The following lists the names, descriptions, and values of the special request parameters that the
     *         <code>SetTopicAttributes</code> action uses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     *         endpoints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FilterPolicy</code> – The simple JSON object that lets your subscriber receive only a subset of
     *         messages, rather than receiving every message published to the topic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RawMessageDelivery</code> – When set to <code>true</code>, enables raw message delivery to Amazon
     *         SQS or HTTP/S endpoints. This eliminates the need for the endpoints to process JSON formatting, which is
     *         otherwise created for Amazon SNS metadata.
     *         </p>
     *         </li>
     */
    public Map<String, String> attributes() {
        return attributes;
    }

    /**
     * <p>
     * Sets whether the response from the <code>Subscribe</code> request includes the subscription ARN, even if the
     * subscription is not yet confirmed.
     * </p>
     * <p>
     * If you set this parameter to <code>false</code>, the response includes the ARN for confirmed subscriptions, but
     * it includes an ARN value of "pending subscription" for subscriptions that are not yet confirmed. A subscription
     * becomes confirmed when the subscriber calls the <code>ConfirmSubscription</code> action with a confirmation
     * token.
     * </p>
     * <p>
     * If you set this parameter to <code>true</code>, the response includes the ARN in all cases, even if the
     * subscription is not yet confirmed.
     * </p>
     * <p>
     * The default value is <code>false</code>.
     * </p>
     * 
     * @return Sets whether the response from the <code>Subscribe</code> request includes the subscription ARN, even if
     *         the subscription is not yet confirmed.</p>
     *         <p>
     *         If you set this parameter to <code>false</code>, the response includes the ARN for confirmed
     *         subscriptions, but it includes an ARN value of "pending subscription" for subscriptions that are not yet
     *         confirmed. A subscription becomes confirmed when the subscriber calls the
     *         <code>ConfirmSubscription</code> action with a confirmation token.
     *         </p>
     *         <p>
     *         If you set this parameter to <code>true</code>, the response includes the ARN in all cases, even if the
     *         subscription is not yet confirmed.
     *         </p>
     *         <p>
     *         The default value is <code>false</code>.
     */
    public Boolean returnSubscriptionArn() {
        return returnSubscriptionArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(protocol());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(attributes());
        hashCode = 31 * hashCode + Objects.hashCode(returnSubscriptionArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SubscribeRequest)) {
            return false;
        }
        SubscribeRequest other = (SubscribeRequest) obj;
        return Objects.equals(topicArn(), other.topicArn()) && Objects.equals(protocol(), other.protocol())
                && Objects.equals(endpoint(), other.endpoint()) && Objects.equals(attributes(), other.attributes())
                && Objects.equals(returnSubscriptionArn(), other.returnSubscriptionArn());
    }

    @Override
    public String toString() {
        return ToString.builder("SubscribeRequest").add("TopicArn", topicArn()).add("Protocol", protocol())
                .add("Endpoint", endpoint()).add("Attributes", attributes())
                .add("ReturnSubscriptionArn", returnSubscriptionArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocol()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "ReturnSubscriptionArn":
            return Optional.ofNullable(clazz.cast(returnSubscriptionArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SubscribeRequest, T> g) {
        return obj -> g.apply((SubscribeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SnsRequest.Builder, SdkPojo, CopyableBuilder<Builder, SubscribeRequest> {
        /**
         * <p>
         * The ARN of the topic you want to subscribe to.
         * </p>
         * 
         * @param topicArn
         *        The ARN of the topic you want to subscribe to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The protocol you want to use. Supported protocols include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>http</code> – delivery of JSON-encoded message via HTTP POST
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>https</code> – delivery of JSON-encoded message via HTTPS POST
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>email</code> – delivery of message via SMTP
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>email-json</code> – delivery of JSON-encoded message via SMTP
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sms</code> – delivery of message via SMS
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sqs</code> – delivery of JSON-encoded message to an Amazon SQS queue
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>application</code> – delivery of JSON-encoded message to an EndpointArn for a mobile app and device.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda</code> – delivery of JSON-encoded message to an AWS Lambda function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocol
         *        The protocol you want to use. Supported protocols include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>http</code> – delivery of JSON-encoded message via HTTP POST
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>https</code> – delivery of JSON-encoded message via HTTPS POST
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>email</code> – delivery of message via SMTP
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>email-json</code> – delivery of JSON-encoded message via SMTP
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sms</code> – delivery of message via SMS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sqs</code> – delivery of JSON-encoded message to an Amazon SQS queue
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>application</code> – delivery of JSON-encoded message to an EndpointArn for a mobile app and
         *        device.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda</code> – delivery of JSON-encoded message to an AWS Lambda function.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The endpoint that you want to receive notifications. Endpoints vary by protocol:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>http</code> protocol, the endpoint is an URL beginning with "http://"
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>https</code> protocol, the endpoint is a URL beginning with "https://"
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>email</code> protocol, the endpoint is an email address
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>email-json</code> protocol, the endpoint is an email address
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>sms</code> protocol, the endpoint is a phone number of an SMS-enabled device
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>sqs</code> protocol, the endpoint is the ARN of an Amazon SQS queue
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>application</code> protocol, the endpoint is the EndpointArn of a mobile app and device.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>lambda</code> protocol, the endpoint is the ARN of an AWS Lambda function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param endpoint
         *        The endpoint that you want to receive notifications. Endpoints vary by protocol:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>http</code> protocol, the endpoint is an URL beginning with "http://"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>https</code> protocol, the endpoint is a URL beginning with "https://"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>email</code> protocol, the endpoint is an email address
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>email-json</code> protocol, the endpoint is an email address
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>sms</code> protocol, the endpoint is a phone number of an SMS-enabled device
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>sqs</code> protocol, the endpoint is the ARN of an Amazon SQS queue
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>application</code> protocol, the endpoint is the EndpointArn of a mobile app and device.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>lambda</code> protocol, the endpoint is the ARN of an AWS Lambda function.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * A map of attributes with their corresponding values.
         * </p>
         * <p>
         * The following lists the names, descriptions, and values of the special request parameters that the
         * <code>SetTopicAttributes</code> action uses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
         * endpoints.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FilterPolicy</code> – The simple JSON object that lets your subscriber receive only a subset of
         * messages, rather than receiving every message published to the topic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RawMessageDelivery</code> – When set to <code>true</code>, enables raw message delivery to Amazon SQS
         * or HTTP/S endpoints. This eliminates the need for the endpoints to process JSON formatting, which is
         * otherwise created for Amazon SNS metadata.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributes
         *        A map of attributes with their corresponding values.</p>
         *        <p>
         *        The following lists the names, descriptions, and values of the special request parameters that the
         *        <code>SetTopicAttributes</code> action uses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to
         *        HTTP/S endpoints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FilterPolicy</code> – The simple JSON object that lets your subscriber receive only a subset of
         *        messages, rather than receiving every message published to the topic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RawMessageDelivery</code> – When set to <code>true</code>, enables raw message delivery to
         *        Amazon SQS or HTTP/S endpoints. This eliminates the need for the endpoints to process JSON formatting,
         *        which is otherwise created for Amazon SNS metadata.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * Sets whether the response from the <code>Subscribe</code> request includes the subscription ARN, even if the
         * subscription is not yet confirmed.
         * </p>
         * <p>
         * If you set this parameter to <code>false</code>, the response includes the ARN for confirmed subscriptions,
         * but it includes an ARN value of "pending subscription" for subscriptions that are not yet confirmed. A
         * subscription becomes confirmed when the subscriber calls the <code>ConfirmSubscription</code> action with a
         * confirmation token.
         * </p>
         * <p>
         * If you set this parameter to <code>true</code>, the response includes the ARN in all cases, even if the
         * subscription is not yet confirmed.
         * </p>
         * <p>
         * The default value is <code>false</code>.
         * </p>
         * 
         * @param returnSubscriptionArn
         *        Sets whether the response from the <code>Subscribe</code> request includes the subscription ARN, even
         *        if the subscription is not yet confirmed.</p>
         *        <p>
         *        If you set this parameter to <code>false</code>, the response includes the ARN for confirmed
         *        subscriptions, but it includes an ARN value of "pending subscription" for subscriptions that are not
         *        yet confirmed. A subscription becomes confirmed when the subscriber calls the
         *        <code>ConfirmSubscription</code> action with a confirmation token.
         *        </p>
         *        <p>
         *        If you set this parameter to <code>true</code>, the response includes the ARN in all cases, even if
         *        the subscription is not yet confirmed.
         *        </p>
         *        <p>
         *        The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnSubscriptionArn(Boolean returnSubscriptionArn);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SnsRequest.BuilderImpl implements Builder {
        private String topicArn;

        private String protocol;

        private String endpoint;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private Boolean returnSubscriptionArn;

        private BuilderImpl() {
        }

        private BuilderImpl(SubscribeRequest model) {
            super(model);
            topicArn(model.topicArn);
            protocol(model.protocol);
            endpoint(model.endpoint);
            attributes(model.attributes);
            returnSubscriptionArn(model.returnSubscriptionArn);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        public final String getProtocol() {
            return protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        public final Map<String, String> getAttributes() {
            return attributes;
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = SubscriptionAttributesMapCopier.copy(attributes);
            return this;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = SubscriptionAttributesMapCopier.copy(attributes);
        }

        public final Boolean getReturnSubscriptionArn() {
            return returnSubscriptionArn;
        }

        @Override
        public final Builder returnSubscriptionArn(Boolean returnSubscriptionArn) {
            this.returnSubscriptionArn = returnSubscriptionArn;
            return this;
        }

        public final void setReturnSubscriptionArn(Boolean returnSubscriptionArn) {
            this.returnSubscriptionArn = returnSubscriptionArn;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SubscribeRequest build() {
            return new SubscribeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
