/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.socialmessaging;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.socialmessaging.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.socialmessaging.internal.SocialMessagingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.socialmessaging.model.AccessDeniedByMetaException;
import software.amazon.awssdk.services.socialmessaging.model.AccessDeniedException;
import software.amazon.awssdk.services.socialmessaging.model.AssociateWhatsAppBusinessAccountRequest;
import software.amazon.awssdk.services.socialmessaging.model.AssociateWhatsAppBusinessAccountResponse;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateFromLibraryRequest;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateFromLibraryResponse;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateMediaRequest;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateMediaResponse;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateRequest;
import software.amazon.awssdk.services.socialmessaging.model.CreateWhatsAppMessageTemplateResponse;
import software.amazon.awssdk.services.socialmessaging.model.DeleteWhatsAppMessageMediaRequest;
import software.amazon.awssdk.services.socialmessaging.model.DeleteWhatsAppMessageMediaResponse;
import software.amazon.awssdk.services.socialmessaging.model.DeleteWhatsAppMessageTemplateRequest;
import software.amazon.awssdk.services.socialmessaging.model.DeleteWhatsAppMessageTemplateResponse;
import software.amazon.awssdk.services.socialmessaging.model.DependencyException;
import software.amazon.awssdk.services.socialmessaging.model.DisassociateWhatsAppBusinessAccountRequest;
import software.amazon.awssdk.services.socialmessaging.model.DisassociateWhatsAppBusinessAccountResponse;
import software.amazon.awssdk.services.socialmessaging.model.GetLinkedWhatsAppBusinessAccountPhoneNumberRequest;
import software.amazon.awssdk.services.socialmessaging.model.GetLinkedWhatsAppBusinessAccountPhoneNumberResponse;
import software.amazon.awssdk.services.socialmessaging.model.GetLinkedWhatsAppBusinessAccountRequest;
import software.amazon.awssdk.services.socialmessaging.model.GetLinkedWhatsAppBusinessAccountResponse;
import software.amazon.awssdk.services.socialmessaging.model.GetWhatsAppMessageMediaRequest;
import software.amazon.awssdk.services.socialmessaging.model.GetWhatsAppMessageMediaResponse;
import software.amazon.awssdk.services.socialmessaging.model.GetWhatsAppMessageTemplateRequest;
import software.amazon.awssdk.services.socialmessaging.model.GetWhatsAppMessageTemplateResponse;
import software.amazon.awssdk.services.socialmessaging.model.InternalServiceException;
import software.amazon.awssdk.services.socialmessaging.model.InvalidParametersException;
import software.amazon.awssdk.services.socialmessaging.model.LimitExceededException;
import software.amazon.awssdk.services.socialmessaging.model.ListLinkedWhatsAppBusinessAccountsRequest;
import software.amazon.awssdk.services.socialmessaging.model.ListLinkedWhatsAppBusinessAccountsResponse;
import software.amazon.awssdk.services.socialmessaging.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.socialmessaging.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.socialmessaging.model.ListWhatsAppMessageTemplatesRequest;
import software.amazon.awssdk.services.socialmessaging.model.ListWhatsAppMessageTemplatesResponse;
import software.amazon.awssdk.services.socialmessaging.model.ListWhatsAppTemplateLibraryRequest;
import software.amazon.awssdk.services.socialmessaging.model.ListWhatsAppTemplateLibraryResponse;
import software.amazon.awssdk.services.socialmessaging.model.PostWhatsAppMessageMediaRequest;
import software.amazon.awssdk.services.socialmessaging.model.PostWhatsAppMessageMediaResponse;
import software.amazon.awssdk.services.socialmessaging.model.PutWhatsAppBusinessAccountEventDestinationsRequest;
import software.amazon.awssdk.services.socialmessaging.model.PutWhatsAppBusinessAccountEventDestinationsResponse;
import software.amazon.awssdk.services.socialmessaging.model.ResourceNotFoundException;
import software.amazon.awssdk.services.socialmessaging.model.SendWhatsAppMessageRequest;
import software.amazon.awssdk.services.socialmessaging.model.SendWhatsAppMessageResponse;
import software.amazon.awssdk.services.socialmessaging.model.SocialMessagingException;
import software.amazon.awssdk.services.socialmessaging.model.TagResourceRequest;
import software.amazon.awssdk.services.socialmessaging.model.TagResourceResponse;
import software.amazon.awssdk.services.socialmessaging.model.ThrottledRequestException;
import software.amazon.awssdk.services.socialmessaging.model.UntagResourceRequest;
import software.amazon.awssdk.services.socialmessaging.model.UntagResourceResponse;
import software.amazon.awssdk.services.socialmessaging.model.UpdateWhatsAppMessageTemplateRequest;
import software.amazon.awssdk.services.socialmessaging.model.UpdateWhatsAppMessageTemplateResponse;
import software.amazon.awssdk.services.socialmessaging.model.ValidationException;
import software.amazon.awssdk.services.socialmessaging.transform.AssociateWhatsAppBusinessAccountRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.CreateWhatsAppMessageTemplateFromLibraryRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.CreateWhatsAppMessageTemplateMediaRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.CreateWhatsAppMessageTemplateRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.DeleteWhatsAppMessageMediaRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.DeleteWhatsAppMessageTemplateRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.DisassociateWhatsAppBusinessAccountRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.GetLinkedWhatsAppBusinessAccountPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.GetLinkedWhatsAppBusinessAccountRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.GetWhatsAppMessageMediaRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.GetWhatsAppMessageTemplateRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.ListLinkedWhatsAppBusinessAccountsRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.ListWhatsAppMessageTemplatesRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.ListWhatsAppTemplateLibraryRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.PostWhatsAppMessageMediaRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.PutWhatsAppBusinessAccountEventDestinationsRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.SendWhatsAppMessageRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.socialmessaging.transform.UpdateWhatsAppMessageTemplateRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SocialMessagingAsyncClient}.
 *
 * @see SocialMessagingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSocialMessagingAsyncClient implements SocialMessagingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSocialMessagingAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSocialMessagingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "SocialMessaging" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This is only used through the Amazon Web Services console during sign-up to associate your WhatsApp Business
     * Account to your Amazon Web Services account.
     * </p>
     *
     * @param associateWhatsAppBusinessAccountRequest
     * @return A Java Future containing the result of the AssociateWhatsAppBusinessAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>LimitExceededException The request was denied because it would exceed one or more service quotas or
     *         limits.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.AssociateWhatsAppBusinessAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/AssociateWhatsAppBusinessAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWhatsAppBusinessAccountResponse> associateWhatsAppBusinessAccount(
            AssociateWhatsAppBusinessAccountRequest associateWhatsAppBusinessAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateWhatsAppBusinessAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateWhatsAppBusinessAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateWhatsAppBusinessAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWhatsAppBusinessAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateWhatsAppBusinessAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateWhatsAppBusinessAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWhatsAppBusinessAccountRequest, AssociateWhatsAppBusinessAccountResponse>()
                            .withOperationName("AssociateWhatsAppBusinessAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateWhatsAppBusinessAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateWhatsAppBusinessAccountRequest));
            CompletableFuture<AssociateWhatsAppBusinessAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new WhatsApp message template from a custom definition.
     * </p>
     *
     * @param createWhatsAppMessageTemplateRequest
     * @return A Java Future containing the result of the CreateWhatsAppMessageTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.CreateWhatsAppMessageTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/CreateWhatsAppMessageTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatsAppMessageTemplateResponse> createWhatsAppMessageTemplate(
            CreateWhatsAppMessageTemplateRequest createWhatsAppMessageTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWhatsAppMessageTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createWhatsAppMessageTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatsAppMessageTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatsAppMessageTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWhatsAppMessageTemplateResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateWhatsAppMessageTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatsAppMessageTemplateRequest, CreateWhatsAppMessageTemplateResponse>()
                            .withOperationName("CreateWhatsAppMessageTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatsAppMessageTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatsAppMessageTemplateRequest));
            CompletableFuture<CreateWhatsAppMessageTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new WhatsApp message template using a template from Meta's template library.
     * </p>
     *
     * @param createWhatsAppMessageTemplateFromLibraryRequest
     * @return A Java Future containing the result of the CreateWhatsAppMessageTemplateFromLibrary operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.CreateWhatsAppMessageTemplateFromLibrary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/CreateWhatsAppMessageTemplateFromLibrary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatsAppMessageTemplateFromLibraryResponse> createWhatsAppMessageTemplateFromLibrary(
            CreateWhatsAppMessageTemplateFromLibraryRequest createWhatsAppMessageTemplateFromLibraryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                createWhatsAppMessageTemplateFromLibraryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createWhatsAppMessageTemplateFromLibraryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatsAppMessageTemplateFromLibrary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatsAppMessageTemplateFromLibraryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateWhatsAppMessageTemplateFromLibraryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateWhatsAppMessageTemplateFromLibraryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatsAppMessageTemplateFromLibraryRequest, CreateWhatsAppMessageTemplateFromLibraryResponse>()
                            .withOperationName("CreateWhatsAppMessageTemplateFromLibrary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatsAppMessageTemplateFromLibraryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatsAppMessageTemplateFromLibraryRequest));
            CompletableFuture<CreateWhatsAppMessageTemplateFromLibraryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads media for use in a WhatsApp message template.
     * </p>
     *
     * @param createWhatsAppMessageTemplateMediaRequest
     * @return A Java Future containing the result of the CreateWhatsAppMessageTemplateMedia operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.CreateWhatsAppMessageTemplateMedia
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/CreateWhatsAppMessageTemplateMedia"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatsAppMessageTemplateMediaResponse> createWhatsAppMessageTemplateMedia(
            CreateWhatsAppMessageTemplateMediaRequest createWhatsAppMessageTemplateMediaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWhatsAppMessageTemplateMediaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createWhatsAppMessageTemplateMediaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatsAppMessageTemplateMedia");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatsAppMessageTemplateMediaResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateWhatsAppMessageTemplateMediaResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateWhatsAppMessageTemplateMediaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatsAppMessageTemplateMediaRequest, CreateWhatsAppMessageTemplateMediaResponse>()
                            .withOperationName("CreateWhatsAppMessageTemplateMedia").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatsAppMessageTemplateMediaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatsAppMessageTemplateMediaRequest));
            CompletableFuture<CreateWhatsAppMessageTemplateMediaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a media object from the WhatsApp service. If the object is still in an Amazon S3 bucket you should delete
     * it from there too.
     * </p>
     *
     * @param deleteWhatsAppMessageMediaRequest
     * @return A Java Future containing the result of the DeleteWhatsAppMessageMedia operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedByMetaException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.DeleteWhatsAppMessageMedia
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/DeleteWhatsAppMessageMedia"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWhatsAppMessageMediaResponse> deleteWhatsAppMessageMedia(
            DeleteWhatsAppMessageMediaRequest deleteWhatsAppMessageMediaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWhatsAppMessageMediaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWhatsAppMessageMediaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWhatsAppMessageMedia");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWhatsAppMessageMediaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWhatsAppMessageMediaResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteWhatsAppMessageMediaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWhatsAppMessageMediaRequest, DeleteWhatsAppMessageMediaResponse>()
                            .withOperationName("DeleteWhatsAppMessageMedia").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWhatsAppMessageMediaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWhatsAppMessageMediaRequest));
            CompletableFuture<DeleteWhatsAppMessageMediaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a WhatsApp message template.
     * </p>
     *
     * @param deleteWhatsAppMessageTemplateRequest
     * @return A Java Future containing the result of the DeleteWhatsAppMessageTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.DeleteWhatsAppMessageTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/DeleteWhatsAppMessageTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWhatsAppMessageTemplateResponse> deleteWhatsAppMessageTemplate(
            DeleteWhatsAppMessageTemplateRequest deleteWhatsAppMessageTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWhatsAppMessageTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteWhatsAppMessageTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWhatsAppMessageTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWhatsAppMessageTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWhatsAppMessageTemplateResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteWhatsAppMessageTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWhatsAppMessageTemplateRequest, DeleteWhatsAppMessageTemplateResponse>()
                            .withOperationName("DeleteWhatsAppMessageTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWhatsAppMessageTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWhatsAppMessageTemplateRequest));
            CompletableFuture<DeleteWhatsAppMessageTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociate a WhatsApp Business Account (WABA) from your Amazon Web Services account.
     * </p>
     *
     * @param disassociateWhatsAppBusinessAccountRequest
     * @return A Java Future containing the result of the DisassociateWhatsAppBusinessAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.DisassociateWhatsAppBusinessAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/DisassociateWhatsAppBusinessAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWhatsAppBusinessAccountResponse> disassociateWhatsAppBusinessAccount(
            DisassociateWhatsAppBusinessAccountRequest disassociateWhatsAppBusinessAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateWhatsAppBusinessAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateWhatsAppBusinessAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateWhatsAppBusinessAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWhatsAppBusinessAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateWhatsAppBusinessAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateWhatsAppBusinessAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWhatsAppBusinessAccountRequest, DisassociateWhatsAppBusinessAccountResponse>()
                            .withOperationName("DisassociateWhatsAppBusinessAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateWhatsAppBusinessAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateWhatsAppBusinessAccountRequest));
            CompletableFuture<DisassociateWhatsAppBusinessAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the details of your linked WhatsApp Business Account.
     * </p>
     *
     * @param getLinkedWhatsAppBusinessAccountRequest
     * @return A Java Future containing the result of the GetLinkedWhatsAppBusinessAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.GetLinkedWhatsAppBusinessAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/GetLinkedWhatsAppBusinessAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLinkedWhatsAppBusinessAccountResponse> getLinkedWhatsAppBusinessAccount(
            GetLinkedWhatsAppBusinessAccountRequest getLinkedWhatsAppBusinessAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLinkedWhatsAppBusinessAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLinkedWhatsAppBusinessAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLinkedWhatsAppBusinessAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLinkedWhatsAppBusinessAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetLinkedWhatsAppBusinessAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetLinkedWhatsAppBusinessAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLinkedWhatsAppBusinessAccountRequest, GetLinkedWhatsAppBusinessAccountResponse>()
                            .withOperationName("GetLinkedWhatsAppBusinessAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLinkedWhatsAppBusinessAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLinkedWhatsAppBusinessAccountRequest));
            CompletableFuture<GetLinkedWhatsAppBusinessAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use your WhatsApp phone number id to get the WABA account id and phone number details.
     * </p>
     *
     * @param getLinkedWhatsAppBusinessAccountPhoneNumberRequest
     * @return A Java Future containing the result of the GetLinkedWhatsAppBusinessAccountPhoneNumber operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.GetLinkedWhatsAppBusinessAccountPhoneNumber
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/GetLinkedWhatsAppBusinessAccountPhoneNumber"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLinkedWhatsAppBusinessAccountPhoneNumberResponse> getLinkedWhatsAppBusinessAccountPhoneNumber(
            GetLinkedWhatsAppBusinessAccountPhoneNumberRequest getLinkedWhatsAppBusinessAccountPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getLinkedWhatsAppBusinessAccountPhoneNumberRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLinkedWhatsAppBusinessAccountPhoneNumberRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLinkedWhatsAppBusinessAccountPhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLinkedWhatsAppBusinessAccountPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetLinkedWhatsAppBusinessAccountPhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetLinkedWhatsAppBusinessAccountPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLinkedWhatsAppBusinessAccountPhoneNumberRequest, GetLinkedWhatsAppBusinessAccountPhoneNumberResponse>()
                            .withOperationName("GetLinkedWhatsAppBusinessAccountPhoneNumber")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLinkedWhatsAppBusinessAccountPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLinkedWhatsAppBusinessAccountPhoneNumberRequest));
            CompletableFuture<GetLinkedWhatsAppBusinessAccountPhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get a media file from the WhatsApp service. On successful completion the media file is retrieved from Meta and
     * stored in the specified Amazon S3 bucket. Use either <code>destinationS3File</code> or
     * <code>destinationS3PresignedUrl</code> for the destination. If both are used then an
     * <code>InvalidParameterException</code> is returned.
     * </p>
     *
     * @param getWhatsAppMessageMediaRequest
     * @return A Java Future containing the result of the GetWhatsAppMessageMedia operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedByMetaException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.GetWhatsAppMessageMedia
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/GetWhatsAppMessageMedia"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWhatsAppMessageMediaResponse> getWhatsAppMessageMedia(
            GetWhatsAppMessageMediaRequest getWhatsAppMessageMediaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWhatsAppMessageMediaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWhatsAppMessageMediaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWhatsAppMessageMedia");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWhatsAppMessageMediaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWhatsAppMessageMediaResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWhatsAppMessageMediaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWhatsAppMessageMediaRequest, GetWhatsAppMessageMediaResponse>()
                            .withOperationName("GetWhatsAppMessageMedia").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWhatsAppMessageMediaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWhatsAppMessageMediaRequest));
            CompletableFuture<GetWhatsAppMessageMediaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specific WhatsApp message template.
     * </p>
     *
     * @param getWhatsAppMessageTemplateRequest
     * @return A Java Future containing the result of the GetWhatsAppMessageTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.GetWhatsAppMessageTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/GetWhatsAppMessageTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWhatsAppMessageTemplateResponse> getWhatsAppMessageTemplate(
            GetWhatsAppMessageTemplateRequest getWhatsAppMessageTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWhatsAppMessageTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWhatsAppMessageTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWhatsAppMessageTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWhatsAppMessageTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWhatsAppMessageTemplateResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWhatsAppMessageTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWhatsAppMessageTemplateRequest, GetWhatsAppMessageTemplateResponse>()
                            .withOperationName("GetWhatsAppMessageTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWhatsAppMessageTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWhatsAppMessageTemplateRequest));
            CompletableFuture<GetWhatsAppMessageTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all WhatsApp Business Accounts linked to your Amazon Web Services account.
     * </p>
     *
     * @param listLinkedWhatsAppBusinessAccountsRequest
     * @return A Java Future containing the result of the ListLinkedWhatsAppBusinessAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.ListLinkedWhatsAppBusinessAccounts
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/ListLinkedWhatsAppBusinessAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinkedWhatsAppBusinessAccountsResponse> listLinkedWhatsAppBusinessAccounts(
            ListLinkedWhatsAppBusinessAccountsRequest listLinkedWhatsAppBusinessAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLinkedWhatsAppBusinessAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLinkedWhatsAppBusinessAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinkedWhatsAppBusinessAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinkedWhatsAppBusinessAccountsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListLinkedWhatsAppBusinessAccountsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLinkedWhatsAppBusinessAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinkedWhatsAppBusinessAccountsRequest, ListLinkedWhatsAppBusinessAccountsResponse>()
                            .withOperationName("ListLinkedWhatsAppBusinessAccounts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLinkedWhatsAppBusinessAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLinkedWhatsAppBusinessAccountsRequest));
            CompletableFuture<ListLinkedWhatsAppBusinessAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all tags associated with a resource, such as a phone number or WABA.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists WhatsApp message templates for a specific WhatsApp Business Account.
     * </p>
     *
     * @param listWhatsAppMessageTemplatesRequest
     * @return A Java Future containing the result of the ListWhatsAppMessageTemplates operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.ListWhatsAppMessageTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/ListWhatsAppMessageTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWhatsAppMessageTemplatesResponse> listWhatsAppMessageTemplates(
            ListWhatsAppMessageTemplatesRequest listWhatsAppMessageTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWhatsAppMessageTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWhatsAppMessageTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWhatsAppMessageTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWhatsAppMessageTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWhatsAppMessageTemplatesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWhatsAppMessageTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWhatsAppMessageTemplatesRequest, ListWhatsAppMessageTemplatesResponse>()
                            .withOperationName("ListWhatsAppMessageTemplates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWhatsAppMessageTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWhatsAppMessageTemplatesRequest));
            CompletableFuture<ListWhatsAppMessageTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists templates available in Meta's template library for WhatsApp messaging.
     * </p>
     *
     * @param listWhatsAppTemplateLibraryRequest
     * @return A Java Future containing the result of the ListWhatsAppTemplateLibrary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.ListWhatsAppTemplateLibrary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/ListWhatsAppTemplateLibrary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWhatsAppTemplateLibraryResponse> listWhatsAppTemplateLibrary(
            ListWhatsAppTemplateLibraryRequest listWhatsAppTemplateLibraryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWhatsAppTemplateLibraryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWhatsAppTemplateLibraryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWhatsAppTemplateLibrary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWhatsAppTemplateLibraryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWhatsAppTemplateLibraryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWhatsAppTemplateLibraryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWhatsAppTemplateLibraryRequest, ListWhatsAppTemplateLibraryResponse>()
                            .withOperationName("ListWhatsAppTemplateLibrary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWhatsAppTemplateLibraryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWhatsAppTemplateLibraryRequest));
            CompletableFuture<ListWhatsAppTemplateLibraryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Upload a media file to the WhatsApp service. Only the specified <code>originationPhoneNumberId</code> has the
     * permissions to send the media file when using <a
     * href="https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_SendWhatsAppMessage.html"
     * >SendWhatsAppMessage</a>. You must use either <code>sourceS3File</code> or <code>sourceS3PresignedUrl</code> for
     * the source. If both or neither are specified then an <code>InvalidParameterException</code> is returned.
     * </p>
     *
     * @param postWhatsAppMessageMediaRequest
     * @return A Java Future containing the result of the PostWhatsAppMessageMedia operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedByMetaException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.PostWhatsAppMessageMedia
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/PostWhatsAppMessageMedia"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PostWhatsAppMessageMediaResponse> postWhatsAppMessageMedia(
            PostWhatsAppMessageMediaRequest postWhatsAppMessageMediaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postWhatsAppMessageMediaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postWhatsAppMessageMediaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostWhatsAppMessageMedia");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostWhatsAppMessageMediaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PostWhatsAppMessageMediaResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PostWhatsAppMessageMediaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostWhatsAppMessageMediaRequest, PostWhatsAppMessageMediaResponse>()
                            .withOperationName("PostWhatsAppMessageMedia").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PostWhatsAppMessageMediaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(postWhatsAppMessageMediaRequest));
            CompletableFuture<PostWhatsAppMessageMediaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add an event destination to log event data from WhatsApp for a WhatsApp Business Account (WABA). A WABA can only
     * have one event destination at a time. All resources associated with the WABA use the same event destination.
     * </p>
     *
     * @param putWhatsAppBusinessAccountEventDestinationsRequest
     * @return A Java Future containing the result of the PutWhatsAppBusinessAccountEventDestinations operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.PutWhatsAppBusinessAccountEventDestinations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/PutWhatsAppBusinessAccountEventDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutWhatsAppBusinessAccountEventDestinationsResponse> putWhatsAppBusinessAccountEventDestinations(
            PutWhatsAppBusinessAccountEventDestinationsRequest putWhatsAppBusinessAccountEventDestinationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                putWhatsAppBusinessAccountEventDestinationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putWhatsAppBusinessAccountEventDestinationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutWhatsAppBusinessAccountEventDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutWhatsAppBusinessAccountEventDestinationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutWhatsAppBusinessAccountEventDestinationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutWhatsAppBusinessAccountEventDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutWhatsAppBusinessAccountEventDestinationsRequest, PutWhatsAppBusinessAccountEventDestinationsResponse>()
                            .withOperationName("PutWhatsAppBusinessAccountEventDestinations")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutWhatsAppBusinessAccountEventDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putWhatsAppBusinessAccountEventDestinationsRequest));
            CompletableFuture<PutWhatsAppBusinessAccountEventDestinationsResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Send a WhatsApp message. For examples of sending a message using the Amazon Web Services CLI, see <a
     * href="https://docs.aws.amazon.com/social-messaging/latest/userguide/send-message.html">Sending messages</a> in
     * the <i> <i>Amazon Web Services End User Messaging Social User Guide</i> </i>.
     * </p>
     *
     * @param sendWhatsAppMessageRequest
     * @return A Java Future containing the result of the SendWhatsAppMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.SendWhatsAppMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/SendWhatsAppMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendWhatsAppMessageResponse> sendWhatsAppMessage(
            SendWhatsAppMessageRequest sendWhatsAppMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendWhatsAppMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendWhatsAppMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendWhatsAppMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendWhatsAppMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendWhatsAppMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SendWhatsAppMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendWhatsAppMessageRequest, SendWhatsAppMessageResponse>()
                            .withOperationName("SendWhatsAppMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendWhatsAppMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendWhatsAppMessageRequest));
            CompletableFuture<SendWhatsAppMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites only the specified tags for the specified resource. When you specify an existing tag key, the
     * value is overwritten with the new value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing WhatsApp message template.
     * </p>
     *
     * @param updateWhatsAppMessageTemplateRequest
     * @return A Java Future containing the result of the UpdateWhatsAppMessageTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InvalidParametersException One or more parameters provided to the action are not valid.</li>
     *         <li>ThrottledRequestException The request was denied due to request throttling.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>DependencyException Thrown when performing an action because a dependency would be broken.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SocialMessagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SocialMessagingAsyncClient.UpdateWhatsAppMessageTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/socialmessaging-2024-01-01/UpdateWhatsAppMessageTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWhatsAppMessageTemplateResponse> updateWhatsAppMessageTemplate(
            UpdateWhatsAppMessageTemplateRequest updateWhatsAppMessageTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWhatsAppMessageTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateWhatsAppMessageTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SocialMessaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWhatsAppMessageTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWhatsAppMessageTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWhatsAppMessageTemplateResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ThrottledRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledRequestException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledRequestException::builder).build());
                case "AccessDeniedByMetaException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedByMetaException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedByMetaException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidParametersException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParametersException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParametersException::builder).build());
                case "DependencyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(502)
                            .exceptionBuilderSupplier(DependencyException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateWhatsAppMessageTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWhatsAppMessageTemplateRequest, UpdateWhatsAppMessageTemplateResponse>()
                            .withOperationName("UpdateWhatsAppMessageTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWhatsAppMessageTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWhatsAppMessageTemplateRequest));
            CompletableFuture<UpdateWhatsAppMessageTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SocialMessagingServiceClientConfiguration serviceClientConfiguration() {
        return new SocialMessagingServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SocialMessagingException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SocialMessagingServiceClientConfigurationBuilder serviceConfigBuilder = new SocialMessagingServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
