/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.runtime.SdkInternalList;
import software.amazon.awssdk.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the <code>Value</code> attribute has the same
 * restrictions on the content as the message body. For more information, see <code> <a>SendMessage</a>.</code>
 * </p>
 * <p>
 * <code>Name</code>, <code>type</code>, <code>value</code> and the message body must not be empty or null. All parts of
 * the message attribute, including <code>Name</code>, <code>Type</code>, and <code>Value</code>, are part of the
 * message size restriction (256 KB or 262,144 bytes).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MessageAttributeValue implements ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private final String stringValue;

    private final ByteBuffer binaryValue;

    private final List<String> stringListValues;

    private final List<ByteBuffer> binaryListValues;

    private final String dataType;

    private MessageAttributeValue(BuilderImpl builder) {
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
        this.stringListValues = builder.stringListValues;
        this.binaryListValues = builder.binaryListValues;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     */
    public String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     * </p>
     * <p>
     * {@code ByteBuffer}s are stateful. Calling their {@code get} methods changes their {@code position}. We recommend
     * using {@link java.nio.ByteBuffer#asReadOnlyBuffer()} to create a read-only view of the buffer with an independent
     * {@code position}, and calling {@code get} methods on this rather than directly on the returned {@code ByteBuffer}
     * . Doing so will ensure that anyone else using the {@code ByteBuffer} will not be affected by changes to the
     * {@code position}.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     */
    public ByteBuffer binaryValue() {
        return binaryValue;
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<String> stringListValues() {
        return stringListValues;
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<ByteBuffer> binaryListValues() {
        return binaryListValues;
    }

    /**
     * <p>
     * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
     * </p>
     * <p>
     * You can also append custom labels. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
     * >Message Attribute Data Types and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * 
     * @return Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     *         <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
     *         <p>
     *         You can also append custom labels. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
     *         >Message Attribute Data Types and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((stringValue() == null) ? 0 : stringValue().hashCode());
        hashCode = 31 * hashCode + ((binaryValue() == null) ? 0 : binaryValue().hashCode());
        hashCode = 31 * hashCode + ((stringListValues() == null) ? 0 : stringListValues().hashCode());
        hashCode = 31 * hashCode + ((binaryListValues() == null) ? 0 : binaryListValues().hashCode());
        hashCode = 31 * hashCode + ((dataType() == null) ? 0 : dataType().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        if (other.stringValue() == null ^ this.stringValue() == null) {
            return false;
        }
        if (other.stringValue() != null && !other.stringValue().equals(this.stringValue())) {
            return false;
        }
        if (other.binaryValue() == null ^ this.binaryValue() == null) {
            return false;
        }
        if (other.binaryValue() != null && !other.binaryValue().equals(this.binaryValue())) {
            return false;
        }
        if (other.stringListValues() == null ^ this.stringListValues() == null) {
            return false;
        }
        if (other.stringListValues() != null && !other.stringListValues().equals(this.stringListValues())) {
            return false;
        }
        if (other.binaryListValues() == null ^ this.binaryListValues() == null) {
            return false;
        }
        if (other.binaryListValues() != null && !other.binaryListValues().equals(this.binaryListValues())) {
            return false;
        }
        if (other.dataType() == null ^ this.dataType() == null) {
            return false;
        }
        if (other.dataType() != null && !other.dataType().equals(this.dataType())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (stringValue() != null) {
            sb.append("StringValue: ").append(stringValue()).append(",");
        }
        if (binaryValue() != null) {
            sb.append("BinaryValue: ").append(binaryValue()).append(",");
        }
        if (stringListValues() != null) {
            sb.append("StringListValues: ").append(stringListValues()).append(",");
        }
        if (binaryListValues() != null) {
            sb.append("BinaryListValues: ").append(binaryListValues()).append(",");
        }
        if (dataType() != null) {
            sb.append("DataType: ").append(dataType()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * </p>
         * 
         * @param binaryValue
         *        Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(ByteBuffer binaryValue);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(Collection<String> stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setStringListValues(java.util.Collection)} or {@link #withStringListValues(java.util.Collection)} if
         * you want to override the existing values.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(String... stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(Collection<ByteBuffer> binaryListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setBinaryListValues(java.util.Collection)} or {@link #withBinaryListValues(java.util.Collection)} if
         * you want to override the existing values.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(ByteBuffer... binaryListValues);

        /**
         * <p>
         * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
         * </p>
         * <p>
         * You can also append custom labels. For more information, see <a href=
         * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
         * >Message Attribute Data Types and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param dataType
         *        Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         *        <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
         *        <p>
         *        You can also append custom labels. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
         *        >Message Attribute Data Types and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    private static final class BuilderImpl implements Builder {
        private String stringValue;

        private ByteBuffer binaryValue;

        private List<String> stringListValues = new SdkInternalList<>();

        private List<ByteBuffer> binaryListValues = new SdkInternalList<>();

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            setStringValue(model.stringValue);
            setBinaryValue(model.binaryValue);
            setStringListValues(model.stringListValues);
            setBinaryListValues(model.binaryListValues);
            setDataType(model.dataType);
        }

        public final String getStringValue() {
            return stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue;
        }

        @Override
        public final Builder binaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
            return this;
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
        }

        public final Collection<String> getStringListValues() {
            return stringListValues;
        }

        @Override
        public final Builder stringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stringListValues(String... stringListValues) {
            if (this.stringListValues == null) {
                this.stringListValues = new SdkInternalList<>(stringListValues.length);
            }
            for (String e : stringListValues) {
                this.stringListValues.add(e);
            }
            return this;
        }

        public final void setStringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
        }

        @SafeVarargs
        public final void setStringListValues(String... stringListValues) {
            if (this.stringListValues == null) {
                this.stringListValues = new SdkInternalList<>(stringListValues.length);
            }
            for (String e : stringListValues) {
                this.stringListValues.add(e);
            }
        }

        public final Collection<ByteBuffer> getBinaryListValues() {
            return binaryListValues;
        }

        @Override
        public final Builder binaryListValues(Collection<ByteBuffer> binaryListValues) {
            this.binaryListValues = BinaryListCopier.copy(binaryListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder binaryListValues(ByteBuffer... binaryListValues) {
            if (this.binaryListValues == null) {
                this.binaryListValues = new SdkInternalList<>(binaryListValues.length);
            }
            for (ByteBuffer e : binaryListValues) {
                this.binaryListValues.add(StandardMemberCopier.copy(e));
            }
            return this;
        }

        public final void setBinaryListValues(Collection<ByteBuffer> binaryListValues) {
            this.binaryListValues = BinaryListCopier.copy(binaryListValues);
        }

        @SafeVarargs
        public final void setBinaryListValues(ByteBuffer... binaryListValues) {
            if (this.binaryListValues == null) {
                this.binaryListValues = new SdkInternalList<>(binaryListValues.length);
            }
            for (ByteBuffer e : binaryListValues) {
                this.binaryListValues.add(StandardMemberCopier.copy(e));
            }
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }
    }
}
