/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the <code>Value</code> attribute has the same
 * restrictions on the content as the message body. For more information, see <code> <a>SendMessage</a>.</code>
 * </p>
 * <p>
 * <code>Name</code>, <code>type</code>, <code>value</code> and the message body must not be empty or null. All parts of
 * the message attribute, including <code>Name</code>, <code>Type</code>, and <code>Value</code>, are part of the
 * message size restriction (256 KB or 262,144 bytes).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MessageAttributeValue implements ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private final String stringValue;

    private final ByteBuffer binaryValue;

    private final List<String> stringListValues;

    private final List<ByteBuffer> binaryListValues;

    private final String dataType;

    private MessageAttributeValue(BuilderImpl builder) {
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
        this.stringListValues = builder.stringListValues;
        this.binaryListValues = builder.binaryListValues;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     */
    public String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     */
    public ByteBuffer binaryValue() {
        return binaryValue == null ? null : binaryValue.asReadOnlyBuffer();
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<String> stringListValues() {
        return stringListValues;
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<ByteBuffer> binaryListValues() {
        return binaryListValues;
    }

    /**
     * <p>
     * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
     * </p>
     * <p>
     * You can also append custom labels. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
     * >Message Attribute Data Types and Validation</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * 
     * @return Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     *         <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
     *         <p>
     *         You can also append custom labels. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
     *         >Message Attribute Data Types and Validation</a> in the <i>Amazon Simple Queue Service Developer
     *         Guide</i>.
     */
    public String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stringValue());
        hashCode = 31 * hashCode + Objects.hashCode(binaryValue());
        hashCode = 31 * hashCode + Objects.hashCode(stringListValues());
        hashCode = 31 * hashCode + Objects.hashCode(binaryListValues());
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        return Objects.equals(stringValue(), other.stringValue()) && Objects.equals(binaryValue(), other.binaryValue())
                && Objects.equals(stringListValues(), other.stringListValues())
                && Objects.equals(binaryListValues(), other.binaryListValues()) && Objects.equals(dataType(), other.dataType());
    }

    @Override
    public String toString() {
        return ToString.builder("MessageAttributeValue").add("StringValue", stringValue()).add("BinaryValue", binaryValue())
                .add("StringListValues", stringListValues()).add("BinaryListValues", binaryListValues())
                .add("DataType", dataType()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StringValue":
            return Optional.of(clazz.cast(stringValue()));
        case "BinaryValue":
            return Optional.of(clazz.cast(binaryValue()));
        case "StringListValues":
            return Optional.of(clazz.cast(stringListValues()));
        case "BinaryListValues":
            return Optional.of(clazz.cast(binaryListValues()));
        case "DataType":
            return Optional.of(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param binaryValue
         *        Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(ByteBuffer binaryValue);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(Collection<String> stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(String... stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(Collection<ByteBuffer> binaryListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(ByteBuffer... binaryListValues);

        /**
         * <p>
         * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
         * </p>
         * <p>
         * You can also append custom labels. For more information, see <a href=
         * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
         * >Message Attribute Data Types and Validation</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * 
         * @param dataType
         *        Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         *        <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
         *        <p>
         *        You can also append custom labels. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-data-types-validation"
         *        >Message Attribute Data Types and Validation</a> in the <i>Amazon Simple Queue Service Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    static final class BuilderImpl implements Builder {
        private String stringValue;

        private ByteBuffer binaryValue;

        private List<String> stringListValues;

        private List<ByteBuffer> binaryListValues;

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            stringValue(model.stringValue);
            binaryValue(model.binaryValue);
            stringListValues(model.stringListValues);
            binaryListValues(model.binaryListValues);
            dataType(model.dataType);
        }

        public final String getStringValue() {
            return stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue;
        }

        @Override
        public final Builder binaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
            return this;
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
        }

        public final Collection<String> getStringListValues() {
            return stringListValues;
        }

        @Override
        public final Builder stringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stringListValues(String... stringListValues) {
            stringListValues(Arrays.asList(stringListValues));
            return this;
        }

        public final void setStringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
        }

        public final Collection<ByteBuffer> getBinaryListValues() {
            return binaryListValues;
        }

        @Override
        public final Builder binaryListValues(Collection<ByteBuffer> binaryListValues) {
            this.binaryListValues = BinaryListCopier.copy(binaryListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder binaryListValues(ByteBuffer... binaryListValues) {
            binaryListValues(Arrays.asList(binaryListValues));
            return this;
        }

        public final void setBinaryListValues(Collection<ByteBuffer> binaryListValues) {
            this.binaryListValues = BinaryListCopier.copy(binaryListValues);
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }
    }
}
