/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs;

import java.util.ArrayList;
import java.util.List;
import javax.annotation.Generated;
import org.w3c.dom.Node;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.AmazonServiceException;
import software.amazon.awssdk.core.SdkBaseException;
import software.amazon.awssdk.core.SdkClientException;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.ClientHandler;
import software.amazon.awssdk.core.client.SdkClientHandler;
import software.amazon.awssdk.core.config.ClientConfiguration;
import software.amazon.awssdk.core.config.SyncClientConfiguration;
import software.amazon.awssdk.core.http.DefaultErrorResponseHandler;
import software.amazon.awssdk.core.http.StaxResponseHandler;
import software.amazon.awssdk.core.runtime.transform.StandardErrorUnmarshaller;
import software.amazon.awssdk.core.runtime.transform.Unmarshaller;
import software.amazon.awssdk.services.sqs.model.AddPermissionRequest;
import software.amazon.awssdk.services.sqs.model.AddPermissionResponse;
import software.amazon.awssdk.services.sqs.model.BatchEntryIdsNotDistinctException;
import software.amazon.awssdk.services.sqs.model.BatchRequestTooLongException;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityBatchRequest;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityBatchResponse;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityRequest;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityResponse;
import software.amazon.awssdk.services.sqs.model.CreateQueueRequest;
import software.amazon.awssdk.services.sqs.model.CreateQueueResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageResponse;
import software.amazon.awssdk.services.sqs.model.DeleteQueueRequest;
import software.amazon.awssdk.services.sqs.model.DeleteQueueResponse;
import software.amazon.awssdk.services.sqs.model.EmptyBatchRequestException;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesRequest;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesResponse;
import software.amazon.awssdk.services.sqs.model.GetQueueUrlRequest;
import software.amazon.awssdk.services.sqs.model.GetQueueUrlResponse;
import software.amazon.awssdk.services.sqs.model.InvalidAttributeNameException;
import software.amazon.awssdk.services.sqs.model.InvalidBatchEntryIdException;
import software.amazon.awssdk.services.sqs.model.InvalidIdFormatException;
import software.amazon.awssdk.services.sqs.model.InvalidMessageContentsException;
import software.amazon.awssdk.services.sqs.model.ListDeadLetterSourceQueuesRequest;
import software.amazon.awssdk.services.sqs.model.ListDeadLetterSourceQueuesResponse;
import software.amazon.awssdk.services.sqs.model.ListQueuesRequest;
import software.amazon.awssdk.services.sqs.model.ListQueuesResponse;
import software.amazon.awssdk.services.sqs.model.MessageNotInflightException;
import software.amazon.awssdk.services.sqs.model.OverLimitException;
import software.amazon.awssdk.services.sqs.model.PurgeQueueInProgressException;
import software.amazon.awssdk.services.sqs.model.PurgeQueueRequest;
import software.amazon.awssdk.services.sqs.model.PurgeQueueResponse;
import software.amazon.awssdk.services.sqs.model.QueueDeletedRecentlyException;
import software.amazon.awssdk.services.sqs.model.QueueDoesNotExistException;
import software.amazon.awssdk.services.sqs.model.QueueNameExistsException;
import software.amazon.awssdk.services.sqs.model.ReceiptHandleIsInvalidException;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageRequest;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageResponse;
import software.amazon.awssdk.services.sqs.model.RemovePermissionRequest;
import software.amazon.awssdk.services.sqs.model.RemovePermissionResponse;
import software.amazon.awssdk.services.sqs.model.SQSException;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.SendMessageRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageResponse;
import software.amazon.awssdk.services.sqs.model.SetQueueAttributesRequest;
import software.amazon.awssdk.services.sqs.model.SetQueueAttributesResponse;
import software.amazon.awssdk.services.sqs.model.TooManyEntriesInBatchRequestException;
import software.amazon.awssdk.services.sqs.model.UnsupportedOperationException;
import software.amazon.awssdk.services.sqs.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.AddPermissionResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.BatchEntryIdsNotDistinctExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.BatchRequestTooLongExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityBatchResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.CreateQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.CreateQueueResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageBatchResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteQueueResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.EmptyBatchRequestExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueAttributesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueAttributesResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueUrlRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueUrlResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.InvalidAttributeNameExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.InvalidBatchEntryIdExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.InvalidIdFormatExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.InvalidMessageContentsExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ListDeadLetterSourceQueuesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListDeadLetterSourceQueuesResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListQueuesResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.MessageNotInflightExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.OverLimitExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.PurgeQueueInProgressExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.PurgeQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.PurgeQueueResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.QueueDeletedRecentlyExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.QueueDoesNotExistExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.QueueNameExistsExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ReceiptHandleIsInvalidExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.ReceiveMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ReceiveMessageResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.RemovePermissionResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageBatchResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.SetQueueAttributesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SetQueueAttributesResponseUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.TooManyEntriesInBatchRequestExceptionUnmarshaller;
import software.amazon.awssdk.services.sqs.transform.UnsupportedOperationExceptionUnmarshaller;

/**
 * Internal implementation of {@link SQSClient}.
 *
 * @see SQSClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSQSClient implements SQSClient {
    private final ClientHandler clientHandler;

    private final List<Unmarshaller<AmazonServiceException, Node>> exceptionUnmarshallers;

    private final ClientConfiguration clientConfiguration;

    protected DefaultSQSClient(SyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkClientHandler(clientConfiguration, null);
        this.exceptionUnmarshallers = init();
        this.clientConfiguration = clientConfiguration;
    }

    /**
     * <p>
     * Adds a permission to a queue for a specific <a
     * href="http://docs.aws.amazon.com/general/latest/gr/glos-chap.html#P">principal</a>. This allows sharing access to
     * the queue.
     * </p>
     * <p>
     * When you create a queue, you have full control access rights for the queue. Only you, the owner of the queue, can
     * grant or deny permissions to the queue. For more information about these permissions, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/acp-overview.html">Shared
     * Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * <code>AddPermission</code> writes an Amazon-SQS-generated policy. If you want to write your own policy, use
     * <code> <a>SetQueueAttributes</a> </code> to upload your policy. For more information about writing your own
     * policy, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/AccessPolicyLanguage.html">Using
     * The Access Policy Language</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param addPermissionRequest
     * @return Result of the AddPermission operation returned by the service.
     * @throws OverLimitException
     *         The action that you requested would violate a limit. For example, <code>ReceiveMessage</code> returns
     *         this error if the maximum number of inflight messages is reached. <code> <a>AddPermission</a> </code>
     *         returns this error if the maximum number of permissions for the queue is reached.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.AddPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddPermissionResponse addPermission(AddPermissionRequest addPermissionRequest) throws OverLimitException,
            SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<AddPermissionResponse> responseHandler = new StaxResponseHandler<AddPermissionResponse>(
                new AddPermissionResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(addPermissionRequest).withMarshaller(new AddPermissionRequestMarshaller()));
    }

    /**
     * <p>
     * Changes the visibility timeout of a specified message in a queue to a new value. The maximum allowed timeout
     * value is 12 hours. Thus, you can't extend the timeout of a message in an existing queue to more than a total
     * visibility timeout of 12 hours. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * For example, you have a message and with the default visibility timeout of 5 minutes. After 3 minutes, you call
     * <code>ChangeMessageVisiblity</code> with a timeout of 10 minutes. At that time, the timeout for the message is
     * extended by 10 minutes beyond the time of the <code>ChangeMessageVisibility</code> action. This results in a
     * total visibility timeout of 13 minutes. You can continue to call the <code>ChangeMessageVisibility</code> to
     * extend the visibility timeout to a maximum of 12 hours. If you try to extend the visibility timeout beyond 12
     * hours, your request is rejected.
     * </p>
     * <p>
     * A message is considered to be <i>in flight</i> after it's received from a queue by a consumer, but not yet
     * deleted from the queue.
     * </p>
     * <p>
     * For standard queues, there can be a maximum of 120,000 inflight messages per queue. If you reach this limit,
     * Amazon SQS returns the <code>OverLimit</code> error message. To avoid reaching the limit, you should delete
     * messages from the queue after they're processed. You can also increase the number of queues you use to process
     * your messages.
     * </p>
     * <p>
     * For FIFO queues, there can be a maximum of 20,000 inflight messages per queue. If you reach this limit, Amazon
     * SQS returns no error messages.
     * </p>
     * <important>
     * <p>
     * If you attempt to set the <code>VisibilityTimeout</code> to a value greater than the maximum time left, Amazon
     * SQS returns an error. Amazon SQS doesn't automatically recalculate and increase the timeout to the maximum
     * remaining time.
     * </p>
     * <p>
     * Unlike with a queue, when you change the visibility timeout for a specific message the timeout value is applied
     * immediately but isn't saved in memory for that message. If you don't delete a message after it is received, the
     * visibility timeout for the message reverts to the original timeout value (not to the value you set using the
     * <code>ChangeMessageVisibility</code> action) the next time the message is received.
     * </p>
     * </important>
     *
     * @param changeMessageVisibilityRequest
     * @return Result of the ChangeMessageVisibility operation returned by the service.
     * @throws MessageNotInflightException
     *         The message referred to isn't in flight.
     * @throws ReceiptHandleIsInvalidException
     *         The receipt handle provided isn't valid.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.ChangeMessageVisibility
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ChangeMessageVisibility" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ChangeMessageVisibilityResponse changeMessageVisibility(ChangeMessageVisibilityRequest changeMessageVisibilityRequest)
            throws MessageNotInflightException, ReceiptHandleIsInvalidException, SdkBaseException, SdkClientException,
            SQSException {

        StaxResponseHandler<ChangeMessageVisibilityResponse> responseHandler = new StaxResponseHandler<ChangeMessageVisibilityResponse>(
                new ChangeMessageVisibilityResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<ChangeMessageVisibilityRequest, ChangeMessageVisibilityResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(changeMessageVisibilityRequest).withMarshaller(new ChangeMessageVisibilityRequestMarshaller()));
    }

    /**
     * <p>
     * Changes the visibility timeout of multiple messages. This is a batch version of
     * <code> <a>ChangeMessageVisibility</a>.</code> The result of the action on each message is reported individually
     * in the response. You can send up to 10 <code> <a>ChangeMessageVisibility</a> </code> requests with each
     * <code>ChangeMessageVisibilityBatch</code> action.
     * </p>
     * <important>
     * <p>
     * Because the batch request can result in a combination of successful and unsuccessful actions, you should check
     * for batch errors even when the call returns an HTTP status code of <code>200</code>.
     * </p>
     * </important> <note>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param changeMessageVisibilityBatchRequest
     * @return Result of the ChangeMessageVisibilityBatch operation returned by the service.
     * @throws TooManyEntriesInBatchRequestException
     *         The batch request contains more entries than permissible.
     * @throws EmptyBatchRequestException
     *         The batch request doesn't contain any entries.
     * @throws BatchEntryIdsNotDistinctException
     *         Two or more batch entries in the request have the same <code>Id</code>.
     * @throws InvalidBatchEntryIdException
     *         The <code>Id</code> of a batch entry in a batch request doesn't abide by the specification.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.ChangeMessageVisibilityBatch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ChangeMessageVisibilityBatch"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ChangeMessageVisibilityBatchResponse changeMessageVisibilityBatch(
            ChangeMessageVisibilityBatchRequest changeMessageVisibilityBatchRequest)
            throws TooManyEntriesInBatchRequestException, EmptyBatchRequestException, BatchEntryIdsNotDistinctException,
            InvalidBatchEntryIdException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<ChangeMessageVisibilityBatchResponse> responseHandler = new StaxResponseHandler<ChangeMessageVisibilityBatchResponse>(
                new ChangeMessageVisibilityBatchResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler
                .execute(new ClientExecutionParams<ChangeMessageVisibilityBatchRequest, ChangeMessageVisibilityBatchResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(changeMessageVisibilityBatchRequest)
                        .withMarshaller(new ChangeMessageVisibilityBatchRequestMarshaller()));
    }

    /**
     * <p>
     * Creates a new standard or FIFO queue. You can pass one or more attributes in the request. Keep the following
     * caveats in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard queue.
     * </p>
     * <note>
     * <p>
     * You can't change the queue type after you create it and you can't convert an existing standard queue into a FIFO
     * queue. You must either create a new FIFO queue for your application or delete your existing standard queue and
     * recreate it as a FIFO queue. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-moving">
     * Moving From a Standard Queue to a FIFO Queue</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * If you don't provide a value for an attribute, the queue is created with the default value for the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you delete a queue, you must wait at least 60 seconds before creating a queue with the same name.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To successfully create a new queue, you must provide a queue name that adheres to the <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/limits-queues.html">limits
     * related to queues</a> and is unique within the scope of your queues.
     * </p>
     * <p>
     * To get the queue URL, use the <code> <a>GetQueueUrl</a> </code> action. <code> <a>GetQueueUrl</a> </code>
     * requires only the <code>QueueName</code> parameter. be aware of existing queue names:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you provide the name of an existing queue along with the exact names and values of all the queue's attributes,
     * <code>CreateQueue</code> returns the queue URL for the existing queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the queue name, attribute names, or attribute values don't match an existing queue, <code>CreateQueue</code>
     * returns an error.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param createQueueRequest
     * @return Result of the CreateQueue operation returned by the service.
     * @throws QueueDeletedRecentlyException
     *         You must wait 60 seconds after deleting a queue before you can create another one with the same name.
     * @throws QueueNameExistsException
     *         A queue already exists with this name. Amazon SQS returns this error only if the request includes
     *         attributes whose values differ from those of the existing queue.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.CreateQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/CreateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateQueueResponse createQueue(CreateQueueRequest createQueueRequest) throws QueueDeletedRecentlyException,
            QueueNameExistsException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<CreateQueueResponse> responseHandler = new StaxResponseHandler<CreateQueueResponse>(
                new CreateQueueResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<CreateQueueRequest, CreateQueueResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createQueueRequest).withMarshaller(new CreateQueueRequestMarshaller()));
    }

    /**
     * <p>
     * Deletes the specified message from the specified queue. You specify the message by using the message's <i>receipt
     * handle</i> and not the <i>MessageId</i> you receive when you send the message. Even if the message is locked by
     * another reader due to the visibility timeout setting, it is still deleted from the queue. If you leave a message
     * in the queue for longer than the queue's configured retention period, Amazon SQS automatically deletes the
     * message.
     * </p>
     * <note>
     * <p>
     * The receipt handle is associated with a specific instance of receiving the message. If you receive a message more
     * than once, the receipt handle you get each time you receive the message is different. If you don't provide the
     * most recently received receipt handle for the message when you use the <code>DeleteMessage</code> action, the
     * request succeeds, but the message might not be deleted.
     * </p>
     * <p>
     * For standard queues, it is possible to receive a message even after you delete it. This might happen on rare
     * occasions if one of the servers storing a copy of the message is unavailable when you send the request to delete
     * the message. The copy remains on the server and might be returned to you on a subsequent receive request. You
     * should ensure that your application is idempotent, so that receiving a message more than once does not cause
     * issues.
     * </p>
     * </note>
     *
     * @param deleteMessageRequest
     * @return Result of the DeleteMessage operation returned by the service.
     * @throws InvalidIdFormatException
     *         The receipt handle isn't valid for the current version.
     * @throws ReceiptHandleIsInvalidException
     *         The receipt handle provided isn't valid.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.DeleteMessage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMessageResponse deleteMessage(DeleteMessageRequest deleteMessageRequest) throws InvalidIdFormatException,
            ReceiptHandleIsInvalidException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<DeleteMessageResponse> responseHandler = new StaxResponseHandler<DeleteMessageResponse>(
                new DeleteMessageResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<DeleteMessageRequest, DeleteMessageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteMessageRequest).withMarshaller(new DeleteMessageRequestMarshaller()));
    }

    /**
     * <p>
     * Deletes up to ten messages from the specified queue. This is a batch version of
     * <code> <a>DeleteMessage</a>.</code> The result of the action on each message is reported individually in the
     * response.
     * </p>
     * <important>
     * <p>
     * Because the batch request can result in a combination of successful and unsuccessful actions, you should check
     * for batch errors even when the call returns an HTTP status code of <code>200</code>.
     * </p>
     * </important> <note>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param deleteMessageBatchRequest
     * @return Result of the DeleteMessageBatch operation returned by the service.
     * @throws TooManyEntriesInBatchRequestException
     *         The batch request contains more entries than permissible.
     * @throws EmptyBatchRequestException
     *         The batch request doesn't contain any entries.
     * @throws BatchEntryIdsNotDistinctException
     *         Two or more batch entries in the request have the same <code>Id</code>.
     * @throws InvalidBatchEntryIdException
     *         The <code>Id</code> of a batch entry in a batch request doesn't abide by the specification.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.DeleteMessageBatch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteMessageBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMessageBatchResponse deleteMessageBatch(DeleteMessageBatchRequest deleteMessageBatchRequest)
            throws TooManyEntriesInBatchRequestException, EmptyBatchRequestException, BatchEntryIdsNotDistinctException,
            InvalidBatchEntryIdException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<DeleteMessageBatchResponse> responseHandler = new StaxResponseHandler<DeleteMessageBatchResponse>(
                new DeleteMessageBatchResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<DeleteMessageBatchRequest, DeleteMessageBatchResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteMessageBatchRequest).withMarshaller(new DeleteMessageBatchRequestMarshaller()));
    }

    /**
     * <p>
     * Deletes the queue specified by the <code>QueueUrl</code>, even if the queue is empty. If the specified queue
     * doesn't exist, Amazon SQS returns a successful response.
     * </p>
     * <important>
     * <p>
     * Be careful with the <code>DeleteQueue</code> action: When you delete a queue, any messages in the queue are no
     * longer available.
     * </p>
     * </important>
     * <p>
     * When you delete a queue, the deletion process takes up to 60 seconds. Requests you send involving that queue
     * during the 60 seconds might succeed. For example, a <code> <a>SendMessage</a> </code> request might succeed, but
     * after 60 seconds the queue and the message you sent no longer exist.
     * </p>
     * <p>
     * When you delete a queue, you must wait at least 60 seconds before creating a queue with the same name.
     * </p>
     *
     * @param deleteQueueRequest
     * @return Result of the DeleteQueue operation returned by the service.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.DeleteQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteQueueResponse deleteQueue(DeleteQueueRequest deleteQueueRequest) throws SdkBaseException, SdkClientException,
            SQSException {

        StaxResponseHandler<DeleteQueueResponse> responseHandler = new StaxResponseHandler<DeleteQueueResponse>(
                new DeleteQueueResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<DeleteQueueRequest, DeleteQueueResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteQueueRequest).withMarshaller(new DeleteQueueRequestMarshaller()));
    }

    /**
     * <p>
     * Gets attributes for the specified queue.
     * </p>
     * <note>
     * <p>
     * To determine whether a queue is <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO</a>, you
     * can check whether <code>QueueName</code> ends with the <code>.fifo</code> suffix.
     * </p>
     * </note> <note>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param getQueueAttributesRequest
     * @return Result of the GetQueueAttributes operation returned by the service.
     * @throws InvalidAttributeNameException
     *         The attribute referred to doesn't exist.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.GetQueueAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/GetQueueAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetQueueAttributesResponse getQueueAttributes(GetQueueAttributesRequest getQueueAttributesRequest)
            throws InvalidAttributeNameException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<GetQueueAttributesResponse> responseHandler = new StaxResponseHandler<GetQueueAttributesResponse>(
                new GetQueueAttributesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<GetQueueAttributesRequest, GetQueueAttributesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getQueueAttributesRequest).withMarshaller(new GetQueueAttributesRequestMarshaller()));
    }

    /**
     * <p>
     * Returns the URL of an existing queue. This action provides a simple way to retrieve the URL of an Amazon SQS
     * queue.
     * </p>
     * <p>
     * To access a queue that belongs to another AWS account, use the <code>QueueOwnerAWSAccountId</code> parameter to
     * specify the account ID of the queue's owner. The queue's owner must grant you permission to access the queue. For
     * more information about shared queue access, see <code> <a>AddPermission</a> </code> or see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/acp-overview.html">Shared
     * Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     *
     * @param getQueueUrlRequest
     * @return Result of the GetQueueUrl operation returned by the service.
     * @throws QueueDoesNotExistException
     *         The queue referred to doesn't exist.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.GetQueueUrl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/GetQueueUrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetQueueUrlResponse getQueueUrl(GetQueueUrlRequest getQueueUrlRequest) throws QueueDoesNotExistException,
            SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<GetQueueUrlResponse> responseHandler = new StaxResponseHandler<GetQueueUrlResponse>(
                new GetQueueUrlResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<GetQueueUrlRequest, GetQueueUrlResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getQueueUrlRequest).withMarshaller(new GetQueueUrlRequestMarshaller()));
    }

    /**
     * <p>
     * Returns a list of your queues that have the <code>RedrivePolicy</code> queue attribute configured with a dead
     * letter queue.
     * </p>
     * <p>
     * For more information about using dead letter queues, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >Using Amazon SQS Dead Letter Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     *
     * @param listDeadLetterSourceQueuesRequest
     * @return Result of the ListDeadLetterSourceQueues operation returned by the service.
     * @throws QueueDoesNotExistException
     *         The queue referred to doesn't exist.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.ListDeadLetterSourceQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListDeadLetterSourceQueues" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDeadLetterSourceQueuesResponse listDeadLetterSourceQueues(
            ListDeadLetterSourceQueuesRequest listDeadLetterSourceQueuesRequest) throws QueueDoesNotExistException,
            SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<ListDeadLetterSourceQueuesResponse> responseHandler = new StaxResponseHandler<ListDeadLetterSourceQueuesResponse>(
                new ListDeadLetterSourceQueuesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler
                .execute(new ClientExecutionParams<ListDeadLetterSourceQueuesRequest, ListDeadLetterSourceQueuesResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(listDeadLetterSourceQueuesRequest)
                        .withMarshaller(new ListDeadLetterSourceQueuesRequestMarshaller()));
    }

    /**
     * <p>
     * Returns a list of your queues. The maximum number of queues that can be returned is 1,000. If you specify a value
     * for the optional <code>QueueNamePrefix</code> parameter, only queues with a name that begins with the specified
     * value are returned.
     * </p>
     *
     * @param listQueuesRequest
     * @return Result of the ListQueues operation returned by the service.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.ListQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueuesResponse listQueues(ListQueuesRequest listQueuesRequest) throws SdkBaseException, SdkClientException,
            SQSException {

        StaxResponseHandler<ListQueuesResponse> responseHandler = new StaxResponseHandler<ListQueuesResponse>(
                new ListQueuesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listQueuesRequest)
                .withMarshaller(new ListQueuesRequestMarshaller()));
    }

    /**
     * <p>
     * Deletes the messages in a queue specified by the <code>QueueURL</code> parameter.
     * </p>
     * <important>
     * <p>
     * When you use the <code>PurgeQueue</code> action, you can't retrieve a message deleted from a queue.
     * </p>
     * </important>
     * <p>
     * When you purge a queue, the message deletion process takes up to 60 seconds. All messages sent to the queue
     * before calling the <code>PurgeQueue</code> action are deleted. Messages sent to the queue while it is being
     * purged might be deleted. While the queue is being purged, messages sent to the queue before
     * <code>PurgeQueue</code> is called might be received, but are deleted within the next minute.
     * </p>
     *
     * @param purgeQueueRequest
     * @return Result of the PurgeQueue operation returned by the service.
     * @throws QueueDoesNotExistException
     *         The queue referred to doesn't exist.
     * @throws PurgeQueueInProgressException
     *         Indicates that the specified queue previously received a <code>PurgeQueue</code> request within the last
     *         60 seconds (the time it can take to delete the messages in the queue).
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.PurgeQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/PurgeQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PurgeQueueResponse purgeQueue(PurgeQueueRequest purgeQueueRequest) throws QueueDoesNotExistException,
            PurgeQueueInProgressException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<PurgeQueueResponse> responseHandler = new StaxResponseHandler<PurgeQueueResponse>(
                new PurgeQueueResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<PurgeQueueRequest, PurgeQueueResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(purgeQueueRequest)
                .withMarshaller(new PurgeQueueRequestMarshaller()));
    }

    /**
     * <p>
     * Retrieves one or more messages (up to 10), from the specified queue. Using the <code>WaitTimeSeconds</code>
     * parameter enables long-poll support. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-long-polling.html">Amazon SQS
     * Long Polling</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Short poll is the default behavior where a weighted random set of machines is sampled on a
     * <code>ReceiveMessage</code> call. Thus, only the messages on the sampled machines are returned. If the number of
     * messages in the queue is small (fewer than 1,000), you most likely get fewer messages than you requested per
     * <code>ReceiveMessage</code> call. If the number of messages in the queue is extremely small, you might not
     * receive any messages in a particular <code>ReceiveMessage</code> response. If this happens, repeat the request.
     * </p>
     * <p>
     * For each message returned, the response includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The message body.
     * </p>
     * </li>
     * <li>
     * <p>
     * An MD5 digest of the message body. For information about MD5, see <a
     * href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>MessageId</code> you received when you sent the message to the queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * The receipt handle.
     * </p>
     * </li>
     * <li>
     * <p>
     * The message attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * An MD5 digest of the message attributes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The receipt handle is the identifier you must provide when deleting the message. For more information, see <a
     * href
     * ="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-message-identifiers.html"
     * >Queue and Message Identifiers</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * You can provide the <code>VisibilityTimeout</code> parameter in your request. The parameter is applied to the
     * messages that Amazon SQS returns in the response. If you don't include the parameter, the overall visibility
     * timeout for the queue is used for the returned messages. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     * >Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * A message that isn't deleted or a message whose visibility isn't extended before the visibility timeout expires
     * counts as a failed receive. Depending on the configuration of the queue, the message might be sent to the dead
     * letter queue.
     * </p>
     * <note>
     * <p>
     * In the future, new attributes might be added. If you write code that calls this action, we recommend that you
     * structure your code so that it can handle new attributes gracefully.
     * </p>
     * </note>
     *
     * @param receiveMessageRequest
     * @return Result of the ReceiveMessage operation returned by the service.
     * @throws OverLimitException
     *         The action that you requested would violate a limit. For example, <code>ReceiveMessage</code> returns
     *         this error if the maximum number of inflight messages is reached. <code> <a>AddPermission</a> </code>
     *         returns this error if the maximum number of permissions for the queue is reached.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.ReceiveMessage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ReceiveMessage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ReceiveMessageResponse receiveMessage(ReceiveMessageRequest receiveMessageRequest) throws OverLimitException,
            SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<ReceiveMessageResponse> responseHandler = new StaxResponseHandler<ReceiveMessageResponse>(
                new ReceiveMessageResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<ReceiveMessageRequest, ReceiveMessageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(receiveMessageRequest).withMarshaller(new ReceiveMessageRequestMarshaller()));
    }

    /**
     * <p>
     * Revokes any permissions in the queue policy that matches the specified <code>Label</code> parameter. Only the
     * owner of the queue can remove permissions.
     * </p>
     *
     * @param removePermissionRequest
     * @return Result of the RemovePermission operation returned by the service.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemovePermissionResponse removePermission(RemovePermissionRequest removePermissionRequest) throws SdkBaseException,
            SdkClientException, SQSException {

        StaxResponseHandler<RemovePermissionResponse> responseHandler = new StaxResponseHandler<RemovePermissionResponse>(
                new RemovePermissionResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(removePermissionRequest).withMarshaller(new RemovePermissionRequestMarshaller()));
    }

    /**
     * <p>
     * Delivers a message to the specified queue.
     * </p>
     * <important>
     * <p>
     * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed:
     * </p>
     * <p>
     * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     * </p>
     * <p>
     * Any characters not included in this list will be rejected. For more information, see the <a
     * href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     * </p>
     * </important>
     *
     * @param sendMessageRequest
     * @return Result of the SendMessage operation returned by the service.
     * @throws InvalidMessageContentsException
     *         The message contains characters outside the allowed set.
     * @throws UnsupportedOperationException
     *         Error code 400. Unsupported operation.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.SendMessage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SendMessage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendMessageResponse sendMessage(SendMessageRequest sendMessageRequest) throws InvalidMessageContentsException,
            UnsupportedOperationException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<SendMessageResponse> responseHandler = new StaxResponseHandler<SendMessageResponse>(
                new SendMessageResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<SendMessageRequest, SendMessageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(sendMessageRequest).withMarshaller(new SendMessageRequestMarshaller()));
    }

    /**
     * <p>
     * Delivers up to ten messages to the specified queue. This is a batch version of <code> <a>SendMessage</a>.</code>
     * For a FIFO queue, multiple messages within a single batch are enqueued in the order they are sent.
     * </p>
     * <p>
     * The result of sending each message is reported individually in the response. Because the batch request can result
     * in a combination of successful and unsuccessful actions, you should check for batch errors even when the call
     * returns an HTTP status code of <code>200</code>.
     * </p>
     * <p>
     * The maximum allowed individual message size and the maximum total payload size (the sum of the individual lengths
     * of all of the batched messages) are both 256 KB (262,144 bytes).
     * </p>
     * <important>
     * <p>
     * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed:
     * </p>
     * <p>
     * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     * </p>
     * <p>
     * Any characters not included in this list will be rejected. For more information, see the <a
     * href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     * </p>
     * </important>
     * <p>
     * If you don't specify the <code>DelaySeconds</code> parameter for an entry, Amazon SQS uses the default value for
     * the queue.
     * </p>
     * <note>
     * <p>
     * Some actions take lists of parameters. These lists are specified using the <code>param.n</code> notation. Values
     * of <code>n</code> are integers starting from 1. For example, a parameter list with two elements looks like this:
     * </p>
     * <p>
     * <code>&amp;Attribute.1=this</code>
     * </p>
     * <p>
     * <code>&amp;Attribute.2=that</code>
     * </p>
     * </note>
     *
     * @param sendMessageBatchRequest
     * @return Result of the SendMessageBatch operation returned by the service.
     * @throws TooManyEntriesInBatchRequestException
     *         The batch request contains more entries than permissible.
     * @throws EmptyBatchRequestException
     *         The batch request doesn't contain any entries.
     * @throws BatchEntryIdsNotDistinctException
     *         Two or more batch entries in the request have the same <code>Id</code>.
     * @throws BatchRequestTooLongException
     *         The length of all the messages put together is more than the limit.
     * @throws InvalidBatchEntryIdException
     *         The <code>Id</code> of a batch entry in a batch request doesn't abide by the specification.
     * @throws UnsupportedOperationException
     *         Error code 400. Unsupported operation.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.SendMessageBatch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SendMessageBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendMessageBatchResponse sendMessageBatch(SendMessageBatchRequest sendMessageBatchRequest)
            throws TooManyEntriesInBatchRequestException, EmptyBatchRequestException, BatchEntryIdsNotDistinctException,
            BatchRequestTooLongException, InvalidBatchEntryIdException, UnsupportedOperationException, SdkBaseException,
            SdkClientException, SQSException {

        StaxResponseHandler<SendMessageBatchResponse> responseHandler = new StaxResponseHandler<SendMessageBatchResponse>(
                new SendMessageBatchResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<SendMessageBatchRequest, SendMessageBatchResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(sendMessageBatchRequest).withMarshaller(new SendMessageBatchRequestMarshaller()));
    }

    /**
     * <p>
     * Sets the value of one or more queue attributes. When you change a queue's attributes, the change can take up to
     * 60 seconds for most of the attributes to propagate throughout the Amazon SQS system. Changes made to the
     * <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes.
     * </p>
     * <note>
     * <p>
     * In the future, new attributes might be added. If you write code that calls this action, we recommend that you
     * structure your code so that it can handle new attributes gracefully.
     * </p>
     * </note>
     *
     * @param setQueueAttributesRequest
     * @return Result of the SetQueueAttributes operation returned by the service.
     * @throws InvalidAttributeNameException
     *         The attribute referred to doesn't exist.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SQSException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SQSClient.SetQueueAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SetQueueAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetQueueAttributesResponse setQueueAttributes(SetQueueAttributesRequest setQueueAttributesRequest)
            throws InvalidAttributeNameException, SdkBaseException, SdkClientException, SQSException {

        StaxResponseHandler<SetQueueAttributesResponse> responseHandler = new StaxResponseHandler<SetQueueAttributesResponse>(
                new SetQueueAttributesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<SetQueueAttributesRequest, SetQueueAttributesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(setQueueAttributesRequest).withMarshaller(new SetQueueAttributesRequestMarshaller()));
    }

    private List<Unmarshaller<AmazonServiceException, Node>> init() {
        List<Unmarshaller<AmazonServiceException, Node>> unmarshallers = new ArrayList<>();
        unmarshallers.add(new PurgeQueueInProgressExceptionUnmarshaller());
        unmarshallers.add(new InvalidAttributeNameExceptionUnmarshaller());
        unmarshallers.add(new BatchEntryIdsNotDistinctExceptionUnmarshaller());
        unmarshallers.add(new TooManyEntriesInBatchRequestExceptionUnmarshaller());
        unmarshallers.add(new BatchRequestTooLongExceptionUnmarshaller());
        unmarshallers.add(new UnsupportedOperationExceptionUnmarshaller());
        unmarshallers.add(new InvalidIdFormatExceptionUnmarshaller());
        unmarshallers.add(new OverLimitExceptionUnmarshaller());
        unmarshallers.add(new QueueDoesNotExistExceptionUnmarshaller());
        unmarshallers.add(new QueueNameExistsExceptionUnmarshaller());
        unmarshallers.add(new MessageNotInflightExceptionUnmarshaller());
        unmarshallers.add(new ReceiptHandleIsInvalidExceptionUnmarshaller());
        unmarshallers.add(new InvalidMessageContentsExceptionUnmarshaller());
        unmarshallers.add(new QueueDeletedRecentlyExceptionUnmarshaller());
        unmarshallers.add(new EmptyBatchRequestExceptionUnmarshaller());
        unmarshallers.add(new InvalidBatchEntryIdExceptionUnmarshaller());
        unmarshallers.add(new StandardErrorUnmarshaller(SQSException.class));
        return unmarshallers;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
