/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Amazon SQS message.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Message implements ToCopyableBuilder<Message.Builder, Message> {
    private final String messageId;

    private final String receiptHandle;

    private final String md5OfBody;

    private final String body;

    private final Map<String, String> attributes;

    private final String md5OfMessageAttributes;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private Message(BuilderImpl builder) {
        this.messageId = builder.messageId;
        this.receiptHandle = builder.receiptHandle;
        this.md5OfBody = builder.md5OfBody;
        this.body = builder.body;
        this.attributes = builder.attributes;
        this.md5OfMessageAttributes = builder.md5OfMessageAttributes;
        this.messageAttributes = builder.messageAttributes;
    }

    /**
     * <p>
     * A unique identifier for the message. A <code>MessageId</code>is considered unique across all AWS accounts for an
     * extended period of time.
     * </p>
     * 
     * @return A unique identifier for the message. A <code>MessageId</code>is considered unique across all AWS accounts
     *         for an extended period of time.
     */
    public String messageId() {
        return messageId;
    }

    /**
     * <p>
     * An identifier associated with the act of receiving the message. A new receipt handle is returned every time you
     * receive a message. When deleting a message, you provide the last received receipt handle to delete the message.
     * </p>
     * 
     * @return An identifier associated with the act of receiving the message. A new receipt handle is returned every
     *         time you receive a message. When deleting a message, you provide the last received receipt handle to
     *         delete the message.
     */
    public String receiptHandle() {
        return receiptHandle;
    }

    /**
     * <p>
     * An MD5 digest of the non-URL-encoded message body string.
     * </p>
     * 
     * @return An MD5 digest of the non-URL-encoded message body string.
     */
    public String md5OfBody() {
        return md5OfBody;
    }

    /**
     * <p>
     * The message's contents (not URL-encoded).
     * </p>
     * 
     * @return The message's contents (not URL-encoded).
     */
    public String body() {
        return body;
    }

    /**
     * <p>
     * <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
     * <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
     * <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
     * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
     *         <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
     *         <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
     *         href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     */
    public Map<MessageSystemAttributeName, String> attributes() {
        return TypeConverter.convert(attributes, MessageSystemAttributeName::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, MessageSystemAttributeName.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * <p>
     * <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
     * <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
     * <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
     * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
     *         <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
     *         <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
     *         href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     */
    public Map<String, String> attributesStrings() {
        return attributes;
    }

    /**
     * <p>
     * An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that Amazon
     * SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5 digest. For
     * information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     * </p>
     * 
     * @return An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that
     *         Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5
     *         digest. For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     */
    public String md5OfMessageAttributes() {
        return md5OfMessageAttributes;
    }

    /**
     * <p>
     * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
     * information, see <a href=
     * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-items-validation"
     * >Message Attribute Items and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
     *         more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-items-validation"
     *         >Message Attribute Items and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((messageId() == null) ? 0 : messageId().hashCode());
        hashCode = 31 * hashCode + ((receiptHandle() == null) ? 0 : receiptHandle().hashCode());
        hashCode = 31 * hashCode + ((md5OfBody() == null) ? 0 : md5OfBody().hashCode());
        hashCode = 31 * hashCode + ((body() == null) ? 0 : body().hashCode());
        hashCode = 31 * hashCode + ((attributesStrings() == null) ? 0 : attributesStrings().hashCode());
        hashCode = 31 * hashCode + ((md5OfMessageAttributes() == null) ? 0 : md5OfMessageAttributes().hashCode());
        hashCode = 31 * hashCode + ((messageAttributes() == null) ? 0 : messageAttributes().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Message)) {
            return false;
        }
        Message other = (Message) obj;
        if (other.messageId() == null ^ this.messageId() == null) {
            return false;
        }
        if (other.messageId() != null && !other.messageId().equals(this.messageId())) {
            return false;
        }
        if (other.receiptHandle() == null ^ this.receiptHandle() == null) {
            return false;
        }
        if (other.receiptHandle() != null && !other.receiptHandle().equals(this.receiptHandle())) {
            return false;
        }
        if (other.md5OfBody() == null ^ this.md5OfBody() == null) {
            return false;
        }
        if (other.md5OfBody() != null && !other.md5OfBody().equals(this.md5OfBody())) {
            return false;
        }
        if (other.body() == null ^ this.body() == null) {
            return false;
        }
        if (other.body() != null && !other.body().equals(this.body())) {
            return false;
        }
        if (other.attributesStrings() == null ^ this.attributesStrings() == null) {
            return false;
        }
        if (other.attributesStrings() != null && !other.attributesStrings().equals(this.attributesStrings())) {
            return false;
        }
        if (other.md5OfMessageAttributes() == null ^ this.md5OfMessageAttributes() == null) {
            return false;
        }
        if (other.md5OfMessageAttributes() != null && !other.md5OfMessageAttributes().equals(this.md5OfMessageAttributes())) {
            return false;
        }
        if (other.messageAttributes() == null ^ this.messageAttributes() == null) {
            return false;
        }
        if (other.messageAttributes() != null && !other.messageAttributes().equals(this.messageAttributes())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (messageId() != null) {
            sb.append("MessageId: ").append(messageId()).append(",");
        }
        if (receiptHandle() != null) {
            sb.append("ReceiptHandle: ").append(receiptHandle()).append(",");
        }
        if (md5OfBody() != null) {
            sb.append("MD5OfBody: ").append(md5OfBody()).append(",");
        }
        if (body() != null) {
            sb.append("Body: ").append(body()).append(",");
        }
        if (attributesStrings() != null) {
            sb.append("Attributes: ").append(attributesStrings()).append(",");
        }
        if (md5OfMessageAttributes() != null) {
            sb.append("MD5OfMessageAttributes: ").append(md5OfMessageAttributes()).append(",");
        }
        if (messageAttributes() != null) {
            sb.append("MessageAttributes: ").append(messageAttributes()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MessageId":
            return Optional.of(clazz.cast(messageId()));
        case "ReceiptHandle":
            return Optional.of(clazz.cast(receiptHandle()));
        case "MD5OfBody":
            return Optional.of(clazz.cast(md5OfBody()));
        case "Body":
            return Optional.of(clazz.cast(body()));
        case "Attributes":
            return Optional.of(clazz.cast(attributesStrings()));
        case "MD5OfMessageAttributes":
            return Optional.of(clazz.cast(md5OfMessageAttributes()));
        case "MessageAttributes":
            return Optional.of(clazz.cast(messageAttributes()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, Message> {
        /**
         * <p>
         * A unique identifier for the message. A <code>MessageId</code>is considered unique across all AWS accounts for
         * an extended period of time.
         * </p>
         * 
         * @param messageId
         *        A unique identifier for the message. A <code>MessageId</code>is considered unique across all AWS
         *        accounts for an extended period of time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageId(String messageId);

        /**
         * <p>
         * An identifier associated with the act of receiving the message. A new receipt handle is returned every time
         * you receive a message. When deleting a message, you provide the last received receipt handle to delete the
         * message.
         * </p>
         * 
         * @param receiptHandle
         *        An identifier associated with the act of receiving the message. A new receipt handle is returned every
         *        time you receive a message. When deleting a message, you provide the last received receipt handle to
         *        delete the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receiptHandle(String receiptHandle);

        /**
         * <p>
         * An MD5 digest of the non-URL-encoded message body string.
         * </p>
         * 
         * @param md5OfBody
         *        An MD5 digest of the non-URL-encoded message body string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder md5OfBody(String md5OfBody);

        /**
         * <p>
         * The message's contents (not URL-encoded).
         * </p>
         * 
         * @param body
         *        The message's contents (not URL-encoded).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
         * <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
         * <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
         * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
         * </p>
         * 
         * @param attributes
         *        <code>SenderId</code>, <code>SentTimestamp</code>, <code>ApproximateReceiveCount</code>, and/or
         *        <code>ApproximateFirstReceiveTimestamp</code>. <code>SentTimestamp</code> and
         *        <code>ApproximateFirstReceiveTimestamp</code> are each returned as an integer representing the <a
         *        href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that
         * Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5 digest.
         * For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
         * </p>
         * 
         * @param md5OfMessageAttributes
         *        An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify
         *        that Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the
         *        MD5 digest. For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder md5OfMessageAttributes(String md5OfMessageAttributes);

        /**
         * <p>
         * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
         * information, see <a href=
         * "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-items-validation"
         * >Message Attribute Items and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param messageAttributes
         *        Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
         *        more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html#message-attributes-items-validation"
         *        >Message Attribute Items and Validation</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String messageId;

        private String receiptHandle;

        private String md5OfBody;

        private String body;

        private Map<String, String> attributes;

        private String md5OfMessageAttributes;

        private Map<String, MessageAttributeValue> messageAttributes;

        private BuilderImpl() {
        }

        private BuilderImpl(Message model) {
            messageId(model.messageId);
            receiptHandle(model.receiptHandle);
            md5OfBody(model.md5OfBody);
            body(model.body);
            attributes(model.attributes);
            md5OfMessageAttributes(model.md5OfMessageAttributes);
            messageAttributes(model.messageAttributes);
        }

        public final String getMessageId() {
            return messageId;
        }

        @Override
        public final Builder messageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        public final void setMessageId(String messageId) {
            this.messageId = messageId;
        }

        public final String getReceiptHandle() {
            return receiptHandle;
        }

        @Override
        public final Builder receiptHandle(String receiptHandle) {
            this.receiptHandle = receiptHandle;
            return this;
        }

        public final void setReceiptHandle(String receiptHandle) {
            this.receiptHandle = receiptHandle;
        }

        public final String getMD5OfBody() {
            return md5OfBody;
        }

        @Override
        public final Builder md5OfBody(String md5OfBody) {
            this.md5OfBody = md5OfBody;
            return this;
        }

        public final void setMD5OfBody(String md5OfBody) {
            this.md5OfBody = md5OfBody;
        }

        public final String getBody() {
            return body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        public final Map<String, String> getAttributes() {
            return attributes;
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = MessageSystemAttributeMapCopier.copy(attributes);
            return this;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = MessageSystemAttributeMapCopier.copy(attributes);
        }

        public final String getMD5OfMessageAttributes() {
            return md5OfMessageAttributes;
        }

        @Override
        public final Builder md5OfMessageAttributes(String md5OfMessageAttributes) {
            this.md5OfMessageAttributes = md5OfMessageAttributes;
            return this;
        }

        public final void setMD5OfMessageAttributes(String md5OfMessageAttributes) {
            this.md5OfMessageAttributes = md5OfMessageAttributes;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            return messageAttributes != null ? CollectionUtils.mapValues(messageAttributes, MessageAttributeValue::toBuilder)
                    : null;
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        @Override
        public Message build() {
            return new Message(this);
        }
    }
}
