/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReceiveMessageRequest extends SqsRequest implements
        ToCopyableBuilder<ReceiveMessageRequest.Builder, ReceiveMessageRequest> {
    private static final SdkField<String> QUEUE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReceiveMessageRequest::queueUrl)).setter(setter(Builder::queueUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueueUrl").build()).build();

    private static final SdkField<List<String>> ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ReceiveMessageRequest::attributeNamesAsStrings))
            .setter(setter(Builder::attributeNamesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeName").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AttributeName")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AttributeName").build()).build()).isFlattened(true).build())
            .build();

    private static final SdkField<List<String>> MESSAGE_ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ReceiveMessageRequest::messageAttributeNames))
            .setter(setter(Builder::messageAttributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttributeName").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("MessageAttributeName")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("MessageAttributeName").build()).build()).isFlattened(true)
                            .build()).build();

    private static final SdkField<Integer> MAX_NUMBER_OF_MESSAGES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReceiveMessageRequest::maxNumberOfMessages)).setter(setter(Builder::maxNumberOfMessages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxNumberOfMessages").build())
            .build();

    private static final SdkField<Integer> VISIBILITY_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReceiveMessageRequest::visibilityTimeout)).setter(setter(Builder::visibilityTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityTimeout").build()).build();

    private static final SdkField<Integer> WAIT_TIME_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReceiveMessageRequest::waitTimeSeconds)).setter(setter(Builder::waitTimeSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WaitTimeSeconds").build()).build();

    private static final SdkField<String> RECEIVE_REQUEST_ATTEMPT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReceiveMessageRequest::receiveRequestAttemptId)).setter(setter(Builder::receiveRequestAttemptId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReceiveRequestAttemptId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUEUE_URL_FIELD,
            ATTRIBUTE_NAMES_FIELD, MESSAGE_ATTRIBUTE_NAMES_FIELD, MAX_NUMBER_OF_MESSAGES_FIELD, VISIBILITY_TIMEOUT_FIELD,
            WAIT_TIME_SECONDS_FIELD, RECEIVE_REQUEST_ATTEMPT_ID_FIELD));

    private final String queueUrl;

    private final List<String> attributeNames;

    private final List<String> messageAttributeNames;

    private final Integer maxNumberOfMessages;

    private final Integer visibilityTimeout;

    private final Integer waitTimeSeconds;

    private final String receiveRequestAttemptId;

    private ReceiveMessageRequest(BuilderImpl builder) {
        super(builder);
        this.queueUrl = builder.queueUrl;
        this.attributeNames = builder.attributeNames;
        this.messageAttributeNames = builder.messageAttributeNames;
        this.maxNumberOfMessages = builder.maxNumberOfMessages;
        this.visibilityTimeout = builder.visibilityTimeout;
        this.waitTimeSeconds = builder.waitTimeSeconds;
        this.receiveRequestAttemptId = builder.receiveRequestAttemptId;
    }

    /**
     * <p>
     * The URL of the Amazon SQS queue from which messages are received.
     * </p>
     * <p>
     * Queue URLs and names are case-sensitive.
     * </p>
     * 
     * @return The URL of the Amazon SQS queue from which messages are received.</p>
     *         <p>
     *         Queue URLs and names are case-sensitive.
     */
    public String queueUrl() {
        return queueUrl;
    }

    /**
     * <p>
     * A list of attributes that need to be returned along with each message. These attributes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>All</code> - Returns all values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the queue
     * (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue but
     * not deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SenderId</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
     * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
     * <code> <a>SendMessage</a> </code> action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
     * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
     * sequence.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of attributes that need to be returned along with each message. These attributes include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>All</code> - Returns all values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
     *         queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the
     *         queue but not deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SenderId</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
     *         href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
     *         <code> <a>SendMessage</a> </code> action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
     *         <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned
     *         in sequence.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
     *         </p>
     *         </li>
     */
    public List<QueueAttributeName> attributeNames() {
        return AttributeNameListCopier.copyStringToEnum(attributeNames);
    }

    /**
     * Returns true if the AttributeNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttributeNames() {
        return attributeNames != null && !(attributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of attributes that need to be returned along with each message. These attributes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>All</code> - Returns all values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the queue
     * (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue but
     * not deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SenderId</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
     * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
     * <code> <a>SendMessage</a> </code> action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
     * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
     * sequence.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of attributes that need to be returned along with each message. These attributes include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>All</code> - Returns all values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
     *         queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the
     *         queue but not deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SenderId</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
     *         href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
     *         <code> <a>SendMessage</a> </code> action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
     *         <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned
     *         in sequence.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
     *         </p>
     *         </li>
     */
    public List<String> attributeNamesAsStrings() {
        return attributeNames;
    }

    /**
     * Returns true if the MessageAttributeNames property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMessageAttributeNames() {
        return messageAttributeNames != null && !(messageAttributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the message attribute, where <i>N</i> is the index.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (<code>-</code>), and
     * period (<code>.</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The name is case-sensitive and must be unique among all attribute names for the message.
     * </p>
     * </li>
     * <li>
     * <p>
     * The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code> (or any
     * casing variants).
     * </p>
     * </li>
     * <li>
     * <p>
     * The name must not start or end with a period (<code>.</code>), and it should not have periods in succession (
     * <code>..</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The name can be up to 256 characters long.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can return all
     * of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You can also use all message
     * attributes starting with a prefix, for example <code>bar.*</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessageAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The name of the message attribute, where <i>N</i> is the index.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (<code>-</code>
     *         ), and period (<code>.</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name is case-sensitive and must be unique among all attribute names for the message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code> (or
     *         any casing variants).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name must not start or end with a period (<code>.</code>), and it should not have periods in
     *         succession (<code>..</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The name can be up to 256 characters long.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can
     *         return all of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You can
     *         also use all message attributes starting with a prefix, for example <code>bar.*</code>.
     */
    public List<String> messageAttributeNames() {
        return messageAttributeNames;
    }

    /**
     * <p>
     * The maximum number of messages to return. Amazon SQS never returns more messages than this value (however, fewer
     * messages might be returned). Valid values: 1 to 10. Default: 1.
     * </p>
     * 
     * @return The maximum number of messages to return. Amazon SQS never returns more messages than this value
     *         (however, fewer messages might be returned). Valid values: 1 to 10. Default: 1.
     */
    public Integer maxNumberOfMessages() {
        return maxNumberOfMessages;
    }

    /**
     * <p>
     * The duration (in seconds) that the received messages are hidden from subsequent retrieve requests after being
     * retrieved by a <code>ReceiveMessage</code> request.
     * </p>
     * 
     * @return The duration (in seconds) that the received messages are hidden from subsequent retrieve requests after
     *         being retrieved by a <code>ReceiveMessage</code> request.
     */
    public Integer visibilityTimeout() {
        return visibilityTimeout;
    }

    /**
     * <p>
     * The duration (in seconds) for which the call waits for a message to arrive in the queue before returning. If a
     * message is available, the call returns sooner than <code>WaitTimeSeconds</code>. If no messages are available and
     * the wait time expires, the call returns successfully with an empty list of messages.
     * </p>
     * 
     * @return The duration (in seconds) for which the call waits for a message to arrive in the queue before returning.
     *         If a message is available, the call returns sooner than <code>WaitTimeSeconds</code>. If no messages are
     *         available and the wait time expires, the call returns successfully with an empty list of messages.
     */
    public Integer waitTimeSeconds() {
        return waitTimeSeconds;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) queues.
     * </p>
     * <p>
     * The token used for deduplication of <code>ReceiveMessage</code> calls. If a networking issue occurs after a
     * <code>ReceiveMessage</code> action, and instead of a response you receive a generic error, you can retry the same
     * action with an identical <code>ReceiveRequestAttemptId</code> to retrieve the same set of messages, even if their
     * visibility timeout has not yet expired.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You can use <code>ReceiveRequestAttemptId</code> only for 5 minutes after a <code>ReceiveMessage</code> action.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you set <code>FifoQueue</code>, a caller of the <code>ReceiveMessage</code> action can provide a
     * <code>ReceiveRequestAttemptId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a caller of the <code>ReceiveMessage</code> action doesn't provide a <code>ReceiveRequestAttemptId</code>,
     * Amazon SQS generates a <code>ReceiveRequestAttemptId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can retry the <code>ReceiveMessage</code> action with the same <code>ReceiveRequestAttemptId</code> if none
     * of the messages have been modified (deleted or had their visibility changes).
     * </p>
     * </li>
     * <li>
     * <p>
     * During a visibility timeout, subsequent calls with the same <code>ReceiveRequestAttemptId</code> return the same
     * messages and receipt handles. If a retry occurs within the deduplication interval, it resets the visibility
     * timeout. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     * >Visibility Timeout</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * If a caller of the <code>ReceiveMessage</code> action still processes messages when the visibility timeout
     * expires and messages become visible, another worker consuming from the same queue can receive the same messages
     * and therefore process duplicates. Also, if a consumer whose message processing time is longer than the visibility
     * timeout tries to delete the processed messages, the action fails with an error.
     * </p>
     * <p>
     * To mitigate this effect, ensure that your application observes a safe threshold before the visibility timeout
     * expires and extend the visibility timeout as necessary.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * While messages with a particular <code>MessageGroupId</code> are invisible, no more messages belonging to the
     * same <code>MessageGroupId</code> are returned until the visibility timeout expires. You can still receive
     * messages with another <code>MessageGroupId</code> as long as it is also visible.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a caller of <code>ReceiveMessage</code> can't track the <code>ReceiveRequestAttemptId</code>, no retries work
     * until the original visibility timeout expires. As a result, delays might occur but the messages in the queue
     * remain in a strict order.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The length of <code>ReceiveRequestAttemptId</code> is 128 characters. <code>ReceiveRequestAttemptId</code> can
     * contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and punctuation (
     * <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     * </p>
     * <p>
     * For best practices of using <code>ReceiveRequestAttemptId</code>, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-receiverequestattemptid-request-parameter.html"
     * >Using the ReceiveRequestAttemptId Request Parameter</a> in the <i>Amazon Simple Queue Service Developer
     * Guide</i>.
     * </p>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) queues.</p>
     *         <p>
     *         The token used for deduplication of <code>ReceiveMessage</code> calls. If a networking issue occurs after
     *         a <code>ReceiveMessage</code> action, and instead of a response you receive a generic error, you can
     *         retry the same action with an identical <code>ReceiveRequestAttemptId</code> to retrieve the same set of
     *         messages, even if their visibility timeout has not yet expired.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You can use <code>ReceiveRequestAttemptId</code> only for 5 minutes after a <code>ReceiveMessage</code>
     *         action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When you set <code>FifoQueue</code>, a caller of the <code>ReceiveMessage</code> action can provide a
     *         <code>ReceiveRequestAttemptId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a caller of the <code>ReceiveMessage</code> action doesn't provide a
     *         <code>ReceiveRequestAttemptId</code>, Amazon SQS generates a <code>ReceiveRequestAttemptId</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can retry the <code>ReceiveMessage</code> action with the same <code>ReceiveRequestAttemptId</code>
     *         if none of the messages have been modified (deleted or had their visibility changes).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         During a visibility timeout, subsequent calls with the same <code>ReceiveRequestAttemptId</code> return
     *         the same messages and receipt handles. If a retry occurs within the deduplication interval, it resets the
     *         visibility timeout. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
     *         >Visibility Timeout</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         If a caller of the <code>ReceiveMessage</code> action still processes messages when the visibility
     *         timeout expires and messages become visible, another worker consuming from the same queue can receive the
     *         same messages and therefore process duplicates. Also, if a consumer whose message processing time is
     *         longer than the visibility timeout tries to delete the processed messages, the action fails with an
     *         error.
     *         </p>
     *         <p>
     *         To mitigate this effect, ensure that your application observes a safe threshold before the visibility
     *         timeout expires and extend the visibility timeout as necessary.
     *         </p>
     *         </important></li>
     *         <li>
     *         <p>
     *         While messages with a particular <code>MessageGroupId</code> are invisible, no more messages belonging to
     *         the same <code>MessageGroupId</code> are returned until the visibility timeout expires. You can still
     *         receive messages with another <code>MessageGroupId</code> as long as it is also visible.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a caller of <code>ReceiveMessage</code> can't track the <code>ReceiveRequestAttemptId</code>, no
     *         retries work until the original visibility timeout expires. As a result, delays might occur but the
     *         messages in the queue remain in a strict order.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The length of <code>ReceiveRequestAttemptId</code> is 128 characters.
     *         <code>ReceiveRequestAttemptId</code> can contain alphanumeric characters (<code>a-z</code>,
     *         <code>A-Z</code>, <code>0-9</code>) and punctuation (
     *         <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     *         </p>
     *         <p>
     *         For best practices of using <code>ReceiveRequestAttemptId</code>, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-receiverequestattemptid-request-parameter.html"
     *         >Using the ReceiveRequestAttemptId Request Parameter</a> in the <i>Amazon Simple Queue Service Developer
     *         Guide</i>.
     */
    public String receiveRequestAttemptId() {
        return receiveRequestAttemptId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(queueUrl());
        hashCode = 31 * hashCode + Objects.hashCode(attributeNamesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(messageAttributeNames());
        hashCode = 31 * hashCode + Objects.hashCode(maxNumberOfMessages());
        hashCode = 31 * hashCode + Objects.hashCode(visibilityTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(waitTimeSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(receiveRequestAttemptId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReceiveMessageRequest)) {
            return false;
        }
        ReceiveMessageRequest other = (ReceiveMessageRequest) obj;
        return Objects.equals(queueUrl(), other.queueUrl())
                && Objects.equals(attributeNamesAsStrings(), other.attributeNamesAsStrings())
                && Objects.equals(messageAttributeNames(), other.messageAttributeNames())
                && Objects.equals(maxNumberOfMessages(), other.maxNumberOfMessages())
                && Objects.equals(visibilityTimeout(), other.visibilityTimeout())
                && Objects.equals(waitTimeSeconds(), other.waitTimeSeconds())
                && Objects.equals(receiveRequestAttemptId(), other.receiveRequestAttemptId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReceiveMessageRequest").add("QueueUrl", queueUrl())
                .add("AttributeNames", attributeNamesAsStrings()).add("MessageAttributeNames", messageAttributeNames())
                .add("MaxNumberOfMessages", maxNumberOfMessages()).add("VisibilityTimeout", visibilityTimeout())
                .add("WaitTimeSeconds", waitTimeSeconds()).add("ReceiveRequestAttemptId", receiveRequestAttemptId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueueUrl":
            return Optional.ofNullable(clazz.cast(queueUrl()));
        case "AttributeNames":
            return Optional.ofNullable(clazz.cast(attributeNamesAsStrings()));
        case "MessageAttributeNames":
            return Optional.ofNullable(clazz.cast(messageAttributeNames()));
        case "MaxNumberOfMessages":
            return Optional.ofNullable(clazz.cast(maxNumberOfMessages()));
        case "VisibilityTimeout":
            return Optional.ofNullable(clazz.cast(visibilityTimeout()));
        case "WaitTimeSeconds":
            return Optional.ofNullable(clazz.cast(waitTimeSeconds()));
        case "ReceiveRequestAttemptId":
            return Optional.ofNullable(clazz.cast(receiveRequestAttemptId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReceiveMessageRequest, T> g) {
        return obj -> g.apply((ReceiveMessageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SqsRequest.Builder, SdkPojo, CopyableBuilder<Builder, ReceiveMessageRequest> {
        /**
         * <p>
         * The URL of the Amazon SQS queue from which messages are received.
         * </p>
         * <p>
         * Queue URLs and names are case-sensitive.
         * </p>
         * 
         * @param queueUrl
         *        The URL of the Amazon SQS queue from which messages are received.</p>
         *        <p>
         *        Queue URLs and names are case-sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queueUrl(String queueUrl);

        /**
         * <p>
         * A list of attributes that need to be returned along with each message. These attributes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>All</code> - Returns all values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
         * queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue
         * but not deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
         * sequence.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeNames
         *        A list of attributes that need to be returned along with each message. These attributes include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>All</code> - Returns all values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from
         *        the queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from
         *        the queue but not deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         *        href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are
         *        returned in sequence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNamesWithStrings(Collection<String> attributeNames);

        /**
         * <p>
         * A list of attributes that need to be returned along with each message. These attributes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>All</code> - Returns all values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
         * queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue
         * but not deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
         * sequence.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeNames
         *        A list of attributes that need to be returned along with each message. These attributes include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>All</code> - Returns all values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from
         *        the queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from
         *        the queue but not deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         *        href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are
         *        returned in sequence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNamesWithStrings(String... attributeNames);

        /**
         * <p>
         * A list of attributes that need to be returned along with each message. These attributes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>All</code> - Returns all values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
         * queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue
         * but not deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
         * sequence.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeNames
         *        A list of attributes that need to be returned along with each message. These attributes include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>All</code> - Returns all values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from
         *        the queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from
         *        the queue but not deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         *        href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are
         *        returned in sequence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(Collection<QueueAttributeName> attributeNames);

        /**
         * <p>
         * A list of attributes that need to be returned along with each message. These attributes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>All</code> - Returns all values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from the
         * queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from the queue
         * but not deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         * href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         * <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are returned in
         * sequence.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeNames
         *        A list of attributes that need to be returned along with each message. These attributes include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>All</code> - Returns all values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> - Returns the time the message was first received from
         *        the queue (<a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code> - Returns the number of times a message has been received from
         *        the queue but not deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSTraceHeader</code> - Returns the AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an IAM user, returns the IAM user ID, for example <code>ABCDEFGHI1JKLMNOPQ23R</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For an IAM role, returns the IAM role ID, for example <code>ABCDE1F2GH3I4JK5LMNOP:i-a123b456</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code> - Returns the time the message was sent to the queue (<a
         *        href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code> - Returns the value provided by the producer that calls the
         *        <code> <a>SendMessage</a> </code> action. Messages with the same <code>MessageGroupId</code> are
         *        returned in sequence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code> - Returns the value provided by Amazon SQS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(QueueAttributeName... attributeNames);

        /**
         * <p>
         * The name of the message attribute, where <i>N</i> is the index.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (<code>-</code>),
         * and period (<code>.</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name is case-sensitive and must be unique among all attribute names for the message.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code> (or any
         * casing variants).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name must not start or end with a period (<code>.</code>), and it should not have periods in succession (
         * <code>..</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name can be up to 256 characters long.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can return
         * all of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You can also use all
         * message attributes starting with a prefix, for example <code>bar.*</code>.
         * </p>
         * 
         * @param messageAttributeNames
         *        The name of the message attribute, where <i>N</i> is the index.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (
         *        <code>-</code>), and period (<code>.</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name is case-sensitive and must be unique among all attribute names for the message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code>
         *        (or any casing variants).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name must not start or end with a period (<code>.</code>), and it should not have periods in
         *        succession (<code>..</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name can be up to 256 characters long.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can
         *        return all of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You
         *        can also use all message attributes starting with a prefix, for example <code>bar.*</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributeNames(Collection<String> messageAttributeNames);

        /**
         * <p>
         * The name of the message attribute, where <i>N</i> is the index.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (<code>-</code>),
         * and period (<code>.</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name is case-sensitive and must be unique among all attribute names for the message.
         * </p>
         * </li>
         * <li>
         * <p>
         * The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code> (or any
         * casing variants).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name must not start or end with a period (<code>.</code>), and it should not have periods in succession (
         * <code>..</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The name can be up to 256 characters long.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can return
         * all of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You can also use all
         * message attributes starting with a prefix, for example <code>bar.*</code>.
         * </p>
         * 
         * @param messageAttributeNames
         *        The name of the message attribute, where <i>N</i> is the index.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The name can contain alphanumeric characters and the underscore (<code>_</code>), hyphen (
         *        <code>-</code>), and period (<code>.</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name is case-sensitive and must be unique among all attribute names for the message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name must not start with AWS-reserved prefixes such as <code>AWS.</code> or <code>Amazon.</code>
         *        (or any casing variants).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name must not start or end with a period (<code>.</code>), and it should not have periods in
         *        succession (<code>..</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The name can be up to 256 characters long.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When using <code>ReceiveMessage</code>, you can send a list of attribute names to receive, or you can
         *        return all of the attributes by specifying <code>All</code> or <code>.*</code> in your request. You
         *        can also use all message attributes starting with a prefix, for example <code>bar.*</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributeNames(String... messageAttributeNames);

        /**
         * <p>
         * The maximum number of messages to return. Amazon SQS never returns more messages than this value (however,
         * fewer messages might be returned). Valid values: 1 to 10. Default: 1.
         * </p>
         * 
         * @param maxNumberOfMessages
         *        The maximum number of messages to return. Amazon SQS never returns more messages than this value
         *        (however, fewer messages might be returned). Valid values: 1 to 10. Default: 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxNumberOfMessages(Integer maxNumberOfMessages);

        /**
         * <p>
         * The duration (in seconds) that the received messages are hidden from subsequent retrieve requests after being
         * retrieved by a <code>ReceiveMessage</code> request.
         * </p>
         * 
         * @param visibilityTimeout
         *        The duration (in seconds) that the received messages are hidden from subsequent retrieve requests
         *        after being retrieved by a <code>ReceiveMessage</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityTimeout(Integer visibilityTimeout);

        /**
         * <p>
         * The duration (in seconds) for which the call waits for a message to arrive in the queue before returning. If
         * a message is available, the call returns sooner than <code>WaitTimeSeconds</code>. If no messages are
         * available and the wait time expires, the call returns successfully with an empty list of messages.
         * </p>
         * 
         * @param waitTimeSeconds
         *        The duration (in seconds) for which the call waits for a message to arrive in the queue before
         *        returning. If a message is available, the call returns sooner than <code>WaitTimeSeconds</code>. If no
         *        messages are available and the wait time expires, the call returns successfully with an empty list of
         *        messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder waitTimeSeconds(Integer waitTimeSeconds);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) queues.
         * </p>
         * <p>
         * The token used for deduplication of <code>ReceiveMessage</code> calls. If a networking issue occurs after a
         * <code>ReceiveMessage</code> action, and instead of a response you receive a generic error, you can retry the
         * same action with an identical <code>ReceiveRequestAttemptId</code> to retrieve the same set of messages, even
         * if their visibility timeout has not yet expired.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You can use <code>ReceiveRequestAttemptId</code> only for 5 minutes after a <code>ReceiveMessage</code>
         * action.
         * </p>
         * </li>
         * <li>
         * <p>
         * When you set <code>FifoQueue</code>, a caller of the <code>ReceiveMessage</code> action can provide a
         * <code>ReceiveRequestAttemptId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a caller of the <code>ReceiveMessage</code> action doesn't provide a <code>ReceiveRequestAttemptId</code>,
         * Amazon SQS generates a <code>ReceiveRequestAttemptId</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can retry the <code>ReceiveMessage</code> action with the same <code>ReceiveRequestAttemptId</code> if
         * none of the messages have been modified (deleted or had their visibility changes).
         * </p>
         * </li>
         * <li>
         * <p>
         * During a visibility timeout, subsequent calls with the same <code>ReceiveRequestAttemptId</code> return the
         * same messages and receipt handles. If a retry occurs within the deduplication interval, it resets the
         * visibility timeout. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         * >Visibility Timeout</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * If a caller of the <code>ReceiveMessage</code> action still processes messages when the visibility timeout
         * expires and messages become visible, another worker consuming from the same queue can receive the same
         * messages and therefore process duplicates. Also, if a consumer whose message processing time is longer than
         * the visibility timeout tries to delete the processed messages, the action fails with an error.
         * </p>
         * <p>
         * To mitigate this effect, ensure that your application observes a safe threshold before the visibility timeout
         * expires and extend the visibility timeout as necessary.
         * </p>
         * </important></li>
         * <li>
         * <p>
         * While messages with a particular <code>MessageGroupId</code> are invisible, no more messages belonging to the
         * same <code>MessageGroupId</code> are returned until the visibility timeout expires. You can still receive
         * messages with another <code>MessageGroupId</code> as long as it is also visible.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a caller of <code>ReceiveMessage</code> can't track the <code>ReceiveRequestAttemptId</code>, no retries
         * work until the original visibility timeout expires. As a result, delays might occur but the messages in the
         * queue remain in a strict order.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The length of <code>ReceiveRequestAttemptId</code> is 128 characters. <code>ReceiveRequestAttemptId</code>
         * can contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and punctuation (
         * <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         * </p>
         * <p>
         * For best practices of using <code>ReceiveRequestAttemptId</code>, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-receiverequestattemptid-request-parameter.html"
         * >Using the ReceiveRequestAttemptId Request Parameter</a> in the <i>Amazon Simple Queue Service Developer
         * Guide</i>.
         * </p>
         * 
         * @param receiveRequestAttemptId
         *        This parameter applies only to FIFO (first-in-first-out) queues.</p>
         *        <p>
         *        The token used for deduplication of <code>ReceiveMessage</code> calls. If a networking issue occurs
         *        after a <code>ReceiveMessage</code> action, and instead of a response you receive a generic error, you
         *        can retry the same action with an identical <code>ReceiveRequestAttemptId</code> to retrieve the same
         *        set of messages, even if their visibility timeout has not yet expired.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use <code>ReceiveRequestAttemptId</code> only for 5 minutes after a
         *        <code>ReceiveMessage</code> action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When you set <code>FifoQueue</code>, a caller of the <code>ReceiveMessage</code> action can provide a
         *        <code>ReceiveRequestAttemptId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a caller of the <code>ReceiveMessage</code> action doesn't provide a
         *        <code>ReceiveRequestAttemptId</code>, Amazon SQS generates a <code>ReceiveRequestAttemptId</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can retry the <code>ReceiveMessage</code> action with the same
         *        <code>ReceiveRequestAttemptId</code> if none of the messages have been modified (deleted or had their
         *        visibility changes).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        During a visibility timeout, subsequent calls with the same <code>ReceiveRequestAttemptId</code>
         *        return the same messages and receipt handles. If a retry occurs within the deduplication interval, it
         *        resets the visibility timeout. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html"
         *        >Visibility Timeout</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        If a caller of the <code>ReceiveMessage</code> action still processes messages when the visibility
         *        timeout expires and messages become visible, another worker consuming from the same queue can receive
         *        the same messages and therefore process duplicates. Also, if a consumer whose message processing time
         *        is longer than the visibility timeout tries to delete the processed messages, the action fails with an
         *        error.
         *        </p>
         *        <p>
         *        To mitigate this effect, ensure that your application observes a safe threshold before the visibility
         *        timeout expires and extend the visibility timeout as necessary.
         *        </p>
         *        </important></li>
         *        <li>
         *        <p>
         *        While messages with a particular <code>MessageGroupId</code> are invisible, no more messages belonging
         *        to the same <code>MessageGroupId</code> are returned until the visibility timeout expires. You can
         *        still receive messages with another <code>MessageGroupId</code> as long as it is also visible.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a caller of <code>ReceiveMessage</code> can't track the <code>ReceiveRequestAttemptId</code>, no
         *        retries work until the original visibility timeout expires. As a result, delays might occur but the
         *        messages in the queue remain in a strict order.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The length of <code>ReceiveRequestAttemptId</code> is 128 characters.
         *        <code>ReceiveRequestAttemptId</code> can contain alphanumeric characters (<code>a-z</code>,
         *        <code>A-Z</code>, <code>0-9</code>) and punctuation (
         *        <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         *        </p>
         *        <p>
         *        For best practices of using <code>ReceiveRequestAttemptId</code>, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-receiverequestattemptid-request-parameter.html"
         *        >Using the ReceiveRequestAttemptId Request Parameter</a> in the <i>Amazon Simple Queue Service
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receiveRequestAttemptId(String receiveRequestAttemptId);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SqsRequest.BuilderImpl implements Builder {
        private String queueUrl;

        private List<String> attributeNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> messageAttributeNames = DefaultSdkAutoConstructList.getInstance();

        private Integer maxNumberOfMessages;

        private Integer visibilityTimeout;

        private Integer waitTimeSeconds;

        private String receiveRequestAttemptId;

        private BuilderImpl() {
        }

        private BuilderImpl(ReceiveMessageRequest model) {
            super(model);
            queueUrl(model.queueUrl);
            attributeNamesWithStrings(model.attributeNames);
            messageAttributeNames(model.messageAttributeNames);
            maxNumberOfMessages(model.maxNumberOfMessages);
            visibilityTimeout(model.visibilityTimeout);
            waitTimeSeconds(model.waitTimeSeconds);
            receiveRequestAttemptId(model.receiveRequestAttemptId);
        }

        public final String getQueueUrl() {
            return queueUrl;
        }

        @Override
        public final Builder queueUrl(String queueUrl) {
            this.queueUrl = queueUrl;
            return this;
        }

        public final void setQueueUrl(String queueUrl) {
            this.queueUrl = queueUrl;
        }

        public final Collection<String> getAttributeNamesAsStrings() {
            return attributeNames;
        }

        @Override
        public final Builder attributeNamesWithStrings(Collection<String> attributeNames) {
            this.attributeNames = AttributeNameListCopier.copy(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNamesWithStrings(String... attributeNames) {
            attributeNamesWithStrings(Arrays.asList(attributeNames));
            return this;
        }

        @Override
        public final Builder attributeNames(Collection<QueueAttributeName> attributeNames) {
            this.attributeNames = AttributeNameListCopier.copyEnumToString(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNames(QueueAttributeName... attributeNames) {
            attributeNames(Arrays.asList(attributeNames));
            return this;
        }

        public final void setAttributeNamesWithStrings(Collection<String> attributeNames) {
            this.attributeNames = AttributeNameListCopier.copy(attributeNames);
        }

        public final Collection<String> getMessageAttributeNames() {
            return messageAttributeNames;
        }

        @Override
        public final Builder messageAttributeNames(Collection<String> messageAttributeNames) {
            this.messageAttributeNames = MessageAttributeNameListCopier.copy(messageAttributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messageAttributeNames(String... messageAttributeNames) {
            messageAttributeNames(Arrays.asList(messageAttributeNames));
            return this;
        }

        public final void setMessageAttributeNames(Collection<String> messageAttributeNames) {
            this.messageAttributeNames = MessageAttributeNameListCopier.copy(messageAttributeNames);
        }

        public final Integer getMaxNumberOfMessages() {
            return maxNumberOfMessages;
        }

        @Override
        public final Builder maxNumberOfMessages(Integer maxNumberOfMessages) {
            this.maxNumberOfMessages = maxNumberOfMessages;
            return this;
        }

        public final void setMaxNumberOfMessages(Integer maxNumberOfMessages) {
            this.maxNumberOfMessages = maxNumberOfMessages;
        }

        public final Integer getVisibilityTimeout() {
            return visibilityTimeout;
        }

        @Override
        public final Builder visibilityTimeout(Integer visibilityTimeout) {
            this.visibilityTimeout = visibilityTimeout;
            return this;
        }

        public final void setVisibilityTimeout(Integer visibilityTimeout) {
            this.visibilityTimeout = visibilityTimeout;
        }

        public final Integer getWaitTimeSeconds() {
            return waitTimeSeconds;
        }

        @Override
        public final Builder waitTimeSeconds(Integer waitTimeSeconds) {
            this.waitTimeSeconds = waitTimeSeconds;
            return this;
        }

        public final void setWaitTimeSeconds(Integer waitTimeSeconds) {
            this.waitTimeSeconds = waitTimeSeconds;
        }

        public final String getReceiveRequestAttemptId() {
            return receiveRequestAttemptId;
        }

        @Override
        public final Builder receiveRequestAttemptId(String receiveRequestAttemptId) {
            this.receiveRequestAttemptId = receiveRequestAttemptId;
            return this;
        }

        public final void setReceiveRequestAttemptId(String receiveRequestAttemptId) {
            this.receiveRequestAttemptId = receiveRequestAttemptId;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ReceiveMessageRequest build() {
            return new ReceiveMessageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
