/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the <code>Value</code> attribute has the same
 * restrictions on the content as the message body. For more information, see <code> <a>SendMessage</a>.</code>
 * </p>
 * <p>
 * <code>Name</code>, <code>type</code>, <code>value</code> and the message body must not be empty or null. All parts of
 * the message attribute, including <code>Name</code>, <code>Type</code>, and <code>Value</code>, are part of the
 * message size restriction (256 KB or 262,144 bytes).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageAttributeValue implements SdkPojo, Serializable,
        ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private static final SdkField<String> STRING_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageAttributeValue::stringValue)).setter(setter(Builder::stringValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StringValue").build()).build();

    private static final SdkField<SdkBytes> BINARY_VALUE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(MessageAttributeValue::binaryValue)).setter(setter(Builder::binaryValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BinaryValue").build()).build();

    private static final SdkField<List<String>> STRING_LIST_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(MessageAttributeValue::stringListValues))
            .setter(setter(Builder::stringListValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StringListValue").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("StringListValue")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("StringListValue").build()).build()).isFlattened(true).build())
            .build();

    private static final SdkField<List<SdkBytes>> BINARY_LIST_VALUES_FIELD = SdkField
            .<List<SdkBytes>> builder(MarshallingType.LIST)
            .getter(getter(MessageAttributeValue::binaryListValues))
            .setter(setter(Builder::binaryListValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BinaryListValue").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("BinaryListValue")
                            .memberFieldInfo(
                                    SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("BinaryListValue").build()).build()).isFlattened(true).build())
            .build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageAttributeValue::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STRING_VALUE_FIELD,
            BINARY_VALUE_FIELD, STRING_LIST_VALUES_FIELD, BINARY_LIST_VALUES_FIELD, DATA_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stringValue;

    private final SdkBytes binaryValue;

    private final List<String> stringListValues;

    private final List<SdkBytes> binaryListValues;

    private final String dataType;

    private MessageAttributeValue(BuilderImpl builder) {
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
        this.stringListValues = builder.stringListValues;
        this.binaryListValues = builder.binaryListValues;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     */
    public String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
     */
    public SdkBytes binaryValue() {
        return binaryValue;
    }

    /**
     * Returns true if the StringListValues property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasStringListValues() {
        return stringListValues != null && !(stringListValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStringListValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<String> stringListValues() {
        return stringListValues;
    }

    /**
     * Returns true if the BinaryListValues property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasBinaryListValues() {
        return binaryListValues != null && !(binaryListValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Not implemented. Reserved for future use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasBinaryListValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Not implemented. Reserved for future use.
     */
    public List<SdkBytes> binaryListValues() {
        return binaryListValues;
    }

    /**
     * <p>
     * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
     * </p>
     * <p>
     * You can also append custom labels. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html"
     * >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * 
     * @return Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
     *         <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
     *         <p>
     *         You can also append custom labels. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html"
     *         >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     */
    public String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stringValue());
        hashCode = 31 * hashCode + Objects.hashCode(binaryValue());
        hashCode = 31 * hashCode + Objects.hashCode(stringListValues());
        hashCode = 31 * hashCode + Objects.hashCode(binaryListValues());
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        return Objects.equals(stringValue(), other.stringValue()) && Objects.equals(binaryValue(), other.binaryValue())
                && Objects.equals(stringListValues(), other.stringListValues())
                && Objects.equals(binaryListValues(), other.binaryListValues()) && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MessageAttributeValue").add("StringValue", stringValue()).add("BinaryValue", binaryValue())
                .add("StringListValues", stringListValues()).add("BinaryListValues", binaryListValues())
                .add("DataType", dataType()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StringValue":
            return Optional.ofNullable(clazz.cast(stringValue()));
        case "BinaryValue":
            return Optional.ofNullable(clazz.cast(binaryValue()));
        case "StringListValues":
            return Optional.ofNullable(clazz.cast(stringListValues()));
        case "BinaryListValues":
            return Optional.ofNullable(clazz.cast(binaryListValues()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageAttributeValue, T> g) {
        return obj -> g.apply((MessageAttributeValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF-8 binary encoding. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * </p>
         * 
         * @param binaryValue
         *        Binary type attributes can store any binary data, such as compressed data, encrypted data, or images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(SdkBytes binaryValue);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(Collection<String> stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param stringListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringListValues(String... stringListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(Collection<SdkBytes> binaryListValues);

        /**
         * <p>
         * Not implemented. Reserved for future use.
         * </p>
         * 
         * @param binaryListValues
         *        Not implemented. Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryListValues(SdkBytes... binaryListValues);

        /**
         * <p>
         * Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         * <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.
         * </p>
         * <p>
         * You can also append custom labels. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html"
         * >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * 
         * @param dataType
         *        Amazon SQS supports the following logical data types: <code>String</code>, <code>Number</code>, and
         *        <code>Binary</code>. For the <code>Number</code> data type, you must use <code>StringValue</code>.</p>
         *        <p>
         *        You can also append custom labels. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-attributes.html"
         *        >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    static final class BuilderImpl implements Builder {
        private String stringValue;

        private SdkBytes binaryValue;

        private List<String> stringListValues = DefaultSdkAutoConstructList.getInstance();

        private List<SdkBytes> binaryListValues = DefaultSdkAutoConstructList.getInstance();

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            stringValue(model.stringValue);
            binaryValue(model.binaryValue);
            stringListValues(model.stringListValues);
            binaryListValues(model.binaryListValues);
            dataType(model.dataType);
        }

        public final String getStringValue() {
            return stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue == null ? null : binaryValue.asByteBuffer();
        }

        @Override
        public final Builder binaryValue(SdkBytes binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
            return this;
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            binaryValue(binaryValue == null ? null : SdkBytes.fromByteBuffer(binaryValue));
        }

        public final Collection<String> getStringListValues() {
            return stringListValues;
        }

        @Override
        public final Builder stringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stringListValues(String... stringListValues) {
            stringListValues(Arrays.asList(stringListValues));
            return this;
        }

        public final void setStringListValues(Collection<String> stringListValues) {
            this.stringListValues = StringListCopier.copy(stringListValues);
        }

        public final List<ByteBuffer> getBinaryListValues() {
            return binaryListValues == null ? null : binaryListValues.stream().map(SdkBytes::asByteBuffer)
                    .collect(Collectors.toList());
        }

        @Override
        public final Builder binaryListValues(Collection<SdkBytes> binaryListValues) {
            this.binaryListValues = BinaryListCopier.copy(binaryListValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder binaryListValues(SdkBytes... binaryListValues) {
            binaryListValues(Arrays.asList(binaryListValues));
            return this;
        }

        public final void setBinaryListValues(Collection<ByteBuffer> binaryListValues) {
            binaryListValues(binaryListValues == null ? null : binaryListValues.stream().map(SdkBytes::fromByteBuffer)
                    .collect(Collectors.toList()));
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
