/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of a single Amazon SQS message along with an <code>Id</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendMessageBatchRequestEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<SendMessageBatchRequestEntry.Builder, SendMessageBatchRequestEntry> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendMessageBatchRequestEntry::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> MESSAGE_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendMessageBatchRequestEntry::messageBody)).setter(setter(Builder::messageBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageBody").build()).build();

    private static final SdkField<Integer> DELAY_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(SendMessageBatchRequestEntry::delaySeconds)).setter(setter(Builder::delaySeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DelaySeconds").build()).build();

    private static final SdkField<Map<String, MessageAttributeValue>> MESSAGE_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageAttributeValue>> builder(MarshallingType.MAP)
            .getter(getter(SendMessageBatchRequestEntry::messageAttributes))
            .setter(setter(Builder::messageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttribute").build(),
                    MapTrait.builder()
                            .keyLocationName("Name")
                            .valueLocationName("Value")
                            .valueFieldInfo(
                                    SdkField.<MessageAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<Map<String, MessageSystemAttributeValue>> MESSAGE_SYSTEM_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageSystemAttributeValue>> builder(MarshallingType.MAP)
            .getter(getter(SendMessageBatchRequestEntry::messageSystemAttributesAsStrings))
            .setter(setter(Builder::messageSystemAttributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageSystemAttribute").build(),
                    MapTrait.builder()
                            .keyLocationName("Name")
                            .valueLocationName("Value")
                            .valueFieldInfo(
                                    SdkField.<MessageSystemAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageSystemAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<String> MESSAGE_DEDUPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendMessageBatchRequestEntry::messageDeduplicationId)).setter(setter(Builder::messageDeduplicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageDeduplicationId").build())
            .build();

    private static final SdkField<String> MESSAGE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendMessageBatchRequestEntry::messageGroupId)).setter(setter(Builder::messageGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageGroupId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, MESSAGE_BODY_FIELD,
            DELAY_SECONDS_FIELD, MESSAGE_ATTRIBUTES_FIELD, MESSAGE_SYSTEM_ATTRIBUTES_FIELD, MESSAGE_DEDUPLICATION_ID_FIELD,
            MESSAGE_GROUP_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String messageBody;

    private final Integer delaySeconds;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private final Map<String, MessageSystemAttributeValue> messageSystemAttributes;

    private final String messageDeduplicationId;

    private final String messageGroupId;

    private SendMessageBatchRequestEntry(BuilderImpl builder) {
        this.id = builder.id;
        this.messageBody = builder.messageBody;
        this.delaySeconds = builder.delaySeconds;
        this.messageAttributes = builder.messageAttributes;
        this.messageSystemAttributes = builder.messageSystemAttributes;
        this.messageDeduplicationId = builder.messageDeduplicationId;
        this.messageGroupId = builder.messageGroupId;
    }

    /**
     * <p>
     * An identifier for a message in this batch used to communicate the result.
     * </p>
     * <note>
     * <p>
     * The <code>Id</code>s of a batch request need to be unique within a request.
     * </p>
     * <p>
     * This identifier can have up to 80 characters. The following characters are accepted: alphanumeric characters,
     * hyphens(-), and underscores (_).
     * </p>
     * </note>
     * 
     * @return An identifier for a message in this batch used to communicate the result.</p> <note>
     *         <p>
     *         The <code>Id</code>s of a batch request need to be unique within a request.
     *         </p>
     *         <p>
     *         This identifier can have up to 80 characters. The following characters are accepted: alphanumeric
     *         characters, hyphens(-), and underscores (_).
     *         </p>
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The body of the message.
     * </p>
     * 
     * @return The body of the message.
     */
    public String messageBody() {
        return messageBody;
    }

    /**
     * <p>
     * The length of time, in seconds, for which a specific message is delayed. Valid values: 0 to 900. Maximum: 15
     * minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing after the delay
     * period is finished. If you don't specify a value, the default value for the queue is applied.
     * </p>
     * <note>
     * <p>
     * When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set this
     * parameter only on a queue level.
     * </p>
     * </note>
     * 
     * @return The length of time, in seconds, for which a specific message is delayed. Valid values: 0 to 900. Maximum:
     *         15 minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing
     *         after the delay period is finished. If you don't specify a value, the default value for the queue is
     *         applied. </p> <note>
     *         <p>
     *         When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set
     *         this parameter only on a queue level.
     *         </p>
     */
    public Integer delaySeconds() {
        return delaySeconds;
    }

    /**
     * Returns true if the MessageAttributes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMessageAttributes() {
        return messageAttributes != null && !(messageAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     * >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessageAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     *         >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     */
    public Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    /**
     * <p>
     * The message system attribute to send Each message system attribute consists of a <code>Name</code>,
     * <code>Type</code>, and <code>Value</code>.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     * <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * The size of a message system attribute doesn't count towards the total size of a message.
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessageSystemAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The message system attribute to send Each message system attribute consists of a <code>Name</code>,
     *         <code>Type</code>, and <code>Value</code>.</p> <important>
     *         <ul>
     *         <li>
     *         <p>
     *         Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     *         <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The size of a message system attribute doesn't count towards the total size of a message.
     *         </p>
     *         </li>
     *         </ul>
     */
    public Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes() {
        return MessageBodySystemAttributeMapCopier.copyStringToEnum(messageSystemAttributes);
    }

    /**
     * Returns true if the MessageSystemAttributes property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMessageSystemAttributes() {
        return messageSystemAttributes != null && !(messageSystemAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The message system attribute to send Each message system attribute consists of a <code>Name</code>,
     * <code>Type</code>, and <code>Value</code>.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     * <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * The size of a message system attribute doesn't count towards the total size of a message.
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessageSystemAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The message system attribute to send Each message system attribute consists of a <code>Name</code>,
     *         <code>Type</code>, and <code>Value</code>.</p> <important>
     *         <ul>
     *         <li>
     *         <p>
     *         Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     *         <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The size of a message system attribute doesn't count towards the total size of a message.
     *         </p>
     *         </li>
     *         </ul>
     */
    public Map<String, MessageSystemAttributeValue> messageSystemAttributesAsStrings() {
        return messageSystemAttributes;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) queues.
     * </p>
     * <p>
     * The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a message with
     * a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages with the same
     * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing"
     * > Exactly-Once Processing</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Every message must have a unique <code>MessageDeduplicationId</code>,
     * </p>
     * <ul>
     * <li>
     * <p>
     * You may provide a <code>MessageDeduplicationId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     * <code>ContentBasedDeduplication</code> set, the action fails with an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code> overrides
     * the generated one.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     * deduplication interval are treated as duplicates and only one copy of the message is delivered.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
     * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the message
     * is delivered.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
     * troubleshooting delivery issues).
     * </p>
     * <p>
     * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate messages.
     * </p>
     * <p>
     * Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
     * deleted.
     * </p>
     * </note>
     * <p>
     * The length of <code>MessageDeduplicationId</code> is 128 characters. <code>MessageDeduplicationId</code> can
     * contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and punctuation (
     * <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     * </p>
     * <p>
     * For best practices of using <code>MessageDeduplicationId</code>, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
     * >Using the MessageDeduplicationId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) queues.</p>
     *         <p>
     *         The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a
     *         message with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages
     *         with the same <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing"
     *         > Exactly-Once Processing</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Every message must have a unique <code>MessageDeduplicationId</code>,
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You may provide a <code>MessageDeduplicationId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     *         <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     *         <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     *         message).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     *         <code>ContentBasedDeduplication</code> set, the action fails with an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
     *         overrides the generated one.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     *         deduplication interval are treated as duplicates and only one copy of the message is delivered.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with
     *         a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     *         <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
     *         message is delivered.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful
     *         for troubleshooting delivery issues).
     *         </p>
     *         <p>
     *         If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     *         <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate
     *         messages.
     *         </p>
     *         <p>
     *         Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
     *         deleted.
     *         </p>
     *         </note>
     *         <p>
     *         The length of <code>MessageDeduplicationId</code> is 128 characters. <code>MessageDeduplicationId</code>
     *         can contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and
     *         punctuation (<code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     *         </p>
     *         <p>
     *         For best practices of using <code>MessageDeduplicationId</code>, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
     *         >Using the MessageDeduplicationId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     */
    public String messageDeduplicationId() {
        return messageDeduplicationId;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) queues.
     * </p>
     * <p>
     * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
     * message group are processed in a FIFO manner (however, messages in different message groups might be processed
     * out of order). To interleave multiple ordered streams within a single queue, use <code>MessageGroupId</code>
     * values (for example, session data for multiple users). In this scenario, multiple consumers can process the
     * queue, but the session data of each user is processed in a FIFO fashion.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     * <code>MessageGroupId</code>, the action fails.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For each
     * <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
     * <code>MessageGroupId</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
     * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     * </p>
     * <p>
     * For best practices of using <code>MessageGroupId</code>, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
     * >Using the MessageGroupId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
     * </p>
     * </important>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) queues.</p>
     *         <p>
     *         The tag that specifies that a message belongs to a specific message group. Messages that belong to the
     *         same message group are processed in a FIFO manner (however, messages in different message groups might be
     *         processed out of order). To interleave multiple ordered streams within a single queue, use
     *         <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
     *         multiple consumers can process the queue, but the session data of each user is processed in a FIFO
     *         fashion.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     *         <code>MessageGroupId</code>, the action fails.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For
     *         each <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
     *         <code>MessageGroupId</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
     *         punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     *         </p>
     *         <p>
     *         For best practices of using <code>MessageGroupId</code>, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
     *         >Using the MessageGroupId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
     *         </p>
     */
    public String messageGroupId() {
        return messageGroupId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(messageBody());
        hashCode = 31 * hashCode + Objects.hashCode(delaySeconds());
        hashCode = 31 * hashCode + Objects.hashCode(messageAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(messageSystemAttributesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(messageDeduplicationId());
        hashCode = 31 * hashCode + Objects.hashCode(messageGroupId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendMessageBatchRequestEntry)) {
            return false;
        }
        SendMessageBatchRequestEntry other = (SendMessageBatchRequestEntry) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(messageBody(), other.messageBody())
                && Objects.equals(delaySeconds(), other.delaySeconds())
                && Objects.equals(messageAttributes(), other.messageAttributes())
                && Objects.equals(messageSystemAttributesAsStrings(), other.messageSystemAttributesAsStrings())
                && Objects.equals(messageDeduplicationId(), other.messageDeduplicationId())
                && Objects.equals(messageGroupId(), other.messageGroupId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SendMessageBatchRequestEntry").add("Id", id()).add("MessageBody", messageBody())
                .add("DelaySeconds", delaySeconds()).add("MessageAttributes", messageAttributes())
                .add("MessageSystemAttributes", messageSystemAttributesAsStrings())
                .add("MessageDeduplicationId", messageDeduplicationId()).add("MessageGroupId", messageGroupId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "MessageBody":
            return Optional.ofNullable(clazz.cast(messageBody()));
        case "DelaySeconds":
            return Optional.ofNullable(clazz.cast(delaySeconds()));
        case "MessageAttributes":
            return Optional.ofNullable(clazz.cast(messageAttributes()));
        case "MessageSystemAttributes":
            return Optional.ofNullable(clazz.cast(messageSystemAttributesAsStrings()));
        case "MessageDeduplicationId":
            return Optional.ofNullable(clazz.cast(messageDeduplicationId()));
        case "MessageGroupId":
            return Optional.ofNullable(clazz.cast(messageGroupId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendMessageBatchRequestEntry, T> g) {
        return obj -> g.apply((SendMessageBatchRequestEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SendMessageBatchRequestEntry> {
        /**
         * <p>
         * An identifier for a message in this batch used to communicate the result.
         * </p>
         * <note>
         * <p>
         * The <code>Id</code>s of a batch request need to be unique within a request.
         * </p>
         * <p>
         * This identifier can have up to 80 characters. The following characters are accepted: alphanumeric characters,
         * hyphens(-), and underscores (_).
         * </p>
         * </note>
         * 
         * @param id
         *        An identifier for a message in this batch used to communicate the result.</p> <note>
         *        <p>
         *        The <code>Id</code>s of a batch request need to be unique within a request.
         *        </p>
         *        <p>
         *        This identifier can have up to 80 characters. The following characters are accepted: alphanumeric
         *        characters, hyphens(-), and underscores (_).
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The body of the message.
         * </p>
         * 
         * @param messageBody
         *        The body of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageBody(String messageBody);

        /**
         * <p>
         * The length of time, in seconds, for which a specific message is delayed. Valid values: 0 to 900. Maximum: 15
         * minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing after the
         * delay period is finished. If you don't specify a value, the default value for the queue is applied.
         * </p>
         * <note>
         * <p>
         * When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set this
         * parameter only on a queue level.
         * </p>
         * </note>
         * 
         * @param delaySeconds
         *        The length of time, in seconds, for which a specific message is delayed. Valid values: 0 to 900.
         *        Maximum: 15 minutes. Messages with a positive <code>DelaySeconds</code> value become available for
         *        processing after the delay period is finished. If you don't specify a value, the default value for the
         *        queue is applied. </p> <note>
         *        <p>
         *        When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set
         *        this parameter only on a queue level.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delaySeconds(Integer delaySeconds);

        /**
         * <p>
         * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         * >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * 
         * @param messageAttributes
         *        Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         *        >Amazon SQS Message Attributes</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);

        /**
         * <p>
         * The message system attribute to send Each message system attribute consists of a <code>Name</code>,
         * <code>Type</code>, and <code>Value</code>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
         * <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * The size of a message system attribute doesn't count towards the total size of a message.
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param messageSystemAttributes
         *        The message system attribute to send Each message system attribute consists of a <code>Name</code>,
         *        <code>Type</code>, and <code>Value</code>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must
         *        be <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The size of a message system attribute doesn't count towards the total size of a message.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageSystemAttributesWithStrings(Map<String, MessageSystemAttributeValue> messageSystemAttributes);

        /**
         * <p>
         * The message system attribute to send Each message system attribute consists of a <code>Name</code>,
         * <code>Type</code>, and <code>Value</code>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
         * <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * The size of a message system attribute doesn't count towards the total size of a message.
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param messageSystemAttributes
         *        The message system attribute to send Each message system attribute consists of a <code>Name</code>,
         *        <code>Type</code>, and <code>Value</code>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must
         *        be <code>String</code> and its value must be a correctly formatted AWS X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The size of a message system attribute doesn't count towards the total size of a message.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageSystemAttributes(
                Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) queues.
         * </p>
         * <p>
         * The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a message
         * with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages with the same
         * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered. For more information, see
         * <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing"
         * > Exactly-Once Processing</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Every message must have a unique <code>MessageDeduplicationId</code>,
         * </p>
         * <ul>
         * <li>
         * <p>
         * You may provide a <code>MessageDeduplicationId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         * <code>ContentBasedDeduplication</code> set, the action fails with an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         * overrides the generated one.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
         * deduplication interval are treated as duplicates and only one copy of the message is delivered.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
         * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
         * message is delivered.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
         * troubleshooting delivery issues).
         * </p>
         * <p>
         * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
         * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate
         * messages.
         * </p>
         * <p>
         * Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
         * deleted.
         * </p>
         * </note>
         * <p>
         * The length of <code>MessageDeduplicationId</code> is 128 characters. <code>MessageDeduplicationId</code> can
         * contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and punctuation (
         * <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         * </p>
         * <p>
         * For best practices of using <code>MessageDeduplicationId</code>, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
         * >Using the MessageDeduplicationId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * 
         * @param messageDeduplicationId
         *        This parameter applies only to FIFO (first-in-first-out) queues.</p>
         *        <p>
         *        The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a
         *        message with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent
         *        messages with the same <code>MessageDeduplicationId</code> are accepted successfully but aren't
         *        delivered. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-exactly-once-processing"
         *        > Exactly-Once Processing</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Every message must have a unique <code>MessageDeduplicationId</code>,
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You may provide a <code>MessageDeduplicationId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         *        <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         *        <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         *        message).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         *        <code>ContentBasedDeduplication</code> set, the action fails with an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         *        overrides the generated one.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within
         *        the deduplication interval are treated as duplicates and only one copy of the message is delivered.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message
         *        with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         *        <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of
         *        the message is delivered.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be
         *        useful for troubleshooting delivery issues).
         *        </p>
         *        <p>
         *        If a message is sent successfully but the acknowledgement is lost and the message is resent with the
         *        same <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect
         *        duplicate messages.
         *        </p>
         *        <p>
         *        Amazon SQS continues to keep track of the message deduplication ID even after the message is received
         *        and deleted.
         *        </p>
         *        </note>
         *        <p>
         *        The length of <code>MessageDeduplicationId</code> is 128 characters.
         *        <code>MessageDeduplicationId</code> can contain alphanumeric characters (<code>a-z</code>,
         *        <code>A-Z</code>, <code>0-9</code>) and punctuation (
         *        <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         *        </p>
         *        <p>
         *        For best practices of using <code>MessageDeduplicationId</code>, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
         *        >Using the MessageDeduplicationId Property</a> in the <i>Amazon Simple Queue Service Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageDeduplicationId(String messageDeduplicationId);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) queues.
         * </p>
         * <p>
         * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
         * message group are processed in a FIFO manner (however, messages in different message groups might be
         * processed out of order). To interleave multiple ordered streams within a single queue, use
         * <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario, multiple
         * consumers can process the queue, but the session data of each user is processed in a FIFO fashion.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         * <code>MessageGroupId</code>, the action fails.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For each
         * <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
         * <code>MessageGroupId</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
         * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         * </p>
         * <p>
         * For best practices of using <code>MessageGroupId</code>, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
         * >Using the MessageGroupId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
         * </p>
         * </important>
         * 
         * @param messageGroupId
         *        This parameter applies only to FIFO (first-in-first-out) queues.</p>
         *        <p>
         *        The tag that specifies that a message belongs to a specific message group. Messages that belong to the
         *        same message group are processed in a FIFO manner (however, messages in different message groups might
         *        be processed out of order). To interleave multiple ordered streams within a single queue, use
         *        <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
         *        multiple consumers can process the queue, but the session data of each user is processed in a FIFO
         *        fashion.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         *        <code>MessageGroupId</code>, the action fails.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values.
         *        For each <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
         *        <code>MessageGroupId</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
         *        punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         *        </p>
         *        <p>
         *        For best practices of using <code>MessageGroupId</code>, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
         *        >Using the MessageGroupId Property</a> in the <i>Amazon Simple Queue Service Developer Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageGroupId(String messageGroupId);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String messageBody;

        private Integer delaySeconds;

        private Map<String, MessageAttributeValue> messageAttributes = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, MessageSystemAttributeValue> messageSystemAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String messageDeduplicationId;

        private String messageGroupId;

        private BuilderImpl() {
        }

        private BuilderImpl(SendMessageBatchRequestEntry model) {
            id(model.id);
            messageBody(model.messageBody);
            delaySeconds(model.delaySeconds);
            messageAttributes(model.messageAttributes);
            messageSystemAttributesWithStrings(model.messageSystemAttributes);
            messageDeduplicationId(model.messageDeduplicationId);
            messageGroupId(model.messageGroupId);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getMessageBody() {
            return messageBody;
        }

        @Override
        public final Builder messageBody(String messageBody) {
            this.messageBody = messageBody;
            return this;
        }

        public final void setMessageBody(String messageBody) {
            this.messageBody = messageBody;
        }

        public final Integer getDelaySeconds() {
            return delaySeconds;
        }

        @Override
        public final Builder delaySeconds(Integer delaySeconds) {
            this.delaySeconds = delaySeconds;
            return this;
        }

        public final void setDelaySeconds(Integer delaySeconds) {
            this.delaySeconds = delaySeconds;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            return messageAttributes != null ? CollectionUtils.mapValues(messageAttributes, MessageAttributeValue::toBuilder)
                    : null;
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        public final Map<String, MessageSystemAttributeValue.Builder> getMessageSystemAttributes() {
            return messageSystemAttributes != null ? CollectionUtils.mapValues(messageSystemAttributes,
                    MessageSystemAttributeValue::toBuilder) : null;
        }

        @Override
        public final Builder messageSystemAttributesWithStrings(Map<String, MessageSystemAttributeValue> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copy(messageSystemAttributes);
            return this;
        }

        @Override
        public final Builder messageSystemAttributes(
                Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copyEnumToString(messageSystemAttributes);
            return this;
        }

        public final void setMessageSystemAttributes(Map<String, MessageSystemAttributeValue.BuilderImpl> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copyFromBuilder(messageSystemAttributes);
        }

        public final String getMessageDeduplicationId() {
            return messageDeduplicationId;
        }

        @Override
        public final Builder messageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
            return this;
        }

        public final void setMessageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
        }

        public final String getMessageGroupId() {
            return messageGroupId;
        }

        @Override
        public final Builder messageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
            return this;
        }

        public final void setMessageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
        }

        @Override
        public SendMessageBatchRequestEntry build() {
            return new SendMessageBatchRequestEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
