/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Amazon SQS message.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Message implements SdkPojo, Serializable, ToCopyableBuilder<Message.Builder, Message> {
    private static final SdkField<String> MESSAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageId").getter(getter(Message::messageId)).setter(setter(Builder::messageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageId").build()).build();

    private static final SdkField<String> RECEIPT_HANDLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReceiptHandle").getter(getter(Message::receiptHandle)).setter(setter(Builder::receiptHandle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReceiptHandle").build()).build();

    private static final SdkField<String> MD5_OF_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MD5OfBody").getter(getter(Message::md5OfBody)).setter(setter(Builder::md5OfBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MD5OfBody").build()).build();

    private static final SdkField<String> BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Body")
            .getter(getter(Message::body)).setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Body").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(Message::attributesAsStrings))
            .setter(setter(Builder::attributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<String> MD5_OF_MESSAGE_ATTRIBUTES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MD5OfMessageAttributes").getter(getter(Message::md5OfMessageAttributes))
            .setter(setter(Builder::md5OfMessageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MD5OfMessageAttributes").build())
            .build();

    private static final SdkField<Map<String, MessageAttributeValue>> MESSAGE_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageAttributeValue>> builder(MarshallingType.MAP)
            .memberName("MessageAttributes")
            .getter(getter(Message::messageAttributes))
            .setter(setter(Builder::messageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MessageAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MESSAGE_ID_FIELD,
            RECEIPT_HANDLE_FIELD, MD5_OF_BODY_FIELD, BODY_FIELD, ATTRIBUTES_FIELD, MD5_OF_MESSAGE_ATTRIBUTES_FIELD,
            MESSAGE_ATTRIBUTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("MessageId", MESSAGE_ID_FIELD);
                    put("ReceiptHandle", RECEIPT_HANDLE_FIELD);
                    put("MD5OfBody", MD5_OF_BODY_FIELD);
                    put("Body", BODY_FIELD);
                    put("Attributes", ATTRIBUTES_FIELD);
                    put("MD5OfMessageAttributes", MD5_OF_MESSAGE_ATTRIBUTES_FIELD);
                    put("MessageAttributes", MESSAGE_ATTRIBUTES_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String messageId;

    private final String receiptHandle;

    private final String md5OfBody;

    private final String body;

    private final Map<String, String> attributes;

    private final String md5OfMessageAttributes;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private Message(BuilderImpl builder) {
        this.messageId = builder.messageId;
        this.receiptHandle = builder.receiptHandle;
        this.md5OfBody = builder.md5OfBody;
        this.body = builder.body;
        this.attributes = builder.attributes;
        this.md5OfMessageAttributes = builder.md5OfMessageAttributes;
        this.messageAttributes = builder.messageAttributes;
    }

    /**
     * <p>
     * A unique identifier for the message. A <code>MessageId</code>is considered unique across all Amazon Web Services
     * accounts for an extended period of time.
     * </p>
     * 
     * @return A unique identifier for the message. A <code>MessageId</code>is considered unique across all Amazon Web
     *         Services accounts for an extended period of time.
     */
    public final String messageId() {
        return messageId;
    }

    /**
     * <p>
     * An identifier associated with the act of receiving the message. A new receipt handle is returned every time you
     * receive a message. When deleting a message, you provide the last received receipt handle to delete the message.
     * </p>
     * 
     * @return An identifier associated with the act of receiving the message. A new receipt handle is returned every
     *         time you receive a message. When deleting a message, you provide the last received receipt handle to
     *         delete the message.
     */
    public final String receiptHandle() {
        return receiptHandle;
    }

    /**
     * <p>
     * An MD5 digest of the non-URL-encoded message body string.
     * </p>
     * 
     * @return An MD5 digest of the non-URL-encoded message body string.
     */
    public final String md5OfBody() {
        return md5OfBody;
    }

    /**
     * <p>
     * The message's contents (not URL-encoded).
     * </p>
     * 
     * @return The message's contents (not URL-encoded).
     */
    public final String body() {
        return body;
    }

    /**
     * <p>
     * A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values. Supported
     * attributes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ApproximateReceiveCount</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageDeduplicationId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageGroupId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SenderId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SentTimestamp</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SequenceNumber</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an integer
     * representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
     *         Supported attributes:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ApproximateReceiveCount</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageDeduplicationId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageGroupId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SenderId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SentTimestamp</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SequenceNumber</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an
     *         integer representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     */
    public final Map<MessageSystemAttributeName, String> attributes() {
        return MessageSystemAttributeMapCopier.copyStringToEnum(attributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values. Supported
     * attributes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ApproximateReceiveCount</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageDeduplicationId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MessageGroupId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SenderId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SentTimestamp</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SequenceNumber</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an integer
     * representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
     *         Supported attributes:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ApproximateReceiveCount</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageDeduplicationId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MessageGroupId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SenderId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SentTimestamp</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SequenceNumber</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an
     *         integer representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
     */
    public final Map<String, String> attributesAsStrings() {
        return attributes;
    }

    /**
     * <p>
     * An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that Amazon
     * SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5 digest. For
     * information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     * </p>
     * 
     * @return An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that
     *         Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5
     *         digest. For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     */
    public final String md5OfMessageAttributes() {
        return md5OfMessageAttributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMessageAttributes() {
        return messageAttributes != null && !(messageAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     * >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageAttributes} method.
     * </p>
     * 
     * @return Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     *         >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public final Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(messageId());
        hashCode = 31 * hashCode + Objects.hashCode(receiptHandle());
        hashCode = 31 * hashCode + Objects.hashCode(md5OfBody());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(md5OfMessageAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageAttributes() ? messageAttributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Message)) {
            return false;
        }
        Message other = (Message) obj;
        return Objects.equals(messageId(), other.messageId()) && Objects.equals(receiptHandle(), other.receiptHandle())
                && Objects.equals(md5OfBody(), other.md5OfBody()) && Objects.equals(body(), other.body())
                && hasAttributes() == other.hasAttributes() && Objects.equals(attributesAsStrings(), other.attributesAsStrings())
                && Objects.equals(md5OfMessageAttributes(), other.md5OfMessageAttributes())
                && hasMessageAttributes() == other.hasMessageAttributes()
                && Objects.equals(messageAttributes(), other.messageAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Message").add("MessageId", messageId()).add("ReceiptHandle", receiptHandle())
                .add("MD5OfBody", md5OfBody()).add("Body", body())
                .add("Attributes", hasAttributes() ? attributesAsStrings() : null)
                .add("MD5OfMessageAttributes", md5OfMessageAttributes())
                .add("MessageAttributes", hasMessageAttributes() ? messageAttributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MessageId":
            return Optional.ofNullable(clazz.cast(messageId()));
        case "ReceiptHandle":
            return Optional.ofNullable(clazz.cast(receiptHandle()));
        case "MD5OfBody":
            return Optional.ofNullable(clazz.cast(md5OfBody()));
        case "Body":
            return Optional.ofNullable(clazz.cast(body()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributesAsStrings()));
        case "MD5OfMessageAttributes":
            return Optional.ofNullable(clazz.cast(md5OfMessageAttributes()));
        case "MessageAttributes":
            return Optional.ofNullable(clazz.cast(messageAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<Message, T> g) {
        return obj -> g.apply((Message) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Message> {
        /**
         * <p>
         * A unique identifier for the message. A <code>MessageId</code>is considered unique across all Amazon Web
         * Services accounts for an extended period of time.
         * </p>
         * 
         * @param messageId
         *        A unique identifier for the message. A <code>MessageId</code>is considered unique across all Amazon
         *        Web Services accounts for an extended period of time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageId(String messageId);

        /**
         * <p>
         * An identifier associated with the act of receiving the message. A new receipt handle is returned every time
         * you receive a message. When deleting a message, you provide the last received receipt handle to delete the
         * message.
         * </p>
         * 
         * @param receiptHandle
         *        An identifier associated with the act of receiving the message. A new receipt handle is returned every
         *        time you receive a message. When deleting a message, you provide the last received receipt handle to
         *        delete the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receiptHandle(String receiptHandle);

        /**
         * <p>
         * An MD5 digest of the non-URL-encoded message body string.
         * </p>
         * 
         * @param md5OfBody
         *        An MD5 digest of the non-URL-encoded message body string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder md5OfBody(String md5OfBody);

        /**
         * <p>
         * The message's contents (not URL-encoded).
         * </p>
         * 
         * @param body
         *        The message's contents (not URL-encoded).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
         * Supported attributes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an integer
         * representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
         * </p>
         * 
         * @param attributes
         *        A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
         *        Supported attributes:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an
         *        integer representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesWithStrings(Map<String, String> attributes);

        /**
         * <p>
         * A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
         * Supported attributes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ApproximateReceiveCount</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageDeduplicationId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MessageGroupId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SenderId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SentTimestamp</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SequenceNumber</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an integer
         * representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in milliseconds.
         * </p>
         * 
         * @param attributes
         *        A map of the attributes requested in <code> <a>ReceiveMessage</a> </code> to their respective values.
         *        Supported attributes:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ApproximateReceiveCount</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageDeduplicationId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MessageGroupId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SenderId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SentTimestamp</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SequenceNumber</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>ApproximateFirstReceiveTimestamp</code> and <code>SentTimestamp</code> are each returned as an
         *        integer representing the <a href="http://en.wikipedia.org/wiki/Unix_time">epoch time</a> in
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<MessageSystemAttributeName, String> attributes);

        /**
         * <p>
         * An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify that
         * Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the MD5 digest.
         * For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
         * </p>
         * 
         * @param md5OfMessageAttributes
         *        An MD5 digest of the non-URL-encoded message attribute string. You can use this attribute to verify
         *        that Amazon SQS received the message correctly. Amazon SQS URL-decodes the message before creating the
         *        MD5 digest. For information about MD5, see <a href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder md5OfMessageAttributes(String md5OfMessageAttributes);

        /**
         * <p>
         * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         * >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param messageAttributes
         *        Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         *        >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String messageId;

        private String receiptHandle;

        private String md5OfBody;

        private String body;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private String md5OfMessageAttributes;

        private Map<String, MessageAttributeValue> messageAttributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Message model) {
            messageId(model.messageId);
            receiptHandle(model.receiptHandle);
            md5OfBody(model.md5OfBody);
            body(model.body);
            attributesWithStrings(model.attributes);
            md5OfMessageAttributes(model.md5OfMessageAttributes);
            messageAttributes(model.messageAttributes);
        }

        public final String getMessageId() {
            return messageId;
        }

        public final void setMessageId(String messageId) {
            this.messageId = messageId;
        }

        @Override
        public final Builder messageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        public final String getReceiptHandle() {
            return receiptHandle;
        }

        public final void setReceiptHandle(String receiptHandle) {
            this.receiptHandle = receiptHandle;
        }

        @Override
        public final Builder receiptHandle(String receiptHandle) {
            this.receiptHandle = receiptHandle;
            return this;
        }

        public final String getMd5OfBody() {
            return md5OfBody;
        }

        public final void setMd5OfBody(String md5OfBody) {
            this.md5OfBody = md5OfBody;
        }

        @Override
        public final Builder md5OfBody(String md5OfBody) {
            this.md5OfBody = md5OfBody;
            return this;
        }

        public final String getBody() {
            return body;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = MessageSystemAttributeMapCopier.copy(attributes);
        }

        @Override
        public final Builder attributesWithStrings(Map<String, String> attributes) {
            this.attributes = MessageSystemAttributeMapCopier.copy(attributes);
            return this;
        }

        @Override
        public final Builder attributes(Map<MessageSystemAttributeName, String> attributes) {
            this.attributes = MessageSystemAttributeMapCopier.copyEnumToString(attributes);
            return this;
        }

        public final String getMd5OfMessageAttributes() {
            return md5OfMessageAttributes;
        }

        public final void setMd5OfMessageAttributes(String md5OfMessageAttributes) {
            this.md5OfMessageAttributes = md5OfMessageAttributes;
        }

        @Override
        public final Builder md5OfMessageAttributes(String md5OfMessageAttributes) {
            this.md5OfMessageAttributes = md5OfMessageAttributes;
            return this;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            Map<String, MessageAttributeValue.Builder> result = MessageBodyAttributeMapCopier
                    .copyToBuilder(this.messageAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        @Override
        public Message build() {
            return new Message(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
