/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAssociationRequest extends SsmRequest implements
        ToCopyableBuilder<CreateAssociationRequest.Builder, CreateAssociationRequest> {
    private final String name;

    private final String documentVersion;

    private final String instanceId;

    private final Map<String, List<String>> parameters;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final String associationName;

    private CreateAssociationRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.documentVersion = builder.documentVersion;
        this.instanceId = builder.instanceId;
        this.parameters = builder.parameters;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The document version you want to associate with the target(s). Can be a specific version or the default version.
     * </p>
     * 
     * @return The document version you want to associate with the target(s). Can be a specific version or the default
     *         version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The instance ID.
     * </p>
     * 
     * @return The instance ID.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The parameters for the documents runtime configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The parameters for the documents runtime configuration.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The targets (either instances or tags) for the association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets (either instances or tags) for the association.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A cron expression when the association will be applied to the target(s).
     * </p>
     * 
     * @return A cron expression when the association will be applied to the target(s).
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An Amazon S3 bucket where you want to store the output details of the request.
     * </p>
     * 
     * @return An Amazon S3 bucket where you want to store the output details of the request.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * Specify a descriptive name for the association.
     * </p>
     * 
     * @return Specify a descriptive name for the association.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAssociationRequest)) {
            return false;
        }
        CreateAssociationRequest other = (CreateAssociationRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(instanceId(), other.instanceId()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(targets(), other.targets()) && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateAssociationRequest").add("Name", name()).add("DocumentVersion", documentVersion())
                .add("InstanceId", instanceId()).add("Parameters", parameters()).add("Targets", targets())
                .add("ScheduleExpression", scheduleExpression()).add("OutputLocation", outputLocation())
                .add("AssociationName", associationName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "AssociationName":
            return Optional.ofNullable(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SsmRequest.Builder, CopyableBuilder<Builder, CreateAssociationRequest> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The document version you want to associate with the target(s). Can be a specific version or the default
         * version.
         * </p>
         * 
         * @param documentVersion
         *        The document version you want to associate with the target(s). Can be a specific version or the
         *        default version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The instance ID.
         * </p>
         * 
         * @param instanceId
         *        The instance ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The parameters for the documents runtime configuration.
         * </p>
         * 
         * @param parameters
         *        The parameters for the documents runtime configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The targets (either instances or tags) for the association.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags) for the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets (either instances or tags) for the association.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags) for the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets (either instances or tags) for the association.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * A cron expression when the association will be applied to the target(s).
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression when the association will be applied to the target(s).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * 
         * @param outputLocation
         *        An Amazon S3 bucket where you want to store the output details of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().applyMutation(outputLocation).build());
        }

        /**
         * <p>
         * Specify a descriptive name for the association.
         * </p>
         * 
         * @param associationName
         *        Specify a descriptive name for the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String name;

        private String documentVersion;

        private String instanceId;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAssociationRequest model) {
            super(model);
            name(model.name);
            documentVersion(model.documentVersion);
            instanceId(model.instanceId);
            parameters(model.parameters);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            associationName(model.associationName);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAssociationRequest build() {
            return new CreateAssociationRequest(this);
        }
    }
}
