/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetParametersByPathRequest extends SsmRequest implements
        ToCopyableBuilder<GetParametersByPathRequest.Builder, GetParametersByPathRequest> {
    private final String path;

    private final Boolean recursive;

    private final List<ParameterStringFilter> parameterFilters;

    private final Boolean withDecryption;

    private final Integer maxResults;

    private final String nextToken;

    private GetParametersByPathRequest(BuilderImpl builder) {
        super(builder);
        this.path = builder.path;
        this.recursive = builder.recursive;
        this.parameterFilters = builder.parameterFilters;
        this.withDecryption = builder.withDecryption;
        this.maxResults = builder.maxResults;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * The hierarchy for the parameter. Hierarchies start with a forward slash (/) and end with the parameter name. A
     * hierarchy can have a maximum of five levels. For example: <code>/Finance/Prod/IAD/WinServ2016/license15</code>
     * </p>
     * 
     * @return The hierarchy for the parameter. Hierarchies start with a forward slash (/) and end with the parameter
     *         name. A hierarchy can have a maximum of five levels. For example:
     *         <code>/Finance/Prod/IAD/WinServ2016/license15</code>
     */
    public String path() {
        return path;
    }

    /**
     * <p>
     * Retrieve all parameters within a hierarchy.
     * </p>
     * 
     * @return Retrieve all parameters within a hierarchy.
     */
    public Boolean recursive() {
        return recursive;
    }

    /**
     * <p>
     * Filters to limit the request results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Filters to limit the request results.
     */
    public List<ParameterStringFilter> parameterFilters() {
        return parameterFilters;
    }

    /**
     * <p>
     * Retrieve all parameters in a hierarchy with their value decrypted.
     * </p>
     * 
     * @return Retrieve all parameters in a hierarchy with their value decrypted.
     */
    public Boolean withDecryption() {
        return withDecryption;
    }

    /**
     * <p>
     * The maximum number of items to return for this call. The call also returns a token that you can specify in a
     * subsequent call to get the next set of results.
     * </p>
     * 
     * @return The maximum number of items to return for this call. The call also returns a token that you can specify
     *         in a subsequent call to get the next set of results.
     */
    public Integer maxResults() {
        return maxResults;
    }

    /**
     * <p>
     * A token to start the list. Use this token to get the next set of results.
     * </p>
     * 
     * @return A token to start the list. Use this token to get the next set of results.
     */
    public String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(recursive());
        hashCode = 31 * hashCode + Objects.hashCode(parameterFilters());
        hashCode = 31 * hashCode + Objects.hashCode(withDecryption());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetParametersByPathRequest)) {
            return false;
        }
        GetParametersByPathRequest other = (GetParametersByPathRequest) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(recursive(), other.recursive())
                && Objects.equals(parameterFilters(), other.parameterFilters())
                && Objects.equals(withDecryption(), other.withDecryption()) && Objects.equals(maxResults(), other.maxResults())
                && Objects.equals(nextToken(), other.nextToken());
    }

    @Override
    public String toString() {
        return ToString.builder("GetParametersByPathRequest").add("Path", path()).add("Recursive", recursive())
                .add("ParameterFilters", parameterFilters()).add("WithDecryption", withDecryption())
                .add("MaxResults", maxResults()).add("NextToken", nextToken()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Recursive":
            return Optional.ofNullable(clazz.cast(recursive()));
        case "ParameterFilters":
            return Optional.ofNullable(clazz.cast(parameterFilters()));
        case "WithDecryption":
            return Optional.ofNullable(clazz.cast(withDecryption()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SsmRequest.Builder, CopyableBuilder<Builder, GetParametersByPathRequest> {
        /**
         * <p>
         * The hierarchy for the parameter. Hierarchies start with a forward slash (/) and end with the parameter name.
         * A hierarchy can have a maximum of five levels. For example:
         * <code>/Finance/Prod/IAD/WinServ2016/license15</code>
         * </p>
         * 
         * @param path
         *        The hierarchy for the parameter. Hierarchies start with a forward slash (/) and end with the parameter
         *        name. A hierarchy can have a maximum of five levels. For example:
         *        <code>/Finance/Prod/IAD/WinServ2016/license15</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Retrieve all parameters within a hierarchy.
         * </p>
         * 
         * @param recursive
         *        Retrieve all parameters within a hierarchy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recursive(Boolean recursive);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * 
         * @param parameterFilters
         *        Filters to limit the request results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterFilters(Collection<ParameterStringFilter> parameterFilters);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * 
         * @param parameterFilters
         *        Filters to limit the request results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterFilters(ParameterStringFilter... parameterFilters);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ParameterStringFilter>.Builder} avoiding
         * the need to create one manually via {@link List<ParameterStringFilter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ParameterStringFilter>.Builder#build()} is called
         * immediately and its result is passed to {@link #parameterFilters(List<ParameterStringFilter>)}.
         * 
         * @param parameterFilters
         *        a consumer that will call methods on {@link List<ParameterStringFilter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterFilters(List<ParameterStringFilter>)
         */
        Builder parameterFilters(Consumer<ParameterStringFilter.Builder>... parameterFilters);

        /**
         * <p>
         * Retrieve all parameters in a hierarchy with their value decrypted.
         * </p>
         * 
         * @param withDecryption
         *        Retrieve all parameters in a hierarchy with their value decrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder withDecryption(Boolean withDecryption);

        /**
         * <p>
         * The maximum number of items to return for this call. The call also returns a token that you can specify in a
         * subsequent call to get the next set of results.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of items to return for this call. The call also returns a token that you can
         *        specify in a subsequent call to get the next set of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        /**
         * <p>
         * A token to start the list. Use this token to get the next set of results.
         * </p>
         * 
         * @param nextToken
         *        A token to start the list. Use this token to get the next set of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String path;

        private Boolean recursive;

        private List<ParameterStringFilter> parameterFilters = DefaultSdkAutoConstructList.getInstance();

        private Boolean withDecryption;

        private Integer maxResults;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(GetParametersByPathRequest model) {
            super(model);
            path(model.path);
            recursive(model.recursive);
            parameterFilters(model.parameterFilters);
            withDecryption(model.withDecryption);
            maxResults(model.maxResults);
            nextToken(model.nextToken);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final Boolean getRecursive() {
            return recursive;
        }

        @Override
        public final Builder recursive(Boolean recursive) {
            this.recursive = recursive;
            return this;
        }

        public final void setRecursive(Boolean recursive) {
            this.recursive = recursive;
        }

        public final Collection<ParameterStringFilter.Builder> getParameterFilters() {
            return parameterFilters != null ? parameterFilters.stream().map(ParameterStringFilter::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameterFilters(Collection<ParameterStringFilter> parameterFilters) {
            this.parameterFilters = ParameterStringFilterListCopier.copy(parameterFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterFilters(ParameterStringFilter... parameterFilters) {
            parameterFilters(Arrays.asList(parameterFilters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterFilters(Consumer<ParameterStringFilter.Builder>... parameterFilters) {
            parameterFilters(Stream.of(parameterFilters).map(c -> ParameterStringFilter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameterFilters(Collection<ParameterStringFilter.BuilderImpl> parameterFilters) {
            this.parameterFilters = ParameterStringFilterListCopier.copyFromBuilder(parameterFilters);
        }

        public final Boolean getWithDecryption() {
            return withDecryption;
        }

        @Override
        public final Builder withDecryption(Boolean withDecryption) {
            this.withDecryption = withDecryption;
            return this;
        }

        public final void setWithDecryption(Boolean withDecryption) {
            this.withDecryption = withDecryption;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetParametersByPathRequest build() {
            return new GetParametersByPathRequest(this);
        }
    }
}
