/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.ssm.transform.InventoryResultEntityMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Inventory query results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InventoryResultEntity implements StructuredPojo,
        ToCopyableBuilder<InventoryResultEntity.Builder, InventoryResultEntity> {
    private final String id;

    private final Map<String, InventoryResultItem> data;

    private InventoryResultEntity(BuilderImpl builder) {
        this.id = builder.id;
        this.data = builder.data;
    }

    /**
     * <p>
     * ID of the inventory result entity. For example, for managed instance inventory the result will be the managed
     * instance ID. For EC2 instance inventory, the result will be the instance ID.
     * </p>
     * 
     * @return ID of the inventory result entity. For example, for managed instance inventory the result will be the
     *         managed instance ID. For EC2 instance inventory, the result will be the instance ID.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The data section in the inventory result entity json.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The data section in the inventory result entity json.
     */
    public Map<String, InventoryResultItem> data() {
        return data;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryResultEntity)) {
            return false;
        }
        InventoryResultEntity other = (InventoryResultEntity) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(data(), other.data());
    }

    @Override
    public String toString() {
        return ToString.builder("InventoryResultEntity").add("Id", id()).add("Data", data()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InventoryResultEntityMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InventoryResultEntity> {
        /**
         * <p>
         * ID of the inventory result entity. For example, for managed instance inventory the result will be the managed
         * instance ID. For EC2 instance inventory, the result will be the instance ID.
         * </p>
         * 
         * @param id
         *        ID of the inventory result entity. For example, for managed instance inventory the result will be the
         *        managed instance ID. For EC2 instance inventory, the result will be the instance ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The data section in the inventory result entity json.
         * </p>
         * 
         * @param data
         *        The data section in the inventory result entity json.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(Map<String, InventoryResultItem> data);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Map<String, InventoryResultItem> data = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryResultEntity model) {
            id(model.id);
            data(model.data);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final Map<String, InventoryResultItem.Builder> getData() {
            return data != null ? CollectionUtils.mapValues(data, InventoryResultItem::toBuilder) : null;
        }

        @Override
        public final Builder data(Map<String, InventoryResultItem> data) {
            this.data = InventoryResultItemMapCopier.copy(data);
            return this;
        }

        public final void setData(Map<String, InventoryResultItem.BuilderImpl> data) {
            this.data = InventoryResultItemMapCopier.copyFromBuilder(data);
        }

        @Override
        public InventoryResultEntity build() {
            return new InventoryResultEntity(this);
        }
    }
}
