/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.PatchBaselineIdentityMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the basic information about a patch baseline.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchBaselineIdentity implements StructuredPojo,
        ToCopyableBuilder<PatchBaselineIdentity.Builder, PatchBaselineIdentity> {
    private final String baselineId;

    private final String baselineName;

    private final String operatingSystem;

    private final String baselineDescription;

    private final Boolean defaultBaseline;

    private PatchBaselineIdentity(BuilderImpl builder) {
        this.baselineId = builder.baselineId;
        this.baselineName = builder.baselineName;
        this.operatingSystem = builder.operatingSystem;
        this.baselineDescription = builder.baselineDescription;
        this.defaultBaseline = builder.defaultBaseline;
    }

    /**
     * <p>
     * The ID of the patch baseline.
     * </p>
     * 
     * @return The ID of the patch baseline.
     */
    public String baselineId() {
        return baselineId;
    }

    /**
     * <p>
     * The name of the patch baseline.
     * </p>
     * 
     * @return The name of the patch baseline.
     */
    public String baselineName() {
        return baselineName;
    }

    /**
     * <p>
     * Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
     * @see OperatingSystem
     */
    public OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
     * @see OperatingSystem
     */
    public String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * The description of the patch baseline.
     * </p>
     * 
     * @return The description of the patch baseline.
     */
    public String baselineDescription() {
        return baselineDescription;
    }

    /**
     * <p>
     * Whether this is the default baseline. Note that Systems Manager supports creating multiple default patch
     * baselines. For example, you can create a default patch baseline for each operating system.
     * </p>
     * 
     * @return Whether this is the default baseline. Note that Systems Manager supports creating multiple default patch
     *         baselines. For example, you can create a default patch baseline for each operating system.
     */
    public Boolean defaultBaseline() {
        return defaultBaseline;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(baselineId());
        hashCode = 31 * hashCode + Objects.hashCode(baselineName());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(baselineDescription());
        hashCode = 31 * hashCode + Objects.hashCode(defaultBaseline());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchBaselineIdentity)) {
            return false;
        }
        PatchBaselineIdentity other = (PatchBaselineIdentity) obj;
        return Objects.equals(baselineId(), other.baselineId()) && Objects.equals(baselineName(), other.baselineName())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(baselineDescription(), other.baselineDescription())
                && Objects.equals(defaultBaseline(), other.defaultBaseline());
    }

    @Override
    public String toString() {
        return ToString.builder("PatchBaselineIdentity").add("BaselineId", baselineId()).add("BaselineName", baselineName())
                .add("OperatingSystem", operatingSystemAsString()).add("BaselineDescription", baselineDescription())
                .add("DefaultBaseline", defaultBaseline()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BaselineId":
            return Optional.ofNullable(clazz.cast(baselineId()));
        case "BaselineName":
            return Optional.ofNullable(clazz.cast(baselineName()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "BaselineDescription":
            return Optional.ofNullable(clazz.cast(baselineDescription()));
        case "DefaultBaseline":
            return Optional.ofNullable(clazz.cast(defaultBaseline()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PatchBaselineIdentityMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, PatchBaselineIdentity> {
        /**
         * <p>
         * The ID of the patch baseline.
         * </p>
         * 
         * @param baselineId
         *        The ID of the patch baseline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineId(String baselineId);

        /**
         * <p>
         * The name of the patch baseline.
         * </p>
         * 
         * @param baselineName
         *        The name of the patch baseline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineName(String baselineName);

        /**
         * <p>
         * Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
         * </p>
         * 
         * @param operatingSystem
         *        Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
         * </p>
         * 
         * @param operatingSystem
         *        Defines the operating system the patch baseline applies to. The Default value is WINDOWS.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * The description of the patch baseline.
         * </p>
         * 
         * @param baselineDescription
         *        The description of the patch baseline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineDescription(String baselineDescription);

        /**
         * <p>
         * Whether this is the default baseline. Note that Systems Manager supports creating multiple default patch
         * baselines. For example, you can create a default patch baseline for each operating system.
         * </p>
         * 
         * @param defaultBaseline
         *        Whether this is the default baseline. Note that Systems Manager supports creating multiple default
         *        patch baselines. For example, you can create a default patch baseline for each operating system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultBaseline(Boolean defaultBaseline);
    }

    static final class BuilderImpl implements Builder {
        private String baselineId;

        private String baselineName;

        private String operatingSystem;

        private String baselineDescription;

        private Boolean defaultBaseline;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchBaselineIdentity model) {
            baselineId(model.baselineId);
            baselineName(model.baselineName);
            operatingSystem(model.operatingSystem);
            baselineDescription(model.baselineDescription);
            defaultBaseline(model.defaultBaseline);
        }

        public final String getBaselineId() {
            return baselineId;
        }

        @Override
        public final Builder baselineId(String baselineId) {
            this.baselineId = baselineId;
            return this;
        }

        public final void setBaselineId(String baselineId) {
            this.baselineId = baselineId;
        }

        public final String getBaselineName() {
            return baselineName;
        }

        @Override
        public final Builder baselineName(String baselineName) {
            this.baselineName = baselineName;
            return this;
        }

        public final void setBaselineName(String baselineName) {
            this.baselineName = baselineName;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem.toString());
            return this;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        public final String getBaselineDescription() {
            return baselineDescription;
        }

        @Override
        public final Builder baselineDescription(String baselineDescription) {
            this.baselineDescription = baselineDescription;
            return this;
        }

        public final void setBaselineDescription(String baselineDescription) {
            this.baselineDescription = baselineDescription;
        }

        public final Boolean getDefaultBaseline() {
            return defaultBaseline;
        }

        @Override
        public final Builder defaultBaseline(Boolean defaultBaseline) {
            this.defaultBaseline = defaultBaseline;
            return this;
        }

        public final void setDefaultBaseline(Boolean defaultBaseline) {
            this.defaultBaseline = defaultBaseline;
        }

        @Override
        public PatchBaselineIdentity build() {
            return new PatchBaselineIdentity(this);
        }
    }
}
