/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Amazon EC2 Systems Manager parameter in Parameter Store.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Parameter implements SdkPojo, Serializable, ToCopyableBuilder<Parameter.Builder, Parameter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Parameter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Parameter::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Parameter::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(Parameter::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            VALUE_FIELD, VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final String value;

    private final Long version;

    private Parameter(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.value = builder.value;
        this.version = builder.version;
    }

    /**
     * <p>
     * The name of the parameter.
     * </p>
     * 
     * @return The name of the parameter.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The type of parameter. Valid values include the following: String, String list, Secure string.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. Valid values include the following: String, String list, Secure string.
     * @see ParameterType
     */
    public ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter. Valid values include the following: String, String list, Secure string.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. Valid values include the following: String, String list, Secure string.
     * @see ParameterType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The parameter value.
     * </p>
     * 
     * @return The parameter value.
     */
    public String value() {
        return value;
    }

    /**
     * <p>
     * The parameter version.
     * </p>
     * 
     * @return The parameter version.
     */
    public Long version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Parameter)) {
            return false;
        }
        Parameter other = (Parameter) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(value(), other.value()) && Objects.equals(version(), other.version());
    }

    @Override
    public String toString() {
        return ToString.builder("Parameter").add("Name", name()).add("Type", typeAsString()).add("Value", value())
                .add("Version", version()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Parameter, T> g) {
        return obj -> g.apply((Parameter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Parameter> {
        /**
         * <p>
         * The name of the parameter.
         * </p>
         * 
         * @param name
         *        The name of the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of parameter. Valid values include the following: String, String list, Secure string.
         * </p>
         * 
         * @param type
         *        The type of parameter. Valid values include the following: String, String list, Secure string.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter. Valid values include the following: String, String list, Secure string.
         * </p>
         * 
         * @param type
         *        The type of parameter. Valid values include the following: String, String list, Secure string.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The parameter value.
         * </p>
         * 
         * @param value
         *        The parameter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The parameter version.
         * </p>
         * 
         * @param version
         *        The parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private String value;

        private Long version;

        private BuilderImpl() {
        }

        private BuilderImpl(Parameter model) {
            name(model.name);
            type(model.type);
            value(model.value);
            version(model.version);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getValue() {
            return value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        public final Long getVersion() {
            return version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        @Override
        public Parameter build() {
            return new Parameter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
