/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a Resource Data Sync configuration, including its current status and last successful sync.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDataSyncItem implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceDataSyncItem.Builder, ResourceDataSyncItem> {
    private static final SdkField<String> SYNC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncItem::syncName)).setter(setter(Builder::syncName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncName").build()).build();

    private static final SdkField<ResourceDataSyncS3Destination> S3_DESTINATION_FIELD = SdkField
            .<ResourceDataSyncS3Destination> builder(MarshallingType.SDK_POJO)
            .getter(getter(ResourceDataSyncItem::s3Destination)).setter(setter(Builder::s3Destination))
            .constructor(ResourceDataSyncS3Destination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Destination").build()).build();

    private static final SdkField<Instant> LAST_SYNC_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ResourceDataSyncItem::lastSyncTime)).setter(setter(Builder::lastSyncTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSyncTime").build()).build();

    private static final SdkField<Instant> LAST_SUCCESSFUL_SYNC_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ResourceDataSyncItem::lastSuccessfulSyncTime)).setter(setter(Builder::lastSuccessfulSyncTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSuccessfulSyncTime").build())
            .build();

    private static final SdkField<String> LAST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncItem::lastStatusAsString)).setter(setter(Builder::lastStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastStatus").build()).build();

    private static final SdkField<Instant> SYNC_CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ResourceDataSyncItem::syncCreatedTime)).setter(setter(Builder::syncCreatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncCreatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SYNC_NAME_FIELD,
            S3_DESTINATION_FIELD, LAST_SYNC_TIME_FIELD, LAST_SUCCESSFUL_SYNC_TIME_FIELD, LAST_STATUS_FIELD,
            SYNC_CREATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String syncName;

    private final ResourceDataSyncS3Destination s3Destination;

    private final Instant lastSyncTime;

    private final Instant lastSuccessfulSyncTime;

    private final String lastStatus;

    private final Instant syncCreatedTime;

    private ResourceDataSyncItem(BuilderImpl builder) {
        this.syncName = builder.syncName;
        this.s3Destination = builder.s3Destination;
        this.lastSyncTime = builder.lastSyncTime;
        this.lastSuccessfulSyncTime = builder.lastSuccessfulSyncTime;
        this.lastStatus = builder.lastStatus;
        this.syncCreatedTime = builder.syncCreatedTime;
    }

    /**
     * <p>
     * The name of the Resource Data Sync.
     * </p>
     * 
     * @return The name of the Resource Data Sync.
     */
    public String syncName() {
        return syncName;
    }

    /**
     * <p>
     * Configuration information for the target Amazon S3 bucket.
     * </p>
     * 
     * @return Configuration information for the target Amazon S3 bucket.
     */
    public ResourceDataSyncS3Destination s3Destination() {
        return s3Destination;
    }

    /**
     * <p>
     * The last time the configuration attempted to sync (UTC).
     * </p>
     * 
     * @return The last time the configuration attempted to sync (UTC).
     */
    public Instant lastSyncTime() {
        return lastSyncTime;
    }

    /**
     * <p>
     * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     * </p>
     * 
     * @return The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     */
    public Instant lastSuccessfulSyncTime() {
        return lastSuccessfulSyncTime;
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusAsString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public LastResourceDataSyncStatus lastStatus() {
        return LastResourceDataSyncStatus.fromValue(lastStatus);
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusAsString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public String lastStatusAsString() {
        return lastStatus;
    }

    /**
     * <p>
     * The date and time the configuration was created (UTC).
     * </p>
     * 
     * @return The date and time the configuration was created (UTC).
     */
    public Instant syncCreatedTime() {
        return syncCreatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(syncName());
        hashCode = 31 * hashCode + Objects.hashCode(s3Destination());
        hashCode = 31 * hashCode + Objects.hashCode(lastSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(syncCreatedTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncItem)) {
            return false;
        }
        ResourceDataSyncItem other = (ResourceDataSyncItem) obj;
        return Objects.equals(syncName(), other.syncName()) && Objects.equals(s3Destination(), other.s3Destination())
                && Objects.equals(lastSyncTime(), other.lastSyncTime())
                && Objects.equals(lastSuccessfulSyncTime(), other.lastSuccessfulSyncTime())
                && Objects.equals(lastStatusAsString(), other.lastStatusAsString())
                && Objects.equals(syncCreatedTime(), other.syncCreatedTime());
    }

    @Override
    public String toString() {
        return ToString.builder("ResourceDataSyncItem").add("SyncName", syncName()).add("S3Destination", s3Destination())
                .add("LastSyncTime", lastSyncTime()).add("LastSuccessfulSyncTime", lastSuccessfulSyncTime())
                .add("LastStatus", lastStatusAsString()).add("SyncCreatedTime", syncCreatedTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SyncName":
            return Optional.ofNullable(clazz.cast(syncName()));
        case "S3Destination":
            return Optional.ofNullable(clazz.cast(s3Destination()));
        case "LastSyncTime":
            return Optional.ofNullable(clazz.cast(lastSyncTime()));
        case "LastSuccessfulSyncTime":
            return Optional.ofNullable(clazz.cast(lastSuccessfulSyncTime()));
        case "LastStatus":
            return Optional.ofNullable(clazz.cast(lastStatusAsString()));
        case "SyncCreatedTime":
            return Optional.ofNullable(clazz.cast(syncCreatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceDataSyncItem, T> g) {
        return obj -> g.apply((ResourceDataSyncItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDataSyncItem> {
        /**
         * <p>
         * The name of the Resource Data Sync.
         * </p>
         * 
         * @param syncName
         *        The name of the Resource Data Sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncName(String syncName);

        /**
         * <p>
         * Configuration information for the target Amazon S3 bucket.
         * </p>
         * 
         * @param s3Destination
         *        Configuration information for the target Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Destination(ResourceDataSyncS3Destination s3Destination);

        /**
         * <p>
         * Configuration information for the target Amazon S3 bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceDataSyncS3Destination.Builder} avoiding
         * the need to create one manually via {@link ResourceDataSyncS3Destination#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceDataSyncS3Destination.Builder#build()} is called
         * immediately and its result is passed to {@link #s3Destination(ResourceDataSyncS3Destination)}.
         * 
         * @param s3Destination
         *        a consumer that will call methods on {@link ResourceDataSyncS3Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Destination(ResourceDataSyncS3Destination)
         */
        default Builder s3Destination(Consumer<ResourceDataSyncS3Destination.Builder> s3Destination) {
            return s3Destination(ResourceDataSyncS3Destination.builder().applyMutation(s3Destination).build());
        }

        /**
         * <p>
         * The last time the configuration attempted to sync (UTC).
         * </p>
         * 
         * @param lastSyncTime
         *        The last time the configuration attempted to sync (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSyncTime(Instant lastSyncTime);

        /**
         * <p>
         * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * </p>
         * 
         * @param lastSuccessfulSyncTime
         *        The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(String lastStatus);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(LastResourceDataSyncStatus lastStatus);

        /**
         * <p>
         * The date and time the configuration was created (UTC).
         * </p>
         * 
         * @param syncCreatedTime
         *        The date and time the configuration was created (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncCreatedTime(Instant syncCreatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String syncName;

        private ResourceDataSyncS3Destination s3Destination;

        private Instant lastSyncTime;

        private Instant lastSuccessfulSyncTime;

        private String lastStatus;

        private Instant syncCreatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncItem model) {
            syncName(model.syncName);
            s3Destination(model.s3Destination);
            lastSyncTime(model.lastSyncTime);
            lastSuccessfulSyncTime(model.lastSuccessfulSyncTime);
            lastStatus(model.lastStatus);
            syncCreatedTime(model.syncCreatedTime);
        }

        public final String getSyncName() {
            return syncName;
        }

        @Override
        public final Builder syncName(String syncName) {
            this.syncName = syncName;
            return this;
        }

        public final void setSyncName(String syncName) {
            this.syncName = syncName;
        }

        public final ResourceDataSyncS3Destination.Builder getS3Destination() {
            return s3Destination != null ? s3Destination.toBuilder() : null;
        }

        @Override
        public final Builder s3Destination(ResourceDataSyncS3Destination s3Destination) {
            this.s3Destination = s3Destination;
            return this;
        }

        public final void setS3Destination(ResourceDataSyncS3Destination.BuilderImpl s3Destination) {
            this.s3Destination = s3Destination != null ? s3Destination.build() : null;
        }

        public final Instant getLastSyncTime() {
            return lastSyncTime;
        }

        @Override
        public final Builder lastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
            return this;
        }

        public final void setLastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
        }

        public final Instant getLastSuccessfulSyncTime() {
            return lastSuccessfulSyncTime;
        }

        @Override
        public final Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
            return this;
        }

        public final void setLastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
        }

        public final String getLastStatus() {
            return lastStatus;
        }

        @Override
        public final Builder lastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
            return this;
        }

        @Override
        public final Builder lastStatus(LastResourceDataSyncStatus lastStatus) {
            this.lastStatus(lastStatus.toString());
            return this;
        }

        public final void setLastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
        }

        public final Instant getSyncCreatedTime() {
            return syncCreatedTime;
        }

        @Override
        public final Builder syncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
            return this;
        }

        public final void setSyncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
        }

        @Override
        public ResourceDataSyncItem build() {
            return new ResourceDataSyncItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
