/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.InventoryItemSchemaMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inventory item schema definition. Users can use this to compose inventory query filters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class InventoryItemSchema implements StructuredPojo, ToCopyableBuilder<InventoryItemSchema.Builder, InventoryItemSchema> {
    private final String typeName;

    private final String version;

    private final List<InventoryItemAttribute> attributes;

    private InventoryItemSchema(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.version = builder.version;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
     * will start with Custom. Default inventory item types include the following: AWS:AWSComponent, AWS:Application,
     * AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     * </p>
     * 
     * @return The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type
     *         names will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
     *         AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     */
    public String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory item.
     * </p>
     * 
     * @return The schema version for the inventory item.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The schema attributes for inventory. This contains data type and attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The schema attributes for inventory. This contains data type and attribute name.
     */
    public List<InventoryItemAttribute> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((typeName() == null) ? 0 : typeName().hashCode());
        hashCode = 31 * hashCode + ((version() == null) ? 0 : version().hashCode());
        hashCode = 31 * hashCode + ((attributes() == null) ? 0 : attributes().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryItemSchema)) {
            return false;
        }
        InventoryItemSchema other = (InventoryItemSchema) obj;
        if (other.typeName() == null ^ this.typeName() == null) {
            return false;
        }
        if (other.typeName() != null && !other.typeName().equals(this.typeName())) {
            return false;
        }
        if (other.version() == null ^ this.version() == null) {
            return false;
        }
        if (other.version() != null && !other.version().equals(this.version())) {
            return false;
        }
        if (other.attributes() == null ^ this.attributes() == null) {
            return false;
        }
        if (other.attributes() != null && !other.attributes().equals(this.attributes())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (typeName() != null) {
            sb.append("TypeName: ").append(typeName()).append(",");
        }
        if (version() != null) {
            sb.append("Version: ").append(version()).append(",");
        }
        if (attributes() != null) {
            sb.append("Attributes: ").append(attributes()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InventoryItemSchemaMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InventoryItemSchema> {
        /**
         * <p>
         * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
         * will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
         * AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory type. Default inventory item type names start with AWS. Custom inventory
         *        type names will start with Custom. Default inventory item types include the following:
         *        AWS:AWSComponent, AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory item.
         * </p>
         * 
         * @param version
         *        The schema version for the inventory item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<InventoryItemAttribute> attributes);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(InventoryItemAttribute... attributes);
    }

    private static final class BuilderImpl implements Builder {
        private String typeName;

        private String version;

        private List<InventoryItemAttribute> attributes;

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryItemSchema model) {
            setTypeName(model.typeName);
            setVersion(model.version);
            setAttributes(model.attributes);
        }

        public final String getTypeName() {
            return typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final Collection<InventoryItemAttribute> getAttributes() {
            return attributes;
        }

        @Override
        public final Builder attributes(Collection<InventoryItemAttribute> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(InventoryItemAttribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        public final void setAttributes(Collection<InventoryItemAttribute> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copy(attributes);
        }

        @SafeVarargs
        public final void setAttributes(InventoryItemAttribute... attributes) {
            attributes(Arrays.asList(attributes));
        }

        @Override
        public InventoryItemSchema build() {
            return new InventoryItemSchema(this);
        }
    }
}
