/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.AssociationMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an association of a Systems Manager document and an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Association implements StructuredPojo, ToCopyableBuilder<Association.Builder, Association> {
    private final String name;

    private final String instanceId;

    private final String associationId;

    private final String documentVersion;

    private final List<Target> targets;

    private final Instant lastExecutionDate;

    private final AssociationOverview overview;

    private final String scheduleExpression;

    private Association(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationId = builder.associationId;
        this.documentVersion = builder.documentVersion;
        this.targets = builder.targets;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.overview = builder.overview;
        this.scheduleExpression = builder.scheduleExpression;
    }

    /**
     * <p>
     * The name of the SSM document.
     * </p>
     * 
     * @return The name of the SSM document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The ID created by the system when you create an association. An association is a binding between a document and a
     * set of targets with a schedule.
     * </p>
     * 
     * @return The ID created by the system when you create an association. An association is a binding between a
     *         document and a set of targets with a schedule.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The version of the document used in the association.
     * </p>
     * 
     * @return The version of the document used in the association.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The instances targeted by the request to create an association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request to create an association.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((instanceId() == null) ? 0 : instanceId().hashCode());
        hashCode = 31 * hashCode + ((associationId() == null) ? 0 : associationId().hashCode());
        hashCode = 31 * hashCode + ((documentVersion() == null) ? 0 : documentVersion().hashCode());
        hashCode = 31 * hashCode + ((targets() == null) ? 0 : targets().hashCode());
        hashCode = 31 * hashCode + ((lastExecutionDate() == null) ? 0 : lastExecutionDate().hashCode());
        hashCode = 31 * hashCode + ((overview() == null) ? 0 : overview().hashCode());
        hashCode = 31 * hashCode + ((scheduleExpression() == null) ? 0 : scheduleExpression().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Association)) {
            return false;
        }
        Association other = (Association) obj;
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.instanceId() == null ^ this.instanceId() == null) {
            return false;
        }
        if (other.instanceId() != null && !other.instanceId().equals(this.instanceId())) {
            return false;
        }
        if (other.associationId() == null ^ this.associationId() == null) {
            return false;
        }
        if (other.associationId() != null && !other.associationId().equals(this.associationId())) {
            return false;
        }
        if (other.documentVersion() == null ^ this.documentVersion() == null) {
            return false;
        }
        if (other.documentVersion() != null && !other.documentVersion().equals(this.documentVersion())) {
            return false;
        }
        if (other.targets() == null ^ this.targets() == null) {
            return false;
        }
        if (other.targets() != null && !other.targets().equals(this.targets())) {
            return false;
        }
        if (other.lastExecutionDate() == null ^ this.lastExecutionDate() == null) {
            return false;
        }
        if (other.lastExecutionDate() != null && !other.lastExecutionDate().equals(this.lastExecutionDate())) {
            return false;
        }
        if (other.overview() == null ^ this.overview() == null) {
            return false;
        }
        if (other.overview() != null && !other.overview().equals(this.overview())) {
            return false;
        }
        if (other.scheduleExpression() == null ^ this.scheduleExpression() == null) {
            return false;
        }
        if (other.scheduleExpression() != null && !other.scheduleExpression().equals(this.scheduleExpression())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (instanceId() != null) {
            sb.append("InstanceId: ").append(instanceId()).append(",");
        }
        if (associationId() != null) {
            sb.append("AssociationId: ").append(associationId()).append(",");
        }
        if (documentVersion() != null) {
            sb.append("DocumentVersion: ").append(documentVersion()).append(",");
        }
        if (targets() != null) {
            sb.append("Targets: ").append(targets()).append(",");
        }
        if (lastExecutionDate() != null) {
            sb.append("LastExecutionDate: ").append(lastExecutionDate()).append(",");
        }
        if (overview() != null) {
            sb.append("Overview: ").append(overview()).append(",");
        }
        if (scheduleExpression() != null) {
            sb.append("ScheduleExpression: ").append(scheduleExpression()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AssociationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Association> {
        /**
         * <p>
         * The name of the SSM document.
         * </p>
         * 
         * @param name
         *        The name of the SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The ID created by the system when you create an association. An association is a binding between a document
         * and a set of targets with a schedule.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when you create an association. An association is a binding between a
         *        document and a set of targets with a schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The version of the document used in the association.
         * </p>
         * 
         * @param documentVersion
         *        The version of the document used in the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);
    }

    private static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationId;

        private String documentVersion;

        private List<Target> targets;

        private Instant lastExecutionDate;

        private AssociationOverview overview;

        private String scheduleExpression;

        private BuilderImpl() {
        }

        private BuilderImpl(Association model) {
            setName(model.name);
            setInstanceId(model.instanceId);
            setAssociationId(model.associationId);
            setDocumentVersion(model.documentVersion);
            setTargets(model.targets);
            setLastExecutionDate(model.lastExecutionDate);
            setOverview(model.overview);
            setScheduleExpression(model.scheduleExpression);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Collection<Target> getTargets() {
            return targets;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        public final AssociationOverview getOverview() {
            return overview;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final void setOverview(AssociationOverview overview) {
            this.overview = overview;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        @Override
        public Association build() {
            return new Association(this);
        }
    }
}
