/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.PatchComplianceDataMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the state of a patch on a particular instance as it relates to the patch baseline used to patch the
 * instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class PatchComplianceData implements StructuredPojo, ToCopyableBuilder<PatchComplianceData.Builder, PatchComplianceData> {
    private final String title;

    private final String kbId;

    private final String classification;

    private final String severity;

    private final String state;

    private final Instant installedTime;

    private PatchComplianceData(BuilderImpl builder) {
        this.title = builder.title;
        this.kbId = builder.kbId;
        this.classification = builder.classification;
        this.severity = builder.severity;
        this.state = builder.state;
        this.installedTime = builder.installedTime;
    }

    /**
     * <p>
     * The title of the patch.
     * </p>
     * 
     * @return The title of the patch.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The Microsoft Knowledge Base ID of the patch.
     * </p>
     * 
     * @return The Microsoft Knowledge Base ID of the patch.
     */
    public String kbId() {
        return kbId;
    }

    /**
     * <p>
     * The classification of the patch (for example, SecurityUpdates, Updates, CriticalUpdates).
     * </p>
     * 
     * @return The classification of the patch (for example, SecurityUpdates, Updates, CriticalUpdates).
     */
    public String classification() {
        return classification;
    }

    /**
     * <p>
     * The severity of the patch (for example, Critical, Important, Moderate).
     * </p>
     * 
     * @return The severity of the patch (for example, Critical, Important, Moderate).
     */
    public String severity() {
        return severity;
    }

    /**
     * <p>
     * The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or FAILED).
     * </p>
     * 
     * @return The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or FAILED).
     * @see PatchComplianceDataState
     */
    public String state() {
        return state;
    }

    /**
     * <p>
     * The date/time the patch was installed on the instance.
     * </p>
     * 
     * @return The date/time the patch was installed on the instance.
     */
    public Instant installedTime() {
        return installedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((title() == null) ? 0 : title().hashCode());
        hashCode = 31 * hashCode + ((kbId() == null) ? 0 : kbId().hashCode());
        hashCode = 31 * hashCode + ((classification() == null) ? 0 : classification().hashCode());
        hashCode = 31 * hashCode + ((severity() == null) ? 0 : severity().hashCode());
        hashCode = 31 * hashCode + ((state() == null) ? 0 : state().hashCode());
        hashCode = 31 * hashCode + ((installedTime() == null) ? 0 : installedTime().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchComplianceData)) {
            return false;
        }
        PatchComplianceData other = (PatchComplianceData) obj;
        if (other.title() == null ^ this.title() == null) {
            return false;
        }
        if (other.title() != null && !other.title().equals(this.title())) {
            return false;
        }
        if (other.kbId() == null ^ this.kbId() == null) {
            return false;
        }
        if (other.kbId() != null && !other.kbId().equals(this.kbId())) {
            return false;
        }
        if (other.classification() == null ^ this.classification() == null) {
            return false;
        }
        if (other.classification() != null && !other.classification().equals(this.classification())) {
            return false;
        }
        if (other.severity() == null ^ this.severity() == null) {
            return false;
        }
        if (other.severity() != null && !other.severity().equals(this.severity())) {
            return false;
        }
        if (other.state() == null ^ this.state() == null) {
            return false;
        }
        if (other.state() != null && !other.state().equals(this.state())) {
            return false;
        }
        if (other.installedTime() == null ^ this.installedTime() == null) {
            return false;
        }
        if (other.installedTime() != null && !other.installedTime().equals(this.installedTime())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (title() != null) {
            sb.append("Title: ").append(title()).append(",");
        }
        if (kbId() != null) {
            sb.append("KBId: ").append(kbId()).append(",");
        }
        if (classification() != null) {
            sb.append("Classification: ").append(classification()).append(",");
        }
        if (severity() != null) {
            sb.append("Severity: ").append(severity()).append(",");
        }
        if (state() != null) {
            sb.append("State: ").append(state()).append(",");
        }
        if (installedTime() != null) {
            sb.append("InstalledTime: ").append(installedTime()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PatchComplianceDataMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, PatchComplianceData> {
        /**
         * <p>
         * The title of the patch.
         * </p>
         * 
         * @param title
         *        The title of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The Microsoft Knowledge Base ID of the patch.
         * </p>
         * 
         * @param kbId
         *        The Microsoft Knowledge Base ID of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kbId(String kbId);

        /**
         * <p>
         * The classification of the patch (for example, SecurityUpdates, Updates, CriticalUpdates).
         * </p>
         * 
         * @param classification
         *        The classification of the patch (for example, SecurityUpdates, Updates, CriticalUpdates).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classification(String classification);

        /**
         * <p>
         * The severity of the patch (for example, Critical, Important, Moderate).
         * </p>
         * 
         * @param severity
         *        The severity of the patch (for example, Critical, Important, Moderate).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or FAILED).
         * </p>
         * 
         * @param state
         *        The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or
         *        FAILED).
         * @see PatchComplianceDataState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceDataState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or FAILED).
         * </p>
         * 
         * @param state
         *        The state of the patch on the instance (INSTALLED, INSTALLED_OTHER, MISSING, NOT_APPLICABLE or
         *        FAILED).
         * @see PatchComplianceDataState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceDataState
         */
        Builder state(PatchComplianceDataState state);

        /**
         * <p>
         * The date/time the patch was installed on the instance.
         * </p>
         * 
         * @param installedTime
         *        The date/time the patch was installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedTime(Instant installedTime);
    }

    private static final class BuilderImpl implements Builder {
        private String title;

        private String kbId;

        private String classification;

        private String severity;

        private String state;

        private Instant installedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchComplianceData model) {
            setTitle(model.title);
            setKBId(model.kbId);
            setClassification(model.classification);
            setSeverity(model.severity);
            setState(model.state);
            setInstalledTime(model.installedTime);
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getKBId() {
            return kbId;
        }

        @Override
        public final Builder kbId(String kbId) {
            this.kbId = kbId;
            return this;
        }

        public final void setKBId(String kbId) {
            this.kbId = kbId;
        }

        public final String getClassification() {
            return classification;
        }

        @Override
        public final Builder classification(String classification) {
            this.classification = classification;
            return this;
        }

        public final void setClassification(String classification) {
            this.classification = classification;
        }

        public final String getSeverity() {
            return severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(PatchComplianceDataState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final Instant getInstalledTime() {
            return installedTime;
        }

        @Override
        public final Builder installedTime(Instant installedTime) {
            this.installedTime = installedTime;
            return this;
        }

        public final void setInstalledTime(Instant installedTime) {
            this.installedTime = installedTime;
        }

        @Override
        public PatchComplianceData build() {
            return new PatchComplianceData(this);
        }
    }
}
