/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.DocumentParameterMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameters specified in a System Manager document that execute on the server when the command is run.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DocumentParameter implements StructuredPojo, ToCopyableBuilder<DocumentParameter.Builder, DocumentParameter> {
    private final String name;

    private final String type;

    private final String description;

    private final String defaultValue;

    private DocumentParameter(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.description = builder.description;
        this.defaultValue = builder.defaultValue;
    }

    /**
     * <p>
     * The name of the parameter.
     * </p>
     * 
     * @return The name of the parameter.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The type of parameter. The type can be either String or StringList.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DocumentParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of parameter. The type can be either String or StringList.
     * @see DocumentParameterType
     */
    public DocumentParameterType type() {
        return DocumentParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter. The type can be either String or StringList.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DocumentParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of parameter. The type can be either String or StringList.
     * @see DocumentParameterType
     */
    public String typeString() {
        return type;
    }

    /**
     * <p>
     * A description of what the parameter does, how to use it, the default value, and whether or not the parameter is
     * optional.
     * </p>
     * 
     * @return A description of what the parameter does, how to use it, the default value, and whether or not the
     *         parameter is optional.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * If specified, the default values for the parameters. Parameters without a default value are required. Parameters
     * with a default value are optional.
     * </p>
     * 
     * @return If specified, the default values for the parameters. Parameters without a default value are required.
     *         Parameters with a default value are optional.
     */
    public String defaultValue() {
        return defaultValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((typeString() == null) ? 0 : typeString().hashCode());
        hashCode = 31 * hashCode + ((description() == null) ? 0 : description().hashCode());
        hashCode = 31 * hashCode + ((defaultValue() == null) ? 0 : defaultValue().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentParameter)) {
            return false;
        }
        DocumentParameter other = (DocumentParameter) obj;
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.typeString() == null ^ this.typeString() == null) {
            return false;
        }
        if (other.typeString() != null && !other.typeString().equals(this.typeString())) {
            return false;
        }
        if (other.description() == null ^ this.description() == null) {
            return false;
        }
        if (other.description() != null && !other.description().equals(this.description())) {
            return false;
        }
        if (other.defaultValue() == null ^ this.defaultValue() == null) {
            return false;
        }
        if (other.defaultValue() != null && !other.defaultValue().equals(this.defaultValue())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (typeString() != null) {
            sb.append("Type: ").append(typeString()).append(",");
        }
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (defaultValue() != null) {
            sb.append("DefaultValue: ").append(defaultValue()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Type":
            return Optional.of(clazz.cast(typeString()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "DefaultValue":
            return Optional.of(clazz.cast(defaultValue()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DocumentParameterMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DocumentParameter> {
        /**
         * <p>
         * The name of the parameter.
         * </p>
         * 
         * @param name
         *        The name of the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of parameter. The type can be either String or StringList.
         * </p>
         * 
         * @param type
         *        The type of parameter. The type can be either String or StringList.
         * @see DocumentParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter. The type can be either String or StringList.
         * </p>
         * 
         * @param type
         *        The type of parameter. The type can be either String or StringList.
         * @see DocumentParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentParameterType
         */
        Builder type(DocumentParameterType type);

        /**
         * <p>
         * A description of what the parameter does, how to use it, the default value, and whether or not the parameter
         * is optional.
         * </p>
         * 
         * @param description
         *        A description of what the parameter does, how to use it, the default value, and whether or not the
         *        parameter is optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * If specified, the default values for the parameters. Parameters without a default value are required.
         * Parameters with a default value are optional.
         * </p>
         * 
         * @param defaultValue
         *        If specified, the default values for the parameters. Parameters without a default value are required.
         *        Parameters with a default value are optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(String defaultValue);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private String description;

        private String defaultValue;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentParameter model) {
            name(model.name);
            type(model.type);
            description(model.description);
            defaultValue(model.defaultValue);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DocumentParameterType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public DocumentParameter build() {
            return new DocumentParameter(this);
        }
    }
}
