/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.AssociationMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an association of a Systems Manager document and an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Association implements StructuredPojo, ToCopyableBuilder<Association.Builder, Association> {
    private final String name;

    private final String instanceId;

    private final String associationId;

    private final String associationVersion;

    private final String documentVersion;

    private final List<Target> targets;

    private final Instant lastExecutionDate;

    private final AssociationOverview overview;

    private final String scheduleExpression;

    private final String associationName;

    private Association(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationId = builder.associationId;
        this.associationVersion = builder.associationVersion;
        this.documentVersion = builder.documentVersion;
        this.targets = builder.targets;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.overview = builder.overview;
        this.scheduleExpression = builder.scheduleExpression;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The ID created by the system when you create an association. An association is a binding between a document and a
     * set of targets with a schedule.
     * </p>
     * 
     * @return The ID created by the system when you create an association. An association is a binding between a
     *         document and a set of targets with a schedule.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The version of the document used in the association.
     * </p>
     * 
     * @return The version of the document used in the association.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The instances targeted by the request to create an association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request to create an association.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The association name.
     * </p>
     * 
     * @return The association name.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(lastExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(overview());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Association)) {
            return false;
        }
        Association other = (Association) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(associationId(), other.associationId())
                && Objects.equals(associationVersion(), other.associationVersion())
                && Objects.equals(documentVersion(), other.documentVersion()) && Objects.equals(targets(), other.targets())
                && Objects.equals(lastExecutionDate(), other.lastExecutionDate()) && Objects.equals(overview(), other.overview())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (instanceId() != null) {
            sb.append("InstanceId: ").append(instanceId()).append(",");
        }
        if (associationId() != null) {
            sb.append("AssociationId: ").append(associationId()).append(",");
        }
        if (associationVersion() != null) {
            sb.append("AssociationVersion: ").append(associationVersion()).append(",");
        }
        if (documentVersion() != null) {
            sb.append("DocumentVersion: ").append(documentVersion()).append(",");
        }
        if (targets() != null) {
            sb.append("Targets: ").append(targets()).append(",");
        }
        if (lastExecutionDate() != null) {
            sb.append("LastExecutionDate: ").append(lastExecutionDate()).append(",");
        }
        if (overview() != null) {
            sb.append("Overview: ").append(overview()).append(",");
        }
        if (scheduleExpression() != null) {
            sb.append("ScheduleExpression: ").append(scheduleExpression()).append(",");
        }
        if (associationName() != null) {
            sb.append("AssociationName: ").append(associationName()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "InstanceId":
            return Optional.of(clazz.cast(instanceId()));
        case "AssociationId":
            return Optional.of(clazz.cast(associationId()));
        case "AssociationVersion":
            return Optional.of(clazz.cast(associationVersion()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "LastExecutionDate":
            return Optional.of(clazz.cast(lastExecutionDate()));
        case "Overview":
            return Optional.of(clazz.cast(overview()));
        case "ScheduleExpression":
            return Optional.of(clazz.cast(scheduleExpression()));
        case "AssociationName":
            return Optional.of(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AssociationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Association> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The ID created by the system when you create an association. An association is a binding between a document
         * and a set of targets with a schedule.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when you create an association. An association is a binding between a
         *        document and a set of targets with a schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The version of the document used in the association.
         * </p>
         * 
         * @param documentVersion
         *        The version of the document used in the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * Information about the association.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationOverview.Builder} avoiding the need
         * to create one manually via {@link AssociationOverview#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationOverview.Builder#build()} is called immediately and
         * its result is passed to {@link #overview(AssociationOverview)}.
         * 
         * @param overview
         *        a consumer that will call methods on {@link AssociationOverview.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overview(AssociationOverview)
         */
        default Builder overview(Consumer<AssociationOverview.Builder> overview) {
            return overview(AssociationOverview.builder().apply(overview).build());
        }

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The association name.
         * </p>
         * 
         * @param associationName
         *        The association name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationId;

        private String associationVersion;

        private String documentVersion;

        private List<Target> targets;

        private Instant lastExecutionDate;

        private AssociationOverview overview;

        private String scheduleExpression;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(Association model) {
            name(model.name);
            instanceId(model.instanceId);
            associationId(model.associationId);
            associationVersion(model.associationVersion);
            documentVersion(model.documentVersion);
            targets(model.targets);
            lastExecutionDate(model.lastExecutionDate);
            overview(model.overview);
            scheduleExpression(model.scheduleExpression);
            associationName(model.associationName);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        public final AssociationOverview.Builder getOverview() {
            return overview != null ? overview.toBuilder() : null;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final void setOverview(AssociationOverview.BuilderImpl overview) {
            this.overview = overview != null ? overview.build() : null;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public Association build() {
            return new Association(this);
        }
    }
}
