/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.AssociationDescriptionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the parameters for a document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class AssociationDescription implements StructuredPojo,
        ToCopyableBuilder<AssociationDescription.Builder, AssociationDescription> {
    private final String name;

    private final String instanceId;

    private final String associationVersion;

    private final Instant dateValue;

    private final Instant lastUpdateAssociationDate;

    private final AssociationStatus status;

    private final AssociationOverview overview;

    private final String documentVersion;

    private final Map<String, List<String>> parameters;

    private final String associationId;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final Instant lastExecutionDate;

    private final Instant lastSuccessfulExecutionDate;

    private final String associationName;

    private AssociationDescription(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationVersion = builder.associationVersion;
        this.dateValue = builder.dateValue;
        this.lastUpdateAssociationDate = builder.lastUpdateAssociationDate;
        this.status = builder.status;
        this.overview = builder.overview;
        this.documentVersion = builder.documentVersion;
        this.parameters = builder.parameters;
        this.associationId = builder.associationId;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.lastSuccessfulExecutionDate = builder.lastSuccessfulExecutionDate;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The date when the association was made.
     * </p>
     * 
     * @return The date when the association was made.
     */
    public Instant date() {
        return dateValue;
    }

    /**
     * <p>
     * The date when the association was last updated.
     * </p>
     * 
     * @return The date when the association was last updated.
     */
    public Instant lastUpdateAssociationDate() {
        return lastUpdateAssociationDate;
    }

    /**
     * <p>
     * The association status.
     * </p>
     * 
     * @return The association status.
     */
    public AssociationStatus status() {
        return status;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * A description of the parameters for a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A description of the parameters for a document.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The association ID.
     * </p>
     * 
     * @return The association ID.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The instances targeted by the request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An Amazon S3 bucket where you want to store the output details of the request.
     * </p>
     * 
     * @return An Amazon S3 bucket where you want to store the output details of the request.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * The last date on which the association was successfully run.
     * </p>
     * 
     * @return The last date on which the association was successfully run.
     */
    public Instant lastSuccessfulExecutionDate() {
        return lastSuccessfulExecutionDate;
    }

    /**
     * <p>
     * The association name.
     * </p>
     * 
     * @return The association name.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateAssociationDate());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(overview());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(lastExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociationDescription)) {
            return false;
        }
        AssociationDescription other = (AssociationDescription) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(associationVersion(), other.associationVersion()) && Objects.equals(date(), other.date())
                && Objects.equals(lastUpdateAssociationDate(), other.lastUpdateAssociationDate())
                && Objects.equals(status(), other.status()) && Objects.equals(overview(), other.overview())
                && Objects.equals(documentVersion(), other.documentVersion()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(associationId(), other.associationId()) && Objects.equals(targets(), other.targets())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(lastExecutionDate(), other.lastExecutionDate())
                && Objects.equals(lastSuccessfulExecutionDate(), other.lastSuccessfulExecutionDate())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (instanceId() != null) {
            sb.append("InstanceId: ").append(instanceId()).append(",");
        }
        if (associationVersion() != null) {
            sb.append("AssociationVersion: ").append(associationVersion()).append(",");
        }
        if (date() != null) {
            sb.append("Date: ").append(date()).append(",");
        }
        if (lastUpdateAssociationDate() != null) {
            sb.append("LastUpdateAssociationDate: ").append(lastUpdateAssociationDate()).append(",");
        }
        if (status() != null) {
            sb.append("Status: ").append(status()).append(",");
        }
        if (overview() != null) {
            sb.append("Overview: ").append(overview()).append(",");
        }
        if (documentVersion() != null) {
            sb.append("DocumentVersion: ").append(documentVersion()).append(",");
        }
        if (parameters() != null) {
            sb.append("Parameters: ").append(parameters()).append(",");
        }
        if (associationId() != null) {
            sb.append("AssociationId: ").append(associationId()).append(",");
        }
        if (targets() != null) {
            sb.append("Targets: ").append(targets()).append(",");
        }
        if (scheduleExpression() != null) {
            sb.append("ScheduleExpression: ").append(scheduleExpression()).append(",");
        }
        if (outputLocation() != null) {
            sb.append("OutputLocation: ").append(outputLocation()).append(",");
        }
        if (lastExecutionDate() != null) {
            sb.append("LastExecutionDate: ").append(lastExecutionDate()).append(",");
        }
        if (lastSuccessfulExecutionDate() != null) {
            sb.append("LastSuccessfulExecutionDate: ").append(lastSuccessfulExecutionDate()).append(",");
        }
        if (associationName() != null) {
            sb.append("AssociationName: ").append(associationName()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "InstanceId":
            return Optional.of(clazz.cast(instanceId()));
        case "AssociationVersion":
            return Optional.of(clazz.cast(associationVersion()));
        case "Date":
            return Optional.of(clazz.cast(date()));
        case "LastUpdateAssociationDate":
            return Optional.of(clazz.cast(lastUpdateAssociationDate()));
        case "Status":
            return Optional.of(clazz.cast(status()));
        case "Overview":
            return Optional.of(clazz.cast(overview()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "AssociationId":
            return Optional.of(clazz.cast(associationId()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.of(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.of(clazz.cast(outputLocation()));
        case "LastExecutionDate":
            return Optional.of(clazz.cast(lastExecutionDate()));
        case "LastSuccessfulExecutionDate":
            return Optional.of(clazz.cast(lastSuccessfulExecutionDate()));
        case "AssociationName":
            return Optional.of(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AssociationDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AssociationDescription> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The date when the association was made.
         * </p>
         * 
         * @param dateValue
         *        The date when the association was made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(Instant dateValue);

        /**
         * <p>
         * The date when the association was last updated.
         * </p>
         * 
         * @param lastUpdateAssociationDate
         *        The date when the association was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateAssociationDate(Instant lastUpdateAssociationDate);

        /**
         * <p>
         * The association status.
         * </p>
         * 
         * @param status
         *        The association status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(AssociationStatus status);

        /**
         * <p>
         * The association status.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationStatus.Builder} avoiding the need to
         * create one manually via {@link AssociationStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(AssociationStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link AssociationStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(AssociationStatus)
         */
        default Builder status(Consumer<AssociationStatus.Builder> status) {
            return status(AssociationStatus.builder().apply(status).build());
        }

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * Information about the association.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationOverview.Builder} avoiding the need
         * to create one manually via {@link AssociationOverview#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationOverview.Builder#build()} is called immediately and
         * its result is passed to {@link #overview(AssociationOverview)}.
         * 
         * @param overview
         *        a consumer that will call methods on {@link AssociationOverview.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overview(AssociationOverview)
         */
        default Builder overview(Consumer<AssociationOverview.Builder> overview) {
            return overview(AssociationOverview.builder().apply(overview).build());
        }

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The association ID.
         * </p>
         * 
         * @param associationId
         *        The association ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * 
         * @param outputLocation
         *        An Amazon S3 bucket where you want to store the output details of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().apply(outputLocation).build());
        }

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * The last date on which the association was successfully run.
         * </p>
         * 
         * @param lastSuccessfulExecutionDate
         *        The last date on which the association was successfully run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate);

        /**
         * <p>
         * The association name.
         * </p>
         * 
         * @param associationName
         *        The association name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationVersion;

        private Instant dateValue;

        private Instant lastUpdateAssociationDate;

        private AssociationStatus status;

        private AssociationOverview overview;

        private String documentVersion;

        private Map<String, List<String>> parameters;

        private String associationId;

        private List<Target> targets;

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private Instant lastExecutionDate;

        private Instant lastSuccessfulExecutionDate;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(AssociationDescription model) {
            name(model.name);
            instanceId(model.instanceId);
            associationVersion(model.associationVersion);
            date(model.dateValue);
            lastUpdateAssociationDate(model.lastUpdateAssociationDate);
            status(model.status);
            overview(model.overview);
            documentVersion(model.documentVersion);
            parameters(model.parameters);
            associationId(model.associationId);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            lastExecutionDate(model.lastExecutionDate);
            lastSuccessfulExecutionDate(model.lastSuccessfulExecutionDate);
            associationName(model.associationName);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        public final Instant getDate() {
            return dateValue;
        }

        @Override
        public final Builder date(Instant dateValue) {
            this.dateValue = dateValue;
            return this;
        }

        public final void setDate(Instant dateValue) {
            this.dateValue = dateValue;
        }

        public final Instant getLastUpdateAssociationDate() {
            return lastUpdateAssociationDate;
        }

        @Override
        public final Builder lastUpdateAssociationDate(Instant lastUpdateAssociationDate) {
            this.lastUpdateAssociationDate = lastUpdateAssociationDate;
            return this;
        }

        public final void setLastUpdateAssociationDate(Instant lastUpdateAssociationDate) {
            this.lastUpdateAssociationDate = lastUpdateAssociationDate;
        }

        public final AssociationStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(AssociationStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(AssociationStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final AssociationOverview.Builder getOverview() {
            return overview != null ? overview.toBuilder() : null;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final void setOverview(AssociationOverview.BuilderImpl overview) {
            this.overview = overview != null ? overview.build() : null;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        public final Instant getLastSuccessfulExecutionDate() {
            return lastSuccessfulExecutionDate;
        }

        @Override
        public final Builder lastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate) {
            this.lastSuccessfulExecutionDate = lastSuccessfulExecutionDate;
            return this;
        }

        public final void setLastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate) {
            this.lastSuccessfulExecutionDate = lastSuccessfulExecutionDate;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public AssociationDescription build() {
            return new AssociationDescription(this);
        }
    }
}
