/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.DocumentKeyValuesFilterMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * One or more filters. Use a filter to return a more specific list of documents.
 * </p>
 * <p>
 * For keys, you can specify one or more tags that have been applied to a document.
 * </p>
 * <p>
 * Other valid values include Owner, Name, PlatformTypes, and DocumentType.
 * </p>
 * <p>
 * Note that only one Owner can be specified in a request. For example: <code>Key=Owner,Values=Self</code>.
 * </p>
 * <p>
 * If you use Name as a key, you can use a name prefix to return a list of documents. For example, in the AWS CLI, to
 * return a list of all documents that begin with <code>Te</code>, run the following command:
 * </p>
 * <p>
 * <code>aws ssm list-documents --filters Key=Name,Values=Te</code>
 * </p>
 * <p>
 * If you specify more than two keys, only documents that are identified by all the tags are returned in the results. If
 * you specify more than two values for a key, documents that are identified by any of the values are returned in the
 * results.
 * </p>
 * <p>
 * To specify a custom key and value pair, use the format <code>Key=tag:[tagName],Values=[valueName]</code>.
 * </p>
 * <p>
 * For example, if you created a Key called region and are using the AWS CLI to call the <code>list-documents</code>
 * command:
 * </p>
 * <p>
 * <code>aws ssm list-documents --filters Key=tag:region,Values=east,west Key=Owner,Values=Self</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DocumentKeyValuesFilter implements StructuredPojo,
        ToCopyableBuilder<DocumentKeyValuesFilter.Builder, DocumentKeyValuesFilter> {
    private final String key;

    private final List<String> values;

    private DocumentKeyValuesFilter(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the filter key.
     * </p>
     * 
     * @return The name of the filter key.
     */
    public String key() {
        return key;
    }

    /**
     * <p>
     * The value for the filter key.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value for the filter key.
     */
    public List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(values());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentKeyValuesFilter)) {
            return false;
        }
        DocumentKeyValuesFilter other = (DocumentKeyValuesFilter) obj;
        return Objects.equals(key(), other.key()) && Objects.equals(values(), other.values());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (key() != null) {
            sb.append("Key: ").append(key()).append(",");
        }
        if (values() != null) {
            sb.append("Values: ").append(values()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.of(clazz.cast(key()));
        case "Values":
            return Optional.of(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DocumentKeyValuesFilterMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DocumentKeyValuesFilter> {
        /**
         * <p>
         * The name of the filter key.
         * </p>
         * 
         * @param key
         *        The name of the filter key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * The value for the filter key.
         * </p>
         * 
         * @param values
         *        The value for the filter key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * The value for the filter key.
         * </p>
         * 
         * @param values
         *        The value for the filter key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentKeyValuesFilter model) {
            key(model.key);
            values(model.values);
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final Collection<String> getValues() {
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = DocumentKeyValuesFilterValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = DocumentKeyValuesFilterValuesCopier.copy(values);
        }

        @Override
        public DocumentKeyValuesFilter build() {
            return new DocumentKeyValuesFilter(this);
        }
    }
}
