/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class ModifyDocumentPermissionRequest extends SSMRequest implements
        ToCopyableBuilder<ModifyDocumentPermissionRequest.Builder, ModifyDocumentPermissionRequest> {
    private final String name;

    private final String permissionType;

    private final List<String> accountIdsToAdd;

    private final List<String> accountIdsToRemove;

    private ModifyDocumentPermissionRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.permissionType = builder.permissionType;
        this.accountIdsToAdd = builder.accountIdsToAdd;
        this.accountIdsToRemove = builder.accountIdsToRemove;
    }

    /**
     * <p>
     * The name of the document that you want to share.
     * </p>
     * 
     * @return The name of the document that you want to share.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The permission type for the document. The permission type can be <i>Share</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link DocumentPermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #permissionTypeString}.
     * </p>
     * 
     * @return The permission type for the document. The permission type can be <i>Share</i>.
     * @see DocumentPermissionType
     */
    public DocumentPermissionType permissionType() {
        return DocumentPermissionType.fromValue(permissionType);
    }

    /**
     * <p>
     * The permission type for the document. The permission type can be <i>Share</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link DocumentPermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #permissionTypeString}.
     * </p>
     * 
     * @return The permission type for the document. The permission type can be <i>Share</i>.
     * @see DocumentPermissionType
     */
    public String permissionTypeString() {
        return permissionType;
    }

    /**
     * <p>
     * The AWS user accounts that should have access to the document. The account IDs can either be a group of account
     * IDs or <i>All</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The AWS user accounts that should have access to the document. The account IDs can either be a group of
     *         account IDs or <i>All</i>.
     */
    public List<String> accountIdsToAdd() {
        return accountIdsToAdd;
    }

    /**
     * <p>
     * The AWS user accounts that should no longer have access to the document. The AWS user account can either be a
     * group of account IDs or <i>All</i>. This action has a higher priority than <i>AccountIdsToAdd</i>. If you specify
     * an account ID to add and the same ID to remove, the system removes access to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The AWS user accounts that should no longer have access to the document. The AWS user account can either
     *         be a group of account IDs or <i>All</i>. This action has a higher priority than <i>AccountIdsToAdd</i>.
     *         If you specify an account ID to add and the same ID to remove, the system removes access to the document.
     */
    public List<String> accountIdsToRemove() {
        return accountIdsToRemove;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(permissionTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(accountIdsToAdd());
        hashCode = 31 * hashCode + Objects.hashCode(accountIdsToRemove());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyDocumentPermissionRequest)) {
            return false;
        }
        ModifyDocumentPermissionRequest other = (ModifyDocumentPermissionRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(permissionTypeString(), other.permissionTypeString())
                && Objects.equals(accountIdsToAdd(), other.accountIdsToAdd())
                && Objects.equals(accountIdsToRemove(), other.accountIdsToRemove());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (permissionTypeString() != null) {
            sb.append("PermissionType: ").append(permissionTypeString()).append(",");
        }
        if (accountIdsToAdd() != null) {
            sb.append("AccountIdsToAdd: ").append(accountIdsToAdd()).append(",");
        }
        if (accountIdsToRemove() != null) {
            sb.append("AccountIdsToRemove: ").append(accountIdsToRemove()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "PermissionType":
            return Optional.of(clazz.cast(permissionTypeString()));
        case "AccountIdsToAdd":
            return Optional.of(clazz.cast(accountIdsToAdd()));
        case "AccountIdsToRemove":
            return Optional.of(clazz.cast(accountIdsToRemove()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SSMRequest.Builder, CopyableBuilder<Builder, ModifyDocumentPermissionRequest> {
        /**
         * <p>
         * The name of the document that you want to share.
         * </p>
         * 
         * @param name
         *        The name of the document that you want to share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The permission type for the document. The permission type can be <i>Share</i>.
         * </p>
         * 
         * @param permissionType
         *        The permission type for the document. The permission type can be <i>Share</i>.
         * @see DocumentPermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentPermissionType
         */
        Builder permissionType(String permissionType);

        /**
         * <p>
         * The permission type for the document. The permission type can be <i>Share</i>.
         * </p>
         * 
         * @param permissionType
         *        The permission type for the document. The permission type can be <i>Share</i>.
         * @see DocumentPermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentPermissionType
         */
        Builder permissionType(DocumentPermissionType permissionType);

        /**
         * <p>
         * The AWS user accounts that should have access to the document. The account IDs can either be a group of
         * account IDs or <i>All</i>.
         * </p>
         * 
         * @param accountIdsToAdd
         *        The AWS user accounts that should have access to the document. The account IDs can either be a group
         *        of account IDs or <i>All</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIdsToAdd(Collection<String> accountIdsToAdd);

        /**
         * <p>
         * The AWS user accounts that should have access to the document. The account IDs can either be a group of
         * account IDs or <i>All</i>.
         * </p>
         * 
         * @param accountIdsToAdd
         *        The AWS user accounts that should have access to the document. The account IDs can either be a group
         *        of account IDs or <i>All</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIdsToAdd(String... accountIdsToAdd);

        /**
         * <p>
         * The AWS user accounts that should no longer have access to the document. The AWS user account can either be a
         * group of account IDs or <i>All</i>. This action has a higher priority than <i>AccountIdsToAdd</i>. If you
         * specify an account ID to add and the same ID to remove, the system removes access to the document.
         * </p>
         * 
         * @param accountIdsToRemove
         *        The AWS user accounts that should no longer have access to the document. The AWS user account can
         *        either be a group of account IDs or <i>All</i>. This action has a higher priority than
         *        <i>AccountIdsToAdd</i>. If you specify an account ID to add and the same ID to remove, the system
         *        removes access to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIdsToRemove(Collection<String> accountIdsToRemove);

        /**
         * <p>
         * The AWS user accounts that should no longer have access to the document. The AWS user account can either be a
         * group of account IDs or <i>All</i>. This action has a higher priority than <i>AccountIdsToAdd</i>. If you
         * specify an account ID to add and the same ID to remove, the system removes access to the document.
         * </p>
         * 
         * @param accountIdsToRemove
         *        The AWS user accounts that should no longer have access to the document. The AWS user account can
         *        either be a group of account IDs or <i>All</i>. This action has a higher priority than
         *        <i>AccountIdsToAdd</i>. If you specify an account ID to add and the same ID to remove, the system
         *        removes access to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIdsToRemove(String... accountIdsToRemove);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends SSMRequest.BuilderImpl implements Builder {
        private String name;

        private String permissionType;

        private List<String> accountIdsToAdd;

        private List<String> accountIdsToRemove;

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyDocumentPermissionRequest model) {
            name(model.name);
            permissionType(model.permissionType);
            accountIdsToAdd(model.accountIdsToAdd);
            accountIdsToRemove(model.accountIdsToRemove);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getPermissionType() {
            return permissionType;
        }

        @Override
        public final Builder permissionType(String permissionType) {
            this.permissionType = permissionType;
            return this;
        }

        @Override
        public final Builder permissionType(DocumentPermissionType permissionType) {
            this.permissionType(permissionType.toString());
            return this;
        }

        public final void setPermissionType(String permissionType) {
            this.permissionType = permissionType;
        }

        public final Collection<String> getAccountIdsToAdd() {
            return accountIdsToAdd;
        }

        @Override
        public final Builder accountIdsToAdd(Collection<String> accountIdsToAdd) {
            this.accountIdsToAdd = AccountIdListCopier.copy(accountIdsToAdd);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accountIdsToAdd(String... accountIdsToAdd) {
            accountIdsToAdd(Arrays.asList(accountIdsToAdd));
            return this;
        }

        public final void setAccountIdsToAdd(Collection<String> accountIdsToAdd) {
            this.accountIdsToAdd = AccountIdListCopier.copy(accountIdsToAdd);
        }

        public final Collection<String> getAccountIdsToRemove() {
            return accountIdsToRemove;
        }

        @Override
        public final Builder accountIdsToRemove(Collection<String> accountIdsToRemove) {
            this.accountIdsToRemove = AccountIdListCopier.copy(accountIdsToRemove);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accountIdsToRemove(String... accountIdsToRemove) {
            accountIdsToRemove(Arrays.asList(accountIdsToRemove));
            return this;
        }

        public final void setAccountIdsToRemove(Collection<String> accountIdsToRemove) {
            this.accountIdsToRemove = AccountIdListCopier.copy(accountIdsToRemove);
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public ModifyDocumentPermissionRequest build() {
            return new ModifyDocumentPermissionRequest(this);
        }
    }
}
