/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.TargetMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An array of search criteria that targets instances using a Key,Value combination that you specify.
 * <code>Targets</code> is required if you don't provide one or more instance IDs in the call.
 * </p>
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public class Target implements StructuredPojo, ToCopyableBuilder<Target.Builder, Target> {
    private final String key;

    private final List<String> values;

    private Target(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
    }

    /**
     * <p>
     * User-defined criteria for sending commands that target instances that meet the criteria. Key can be
     * tag:&lt;Amazon EC2 tag&gt; or InstanceIds. For more information about how to send commands that target instances
     * using Key,Value parameters, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing a
     * Command Using Systems Manager Run Command</a>.
     * </p>
     * 
     * @return User-defined criteria for sending commands that target instances that meet the criteria. Key can be
     *         tag:&lt;Amazon EC2 tag&gt; or InstanceIds. For more information about how to send commands that target
     *         instances using Key,Value parameters, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing
     *         a Command Using Systems Manager Run Command</a>.
     */
    public String key() {
        return key;
    }

    /**
     * <p>
     * User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could specify
     * value:WebServer to execute a command on instances that include Amazon EC2 tags of ServerRole,WebServer. For more
     * information about how to send commands that target instances using Key,Value parameters, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing a
     * Command Using Systems Manager Run Command</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could specify
     *         value:WebServer to execute a command on instances that include Amazon EC2 tags of ServerRole,WebServer.
     *         For more information about how to send commands that target instances using Key,Value parameters, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing
     *         a Command Using Systems Manager Run Command</a>.
     */
    public List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(values());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Target)) {
            return false;
        }
        Target other = (Target) obj;
        return Objects.equals(key(), other.key()) && Objects.equals(values(), other.values());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (key() != null) {
            sb.append("Key: ").append(key()).append(",");
        }
        if (values() != null) {
            sb.append("Values: ").append(values()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.of(clazz.cast(key()));
        case "Values":
            return Optional.of(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        TargetMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Target> {
        /**
         * <p>
         * User-defined criteria for sending commands that target instances that meet the criteria. Key can be
         * tag:&lt;Amazon EC2 tag&gt; or InstanceIds. For more information about how to send commands that target
         * instances using Key,Value parameters, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing a
         * Command Using Systems Manager Run Command</a>.
         * </p>
         * 
         * @param key
         *        User-defined criteria for sending commands that target instances that meet the criteria. Key can be
         *        tag:&lt;Amazon EC2 tag&gt; or InstanceIds. For more information about how to send commands that target
         *        instances using Key,Value parameters, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html"
         *        >Executing a Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could specify
         * value:WebServer to execute a command on instances that include Amazon EC2 tags of ServerRole,WebServer. For
         * more information about how to send commands that target instances using Key,Value parameters, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing a
         * Command Using Systems Manager Run Command</a>.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could
         *        specify value:WebServer to execute a command on instances that include Amazon EC2 tags of
         *        ServerRole,WebServer. For more information about how to send commands that target instances using
         *        Key,Value parameters, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html"
         *        >Executing a Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could specify
         * value:WebServer to execute a command on instances that include Amazon EC2 tags of ServerRole,WebServer. For
         * more information about how to send commands that target instances using Key,Value parameters, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Executing a
         * Command Using Systems Manager Run Command</a>.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to Key. For example, if you specified tag:ServerRole, you could
         *        specify value:WebServer to execute a command on instances that include Amazon EC2 tags of
         *        ServerRole,WebServer. For more information about how to send commands that target instances using
         *        Key,Value parameters, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html"
         *        >Executing a Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values;

        private BuilderImpl() {
        }

        private BuilderImpl(Target model) {
            key(model.key);
            values(model.values);
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final Collection<String> getValues() {
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
        }

        @Override
        public Target build() {
            return new Target(this);
        }
    }
}
