/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.ResourceDataSyncItemMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a Resource Data Sync configuration, including its current status and last successful sync.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ResourceDataSyncItem implements StructuredPojo,
        ToCopyableBuilder<ResourceDataSyncItem.Builder, ResourceDataSyncItem> {
    private final String syncName;

    private final ResourceDataSyncS3Destination s3Destination;

    private final Instant lastSyncTime;

    private final Instant lastSuccessfulSyncTime;

    private final String lastStatus;

    private final Instant syncCreatedTime;

    private ResourceDataSyncItem(BuilderImpl builder) {
        this.syncName = builder.syncName;
        this.s3Destination = builder.s3Destination;
        this.lastSyncTime = builder.lastSyncTime;
        this.lastSuccessfulSyncTime = builder.lastSuccessfulSyncTime;
        this.lastStatus = builder.lastStatus;
        this.syncCreatedTime = builder.syncCreatedTime;
    }

    /**
     * <p>
     * The name of the Resource Data Sync.
     * </p>
     * 
     * @return The name of the Resource Data Sync.
     */
    public String syncName() {
        return syncName;
    }

    /**
     * <p>
     * Configuration information for the target Amazon S3 bucket.
     * </p>
     * 
     * @return Configuration information for the target Amazon S3 bucket.
     */
    public ResourceDataSyncS3Destination s3Destination() {
        return s3Destination;
    }

    /**
     * <p>
     * The last time the configuration attempted to sync (UTC).
     * </p>
     * 
     * @return The last time the configuration attempted to sync (UTC).
     */
    public Instant lastSyncTime() {
        return lastSyncTime;
    }

    /**
     * <p>
     * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     * </p>
     * 
     * @return The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     */
    public Instant lastSuccessfulSyncTime() {
        return lastSuccessfulSyncTime;
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public LastResourceDataSyncStatus lastStatus() {
        return LastResourceDataSyncStatus.fromValue(lastStatus);
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public String lastStatusString() {
        return lastStatus;
    }

    /**
     * <p>
     * The date and time the configuration was created (UTC).
     * </p>
     * 
     * @return The date and time the configuration was created (UTC).
     */
    public Instant syncCreatedTime() {
        return syncCreatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(syncName());
        hashCode = 31 * hashCode + Objects.hashCode(s3Destination());
        hashCode = 31 * hashCode + Objects.hashCode(lastSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastStatusString());
        hashCode = 31 * hashCode + Objects.hashCode(syncCreatedTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncItem)) {
            return false;
        }
        ResourceDataSyncItem other = (ResourceDataSyncItem) obj;
        return Objects.equals(syncName(), other.syncName()) && Objects.equals(s3Destination(), other.s3Destination())
                && Objects.equals(lastSyncTime(), other.lastSyncTime())
                && Objects.equals(lastSuccessfulSyncTime(), other.lastSuccessfulSyncTime())
                && Objects.equals(lastStatusString(), other.lastStatusString())
                && Objects.equals(syncCreatedTime(), other.syncCreatedTime());
    }

    @Override
    public String toString() {
        return ToString.builder("ResourceDataSyncItem").add("SyncName", syncName()).add("S3Destination", s3Destination())
                .add("LastSyncTime", lastSyncTime()).add("LastSuccessfulSyncTime", lastSuccessfulSyncTime())
                .add("LastStatus", lastStatusString()).add("SyncCreatedTime", syncCreatedTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SyncName":
            return Optional.of(clazz.cast(syncName()));
        case "S3Destination":
            return Optional.of(clazz.cast(s3Destination()));
        case "LastSyncTime":
            return Optional.of(clazz.cast(lastSyncTime()));
        case "LastSuccessfulSyncTime":
            return Optional.of(clazz.cast(lastSuccessfulSyncTime()));
        case "LastStatus":
            return Optional.of(clazz.cast(lastStatusString()));
        case "SyncCreatedTime":
            return Optional.of(clazz.cast(syncCreatedTime()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ResourceDataSyncItemMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ResourceDataSyncItem> {
        /**
         * <p>
         * The name of the Resource Data Sync.
         * </p>
         * 
         * @param syncName
         *        The name of the Resource Data Sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncName(String syncName);

        /**
         * <p>
         * Configuration information for the target Amazon S3 bucket.
         * </p>
         * 
         * @param s3Destination
         *        Configuration information for the target Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Destination(ResourceDataSyncS3Destination s3Destination);

        /**
         * <p>
         * Configuration information for the target Amazon S3 bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceDataSyncS3Destination.Builder} avoiding
         * the need to create one manually via {@link ResourceDataSyncS3Destination#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceDataSyncS3Destination.Builder#build()} is called
         * immediately and its result is passed to {@link #s3Destination(ResourceDataSyncS3Destination)}.
         * 
         * @param s3Destination
         *        a consumer that will call methods on {@link ResourceDataSyncS3Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Destination(ResourceDataSyncS3Destination)
         */
        default Builder s3Destination(Consumer<ResourceDataSyncS3Destination.Builder> s3Destination) {
            return s3Destination(ResourceDataSyncS3Destination.builder().apply(s3Destination).build());
        }

        /**
         * <p>
         * The last time the configuration attempted to sync (UTC).
         * </p>
         * 
         * @param lastSyncTime
         *        The last time the configuration attempted to sync (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSyncTime(Instant lastSyncTime);

        /**
         * <p>
         * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * </p>
         * 
         * @param lastSuccessfulSyncTime
         *        The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(String lastStatus);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(LastResourceDataSyncStatus lastStatus);

        /**
         * <p>
         * The date and time the configuration was created (UTC).
         * </p>
         * 
         * @param syncCreatedTime
         *        The date and time the configuration was created (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncCreatedTime(Instant syncCreatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String syncName;

        private ResourceDataSyncS3Destination s3Destination;

        private Instant lastSyncTime;

        private Instant lastSuccessfulSyncTime;

        private String lastStatus;

        private Instant syncCreatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncItem model) {
            syncName(model.syncName);
            s3Destination(model.s3Destination);
            lastSyncTime(model.lastSyncTime);
            lastSuccessfulSyncTime(model.lastSuccessfulSyncTime);
            lastStatus(model.lastStatus);
            syncCreatedTime(model.syncCreatedTime);
        }

        public final String getSyncName() {
            return syncName;
        }

        @Override
        public final Builder syncName(String syncName) {
            this.syncName = syncName;
            return this;
        }

        public final void setSyncName(String syncName) {
            this.syncName = syncName;
        }

        public final ResourceDataSyncS3Destination.Builder getS3Destination() {
            return s3Destination != null ? s3Destination.toBuilder() : null;
        }

        @Override
        public final Builder s3Destination(ResourceDataSyncS3Destination s3Destination) {
            this.s3Destination = s3Destination;
            return this;
        }

        public final void setS3Destination(ResourceDataSyncS3Destination.BuilderImpl s3Destination) {
            this.s3Destination = s3Destination != null ? s3Destination.build() : null;
        }

        public final Instant getLastSyncTime() {
            return lastSyncTime;
        }

        @Override
        public final Builder lastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
            return this;
        }

        public final void setLastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
        }

        public final Instant getLastSuccessfulSyncTime() {
            return lastSuccessfulSyncTime;
        }

        @Override
        public final Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
            return this;
        }

        public final void setLastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
        }

        public final String getLastStatus() {
            return lastStatus;
        }

        @Override
        public final Builder lastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
            return this;
        }

        @Override
        public final Builder lastStatus(LastResourceDataSyncStatus lastStatus) {
            this.lastStatus(lastStatus.toString());
            return this;
        }

        public final void setLastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
        }

        public final Instant getSyncCreatedTime() {
            return syncCreatedTime;
        }

        @Override
        public final Builder syncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
            return this;
        }

        public final void setSyncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
        }

        @Override
        public ResourceDataSyncItem build() {
            return new ResourceDataSyncItem(this);
        }
    }
}
