/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the parameters for a document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssociationDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<AssociationDescription.Builder, AssociationDescription> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> ASSOCIATION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::associationVersion)).setter(setter(Builder::associationVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationVersion").build())
            .build();

    private static final SdkField<Instant> DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssociationDescription::date)).setter(setter(Builder::date))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Date").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_ASSOCIATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).getter(getter(AssociationDescription::lastUpdateAssociationDate))
            .setter(setter(Builder::lastUpdateAssociationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateAssociationDate").build())
            .build();

    private static final SdkField<AssociationStatus> STATUS_FIELD = SdkField
            .<AssociationStatus> builder(MarshallingType.SDK_POJO).getter(getter(AssociationDescription::status))
            .setter(setter(Builder::status)).constructor(AssociationStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<AssociationOverview> OVERVIEW_FIELD = SdkField
            .<AssociationOverview> builder(MarshallingType.SDK_POJO).getter(getter(AssociationDescription::overview))
            .setter(setter(Builder::overview)).constructor(AssociationOverview::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overview").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(AssociationDescription::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::associationId)).setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(AssociationDescription::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::scheduleExpression)).setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<InstanceAssociationOutputLocation> OUTPUT_LOCATION_FIELD = SdkField
            .<InstanceAssociationOutputLocation> builder(MarshallingType.SDK_POJO)
            .getter(getter(AssociationDescription::outputLocation)).setter(setter(Builder::outputLocation))
            .constructor(InstanceAssociationOutputLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocation").build()).build();

    private static final SdkField<Instant> LAST_EXECUTION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssociationDescription::lastExecutionDate)).setter(setter(Builder::lastExecutionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastExecutionDate").build()).build();

    private static final SdkField<Instant> LAST_SUCCESSFUL_EXECUTION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssociationDescription::lastSuccessfulExecutionDate))
            .setter(setter(Builder::lastSuccessfulExecutionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSuccessfulExecutionDate")
                    .build()).build();

    private static final SdkField<String> ASSOCIATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::associationName)).setter(setter(Builder::associationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationName").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::maxConcurrency)).setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> COMPLIANCE_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssociationDescription::complianceSeverityAsString)).setter(setter(Builder::complianceSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceSeverity").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, INSTANCE_ID_FIELD,
            ASSOCIATION_VERSION_FIELD, DATE_FIELD, LAST_UPDATE_ASSOCIATION_DATE_FIELD, STATUS_FIELD, OVERVIEW_FIELD,
            DOCUMENT_VERSION_FIELD, PARAMETERS_FIELD, ASSOCIATION_ID_FIELD, TARGETS_FIELD, SCHEDULE_EXPRESSION_FIELD,
            OUTPUT_LOCATION_FIELD, LAST_EXECUTION_DATE_FIELD, LAST_SUCCESSFUL_EXECUTION_DATE_FIELD, ASSOCIATION_NAME_FIELD,
            MAX_ERRORS_FIELD, MAX_CONCURRENCY_FIELD, COMPLIANCE_SEVERITY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String instanceId;

    private final String associationVersion;

    private final Instant date;

    private final Instant lastUpdateAssociationDate;

    private final AssociationStatus status;

    private final AssociationOverview overview;

    private final String documentVersion;

    private final Map<String, List<String>> parameters;

    private final String associationId;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final Instant lastExecutionDate;

    private final Instant lastSuccessfulExecutionDate;

    private final String associationName;

    private final String maxErrors;

    private final String maxConcurrency;

    private final String complianceSeverity;

    private AssociationDescription(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationVersion = builder.associationVersion;
        this.date = builder.date;
        this.lastUpdateAssociationDate = builder.lastUpdateAssociationDate;
        this.status = builder.status;
        this.overview = builder.overview;
        this.documentVersion = builder.documentVersion;
        this.parameters = builder.parameters;
        this.associationId = builder.associationId;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.lastSuccessfulExecutionDate = builder.lastSuccessfulExecutionDate;
        this.associationName = builder.associationName;
        this.maxErrors = builder.maxErrors;
        this.maxConcurrency = builder.maxConcurrency;
        this.complianceSeverity = builder.complianceSeverity;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The date when the association was made.
     * </p>
     * 
     * @return The date when the association was made.
     */
    public Instant date() {
        return date;
    }

    /**
     * <p>
     * The date when the association was last updated.
     * </p>
     * 
     * @return The date when the association was last updated.
     */
    public Instant lastUpdateAssociationDate() {
        return lastUpdateAssociationDate;
    }

    /**
     * <p>
     * The association status.
     * </p>
     * 
     * @return The association status.
     */
    public AssociationStatus status() {
        return status;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * A description of the parameters for a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A description of the parameters for a document.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The association ID.
     * </p>
     * 
     * @return The association ID.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The instances targeted by the request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An Amazon S3 bucket where you want to store the output details of the request.
     * </p>
     * 
     * @return An Amazon S3 bucket where you want to store the output details of the request.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * The last date on which the association was successfully run.
     * </p>
     * 
     * @return The last date on which the association was successfully run.
     */
    public Instant lastSuccessfulExecutionDate() {
        return lastSuccessfulExecutionDate;
    }

    /**
     * <p>
     * The association name.
     * </p>
     * 
     * @return The association name.
     */
    public String associationName() {
        return associationName;
    }

    /**
     * <p>
     * The number of errors that are allowed before the system stops sending requests to run the association on
     * additional targets. You can specify either an absolute number of errors, for example 10, or a percentage of the
     * target set, for example 10%. If you specify 3, for example, the system stops sending requests when the fourth
     * error is received. If you specify 0, then the system stops sending requests after the first error is returned. If
     * you run an association on 50 instances and set MaxError to 10%, then the system stops sending the request when
     * the sixth error is received.
     * </p>
     * <p>
     * Executions that are already running an association when MaxErrors is reached are allowed to complete, but some of
     * these executions may fail as well. If you need to ensure that there won't be more than max-errors failed
     * executions, set MaxConcurrency to 1 so that executions proceed one at a time.
     * </p>
     * 
     * @return The number of errors that are allowed before the system stops sending requests to run the association on
     *         additional targets. You can specify either an absolute number of errors, for example 10, or a percentage
     *         of the target set, for example 10%. If you specify 3, for example, the system stops sending requests when
     *         the fourth error is received. If you specify 0, then the system stops sending requests after the first
     *         error is returned. If you run an association on 50 instances and set MaxError to 10%, then the system
     *         stops sending the request when the sixth error is received.</p>
     *         <p>
     *         Executions that are already running an association when MaxErrors is reached are allowed to complete, but
     *         some of these executions may fail as well. If you need to ensure that there won't be more than max-errors
     *         failed executions, set MaxConcurrency to 1 so that executions proceed one at a time.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The maximum number of targets allowed to run the association at the same time. You can specify a number, for
     * example 10, or a percentage of the target set, for example 10%. The default value is 100%, which means all
     * targets run the association at the same time.
     * </p>
     * <p>
     * If a new instance starts and attempts to execute an association while Systems Manager is executing MaxConcurrency
     * associations, the association is allowed to run. During the next association interval, the new instance will
     * process its association within the limit specified for MaxConcurrency.
     * </p>
     * 
     * @return The maximum number of targets allowed to run the association at the same time. You can specify a number,
     *         for example 10, or a percentage of the target set, for example 10%. The default value is 100%, which
     *         means all targets run the association at the same time.</p>
     *         <p>
     *         If a new instance starts and attempts to execute an association while Systems Manager is executing
     *         MaxConcurrency associations, the association is allowed to run. During the next association interval, the
     *         new instance will process its association within the limit specified for MaxConcurrency.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The severity level that is assigned to the association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #complianceSeverity} will return {@link AssociationComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #complianceSeverityAsString}.
     * </p>
     * 
     * @return The severity level that is assigned to the association.
     * @see AssociationComplianceSeverity
     */
    public AssociationComplianceSeverity complianceSeverity() {
        return AssociationComplianceSeverity.fromValue(complianceSeverity);
    }

    /**
     * <p>
     * The severity level that is assigned to the association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #complianceSeverity} will return {@link AssociationComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #complianceSeverityAsString}.
     * </p>
     * 
     * @return The severity level that is assigned to the association.
     * @see AssociationComplianceSeverity
     */
    public String complianceSeverityAsString() {
        return complianceSeverity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateAssociationDate());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(overview());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(lastExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(complianceSeverityAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociationDescription)) {
            return false;
        }
        AssociationDescription other = (AssociationDescription) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(associationVersion(), other.associationVersion()) && Objects.equals(date(), other.date())
                && Objects.equals(lastUpdateAssociationDate(), other.lastUpdateAssociationDate())
                && Objects.equals(status(), other.status()) && Objects.equals(overview(), other.overview())
                && Objects.equals(documentVersion(), other.documentVersion()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(associationId(), other.associationId()) && Objects.equals(targets(), other.targets())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(lastExecutionDate(), other.lastExecutionDate())
                && Objects.equals(lastSuccessfulExecutionDate(), other.lastSuccessfulExecutionDate())
                && Objects.equals(associationName(), other.associationName()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(maxConcurrency(), other.maxConcurrency())
                && Objects.equals(complianceSeverityAsString(), other.complianceSeverityAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("AssociationDescription").add("Name", name()).add("InstanceId", instanceId())
                .add("AssociationVersion", associationVersion()).add("Date", date())
                .add("LastUpdateAssociationDate", lastUpdateAssociationDate()).add("Status", status())
                .add("Overview", overview()).add("DocumentVersion", documentVersion()).add("Parameters", parameters())
                .add("AssociationId", associationId()).add("Targets", targets()).add("ScheduleExpression", scheduleExpression())
                .add("OutputLocation", outputLocation()).add("LastExecutionDate", lastExecutionDate())
                .add("LastSuccessfulExecutionDate", lastSuccessfulExecutionDate()).add("AssociationName", associationName())
                .add("MaxErrors", maxErrors()).add("MaxConcurrency", maxConcurrency())
                .add("ComplianceSeverity", complianceSeverityAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "AssociationVersion":
            return Optional.ofNullable(clazz.cast(associationVersion()));
        case "Date":
            return Optional.ofNullable(clazz.cast(date()));
        case "LastUpdateAssociationDate":
            return Optional.ofNullable(clazz.cast(lastUpdateAssociationDate()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Overview":
            return Optional.ofNullable(clazz.cast(overview()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "LastExecutionDate":
            return Optional.ofNullable(clazz.cast(lastExecutionDate()));
        case "LastSuccessfulExecutionDate":
            return Optional.ofNullable(clazz.cast(lastSuccessfulExecutionDate()));
        case "AssociationName":
            return Optional.ofNullable(clazz.cast(associationName()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "ComplianceSeverity":
            return Optional.ofNullable(clazz.cast(complianceSeverityAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssociationDescription, T> g) {
        return obj -> g.apply((AssociationDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssociationDescription> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The date when the association was made.
         * </p>
         * 
         * @param date
         *        The date when the association was made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(Instant date);

        /**
         * <p>
         * The date when the association was last updated.
         * </p>
         * 
         * @param lastUpdateAssociationDate
         *        The date when the association was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateAssociationDate(Instant lastUpdateAssociationDate);

        /**
         * <p>
         * The association status.
         * </p>
         * 
         * @param status
         *        The association status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(AssociationStatus status);

        /**
         * <p>
         * The association status.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationStatus.Builder} avoiding the need to
         * create one manually via {@link AssociationStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(AssociationStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link AssociationStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(AssociationStatus)
         */
        default Builder status(Consumer<AssociationStatus.Builder> status) {
            return status(AssociationStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * Information about the association.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationOverview.Builder} avoiding the need
         * to create one manually via {@link AssociationOverview#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationOverview.Builder#build()} is called immediately and
         * its result is passed to {@link #overview(AssociationOverview)}.
         * 
         * @param overview
         *        a consumer that will call methods on {@link AssociationOverview.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overview(AssociationOverview)
         */
        default Builder overview(Consumer<AssociationOverview.Builder> overview) {
            return overview(AssociationOverview.builder().applyMutation(overview).build());
        }

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The association ID.
         * </p>
         * 
         * @param associationId
         *        The association ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * 
         * @param outputLocation
         *        An Amazon S3 bucket where you want to store the output details of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the output details of the request.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().applyMutation(outputLocation).build());
        }

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * The last date on which the association was successfully run.
         * </p>
         * 
         * @param lastSuccessfulExecutionDate
         *        The last date on which the association was successfully run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate);

        /**
         * <p>
         * The association name.
         * </p>
         * 
         * @param associationName
         *        The association name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);

        /**
         * <p>
         * The number of errors that are allowed before the system stops sending requests to run the association on
         * additional targets. You can specify either an absolute number of errors, for example 10, or a percentage of
         * the target set, for example 10%. If you specify 3, for example, the system stops sending requests when the
         * fourth error is received. If you specify 0, then the system stops sending requests after the first error is
         * returned. If you run an association on 50 instances and set MaxError to 10%, then the system stops sending
         * the request when the sixth error is received.
         * </p>
         * <p>
         * Executions that are already running an association when MaxErrors is reached are allowed to complete, but
         * some of these executions may fail as well. If you need to ensure that there won't be more than max-errors
         * failed executions, set MaxConcurrency to 1 so that executions proceed one at a time.
         * </p>
         * 
         * @param maxErrors
         *        The number of errors that are allowed before the system stops sending requests to run the association
         *        on additional targets. You can specify either an absolute number of errors, for example 10, or a
         *        percentage of the target set, for example 10%. If you specify 3, for example, the system stops sending
         *        requests when the fourth error is received. If you specify 0, then the system stops sending requests
         *        after the first error is returned. If you run an association on 50 instances and set MaxError to 10%,
         *        then the system stops sending the request when the sixth error is received.</p>
         *        <p>
         *        Executions that are already running an association when MaxErrors is reached are allowed to complete,
         *        but some of these executions may fail as well. If you need to ensure that there won't be more than
         *        max-errors failed executions, set MaxConcurrency to 1 so that executions proceed one at a time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The maximum number of targets allowed to run the association at the same time. You can specify a number, for
         * example 10, or a percentage of the target set, for example 10%. The default value is 100%, which means all
         * targets run the association at the same time.
         * </p>
         * <p>
         * If a new instance starts and attempts to execute an association while Systems Manager is executing
         * MaxConcurrency associations, the association is allowed to run. During the next association interval, the new
         * instance will process its association within the limit specified for MaxConcurrency.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of targets allowed to run the association at the same time. You can specify a
         *        number, for example 10, or a percentage of the target set, for example 10%. The default value is 100%,
         *        which means all targets run the association at the same time.</p>
         *        <p>
         *        If a new instance starts and attempts to execute an association while Systems Manager is executing
         *        MaxConcurrency associations, the association is allowed to run. During the next association interval,
         *        the new instance will process its association within the limit specified for MaxConcurrency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The severity level that is assigned to the association.
         * </p>
         * 
         * @param complianceSeverity
         *        The severity level that is assigned to the association.
         * @see AssociationComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationComplianceSeverity
         */
        Builder complianceSeverity(String complianceSeverity);

        /**
         * <p>
         * The severity level that is assigned to the association.
         * </p>
         * 
         * @param complianceSeverity
         *        The severity level that is assigned to the association.
         * @see AssociationComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationComplianceSeverity
         */
        Builder complianceSeverity(AssociationComplianceSeverity complianceSeverity);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationVersion;

        private Instant date;

        private Instant lastUpdateAssociationDate;

        private AssociationStatus status;

        private AssociationOverview overview;

        private String documentVersion;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String associationId;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private Instant lastExecutionDate;

        private Instant lastSuccessfulExecutionDate;

        private String associationName;

        private String maxErrors;

        private String maxConcurrency;

        private String complianceSeverity;

        private BuilderImpl() {
        }

        private BuilderImpl(AssociationDescription model) {
            name(model.name);
            instanceId(model.instanceId);
            associationVersion(model.associationVersion);
            date(model.date);
            lastUpdateAssociationDate(model.lastUpdateAssociationDate);
            status(model.status);
            overview(model.overview);
            documentVersion(model.documentVersion);
            parameters(model.parameters);
            associationId(model.associationId);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            lastExecutionDate(model.lastExecutionDate);
            lastSuccessfulExecutionDate(model.lastSuccessfulExecutionDate);
            associationName(model.associationName);
            maxErrors(model.maxErrors);
            maxConcurrency(model.maxConcurrency);
            complianceSeverity(model.complianceSeverity);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        public final Instant getDate() {
            return date;
        }

        @Override
        public final Builder date(Instant date) {
            this.date = date;
            return this;
        }

        public final void setDate(Instant date) {
            this.date = date;
        }

        public final Instant getLastUpdateAssociationDate() {
            return lastUpdateAssociationDate;
        }

        @Override
        public final Builder lastUpdateAssociationDate(Instant lastUpdateAssociationDate) {
            this.lastUpdateAssociationDate = lastUpdateAssociationDate;
            return this;
        }

        public final void setLastUpdateAssociationDate(Instant lastUpdateAssociationDate) {
            this.lastUpdateAssociationDate = lastUpdateAssociationDate;
        }

        public final AssociationStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(AssociationStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(AssociationStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final AssociationOverview.Builder getOverview() {
            return overview != null ? overview.toBuilder() : null;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final void setOverview(AssociationOverview.BuilderImpl overview) {
            this.overview = overview != null ? overview.build() : null;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        public final Instant getLastSuccessfulExecutionDate() {
            return lastSuccessfulExecutionDate;
        }

        @Override
        public final Builder lastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate) {
            this.lastSuccessfulExecutionDate = lastSuccessfulExecutionDate;
            return this;
        }

        public final void setLastSuccessfulExecutionDate(Instant lastSuccessfulExecutionDate) {
            this.lastSuccessfulExecutionDate = lastSuccessfulExecutionDate;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getComplianceSeverity() {
            return complianceSeverity;
        }

        @Override
        public final Builder complianceSeverity(String complianceSeverity) {
            this.complianceSeverity = complianceSeverity;
            return this;
        }

        @Override
        public final Builder complianceSeverity(AssociationComplianceSeverity complianceSeverity) {
            this.complianceSeverity(complianceSeverity.toString());
            return this;
        }

        public final void setComplianceSeverity(String complianceSeverity) {
            this.complianceSeverity = complianceSeverity;
        }

        @Override
        public AssociationDescription build() {
            return new AssociationDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
