/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the name of a Systems Manager document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentIdentifier implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentIdentifier.Builder, DocumentIdentifier> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::versionName)).setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<List<String>> PLATFORM_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DocumentIdentifier::platformTypesAsStrings))
            .setter(setter(Builder::platformTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::documentTypeAsString)).setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::schemaVersion)).setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> DOCUMENT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::documentFormatAsString)).setter(setter(Builder::documentFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentFormat").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DocumentIdentifier::targetType)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(DocumentIdentifier::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, OWNER_FIELD,
            VERSION_NAME_FIELD, PLATFORM_TYPES_FIELD, DOCUMENT_VERSION_FIELD, DOCUMENT_TYPE_FIELD, SCHEMA_VERSION_FIELD,
            DOCUMENT_FORMAT_FIELD, TARGET_TYPE_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String owner;

    private final String versionName;

    private final List<String> platformTypes;

    private final String documentVersion;

    private final String documentType;

    private final String schemaVersion;

    private final String documentFormat;

    private final String targetType;

    private final List<Tag> tags;

    private DocumentIdentifier(BuilderImpl builder) {
        this.name = builder.name;
        this.owner = builder.owner;
        this.versionName = builder.versionName;
        this.platformTypes = builder.platformTypes;
        this.documentVersion = builder.documentVersion;
        this.documentType = builder.documentType;
        this.schemaVersion = builder.schemaVersion;
        this.documentFormat = builder.documentFormat;
        this.targetType = builder.targetType;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The AWS user account that created the document.
     * </p>
     * 
     * @return The AWS user account that created the document.
     */
    public String owner() {
        return owner;
    }

    /**
     * <p>
     * An optional field specifying the version of the artifact associated with the document. For example,
     * "Release 12, Update 6". This value is unique across all versions of a document, and cannot be changed.
     * </p>
     * 
     * @return An optional field specifying the version of the artifact associated with the document. For example,
     *         "Release 12, Update 6". This value is unique across all versions of a document, and cannot be changed.
     */
    public String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The operating system platform.
     */
    public List<PlatformType> platformTypes() {
        return TypeConverter.convert(platformTypes, PlatformType::fromValue);
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The operating system platform.
     */
    public List<String> platformTypesAsStrings() {
        return platformTypes;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * The schema version.
     * </p>
     * 
     * @return The schema version.
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public DocumentFormat documentFormat() {
        return DocumentFormat.fromValue(documentFormat);
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public String documentFormatAsString() {
        return documentFormat;
    }

    /**
     * <p>
     * The target type which defines the kinds of resources the document can run on. For example, /AWS::EC2::Instance.
     * For a list of valid resource types, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
     * Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The target type which defines the kinds of resources the document can run on. For example,
     *         /AWS::EC2::Instance. For a list of valid resource types, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
     *         Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String targetType() {
        return targetType;
    }

    /**
     * <p>
     * The tags, or metadata, that have been applied to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags, or metadata, that have been applied to the document.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(platformTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetType());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentIdentifier)) {
            return false;
        }
        DocumentIdentifier other = (DocumentIdentifier) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(owner(), other.owner())
                && Objects.equals(versionName(), other.versionName())
                && Objects.equals(platformTypesAsStrings(), other.platformTypesAsStrings())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(documentFormatAsString(), other.documentFormatAsString())
                && Objects.equals(targetType(), other.targetType()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("DocumentIdentifier").add("Name", name()).add("Owner", owner()).add("VersionName", versionName())
                .add("PlatformTypes", platformTypesAsStrings()).add("DocumentVersion", documentVersion())
                .add("DocumentType", documentTypeAsString()).add("SchemaVersion", schemaVersion())
                .add("DocumentFormat", documentFormatAsString()).add("TargetType", targetType()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "PlatformTypes":
            return Optional.ofNullable(clazz.cast(platformTypesAsStrings()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "DocumentFormat":
            return Optional.ofNullable(clazz.cast(documentFormatAsString()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentIdentifier, T> g) {
        return obj -> g.apply((DocumentIdentifier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentIdentifier> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The AWS user account that created the document.
         * </p>
         * 
         * @param owner
         *        The AWS user account that created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * An optional field specifying the version of the artifact associated with the document. For example,
         * "Release 12, Update 6". This value is unique across all versions of a document, and cannot be changed.
         * </p>
         * 
         * @param versionName
         *        An optional field specifying the version of the artifact associated with the document. For example,
         *        "Release 12, Update 6". This value is unique across all versions of a document, and cannot be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(Collection<String> platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(String... platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(Collection<PlatformType> platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(PlatformType... platformTypes);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * The schema version.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(String documentFormat);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(DocumentFormat documentFormat);

        /**
         * <p>
         * The target type which defines the kinds of resources the document can run on. For example,
         * /AWS::EC2::Instance. For a list of valid resource types, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">AWS
         * Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param targetType
         *        The target type which defines the kinds of resources the document can run on. For example,
         *        /AWS::EC2::Instance. For a list of valid resource types, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >AWS Resource Types Reference</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String owner;

        private String versionName;

        private List<String> platformTypes = DefaultSdkAutoConstructList.getInstance();

        private String documentVersion;

        private String documentType;

        private String schemaVersion;

        private String documentFormat;

        private String targetType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentIdentifier model) {
            name(model.name);
            owner(model.owner);
            versionName(model.versionName);
            platformTypesWithStrings(model.platformTypes);
            documentVersion(model.documentVersion);
            documentType(model.documentType);
            schemaVersion(model.schemaVersion);
            documentFormat(model.documentFormat);
            targetType(model.targetType);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getOwner() {
            return owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        public final String getVersionName() {
            return versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        public final Collection<String> getPlatformTypes() {
            return platformTypes;
        }

        @Override
        public final Builder platformTypesWithStrings(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypesWithStrings(String... platformTypes) {
            platformTypesWithStrings(Arrays.asList(platformTypes));
            return this;
        }

        @Override
        public final Builder platformTypes(Collection<PlatformType> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copyEnumToString(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypes(PlatformType... platformTypes) {
            platformTypes(Arrays.asList(platformTypes));
            return this;
        }

        public final void setPlatformTypes(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getDocumentType() {
            return documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType.toString());
            return this;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final String getDocumentFormat() {
            return documentFormat;
        }

        @Override
        public final Builder documentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
            return this;
        }

        @Override
        public final Builder documentFormat(DocumentFormat documentFormat) {
            this.documentFormat(documentFormat.toString());
            return this;
        }

        public final void setDocumentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
        }

        public final String getTargetType() {
            return targetType;
        }

        @Override
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public DocumentIdentifier build() {
            return new DocumentIdentifier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
