/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the patches to use to update the instances, including target operating systems and source
 * repository. Applies to Linux instances only.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchSource implements SdkPojo, Serializable, ToCopyableBuilder<PatchSource.Builder, PatchSource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PatchSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> PRODUCTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(PatchSource::products))
            .setter(setter(Builder::products))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Products").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PatchSource::configuration)).setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, PRODUCTS_FIELD,
            CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> products;

    private final String configuration;

    private PatchSource(BuilderImpl builder) {
        this.name = builder.name;
        this.products = builder.products;
        this.configuration = builder.configuration;
    }

    /**
     * <p>
     * The name specified to identify the patch source.
     * </p>
     * 
     * @return The name specified to identify the patch source.
     */
    public String name() {
        return name;
    }

    /**
     * Returns true if the Products property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasProducts() {
        return products != null && !(products instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
     * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
     * <a>PatchFilter</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProducts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
     *         "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
     *         see <a>PatchFilter</a>.
     */
    public List<String> products() {
        return products;
    }

    /**
     * <p>
     * The value of the yum repo configuration. For example:
     * </p>
     * <p>
     * <code>[main]</code>
     * </p>
     * <p>
     * <code>cachedir=/var/cache/yum/$basesearch$releasever</code>
     * </p>
     * <p>
     * <code>keepcache=0</code>
     * </p>
     * <p>
     * <code>debuglevel=2</code>
     * </p>
     * 
     * @return The value of the yum repo configuration. For example:</p>
     *         <p>
     *         <code>[main]</code>
     *         </p>
     *         <p>
     *         <code>cachedir=/var/cache/yum/$basesearch$releasever</code>
     *         </p>
     *         <p>
     *         <code>keepcache=0</code>
     *         </p>
     *         <p>
     *         <code>debuglevel=2</code>
     */
    public String configuration() {
        return configuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(products());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchSource)) {
            return false;
        }
        PatchSource other = (PatchSource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(products(), other.products())
                && Objects.equals(configuration(), other.configuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PatchSource").add("Name", name()).add("Products", products())
                .add("Configuration", configuration() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Products":
            return Optional.ofNullable(clazz.cast(products()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PatchSource, T> g) {
        return obj -> g.apply((PatchSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PatchSource> {
        /**
         * <p>
         * The name specified to identify the patch source.
         * </p>
         * 
         * @param name
         *        The name specified to identify the patch source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
         * <a>PatchFilter</a>.
         * </p>
         * 
         * @param products
         *        The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         *        "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
         *        see <a>PatchFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder products(Collection<String> products);

        /**
         * <p>
         * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
         * <a>PatchFilter</a>.
         * </p>
         * 
         * @param products
         *        The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         *        "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
         *        see <a>PatchFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder products(String... products);

        /**
         * <p>
         * The value of the yum repo configuration. For example:
         * </p>
         * <p>
         * <code>[main]</code>
         * </p>
         * <p>
         * <code>cachedir=/var/cache/yum/$basesearch$releasever</code>
         * </p>
         * <p>
         * <code>keepcache=0</code>
         * </p>
         * <p>
         * <code>debuglevel=2</code>
         * </p>
         * 
         * @param configuration
         *        The value of the yum repo configuration. For example:</p>
         *        <p>
         *        <code>[main]</code>
         *        </p>
         *        <p>
         *        <code>cachedir=/var/cache/yum/$basesearch$releasever</code>
         *        </p>
         *        <p>
         *        <code>keepcache=0</code>
         *        </p>
         *        <p>
         *        <code>debuglevel=2</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(String configuration);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> products = DefaultSdkAutoConstructList.getInstance();

        private String configuration;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchSource model) {
            name(model.name);
            products(model.products);
            configuration(model.configuration);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<String> getProducts() {
            return products;
        }

        @Override
        public final Builder products(Collection<String> products) {
            this.products = PatchSourceProductListCopier.copy(products);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder products(String... products) {
            products(Arrays.asList(products));
            return this;
        }

        public final void setProducts(Collection<String> products) {
            this.products = PatchSourceProductListCopier.copy(products);
        }

        public final String getConfiguration() {
            return configuration;
        }

        @Override
        public final Builder configuration(String configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(String configuration) {
            this.configuration = configuration;
        }

        @Override
        public PatchSource build() {
            return new PatchSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
