/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutParameterRequest extends SsmRequest implements
        ToCopyableBuilder<PutParameterRequest.Builder, PutParameterRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<Boolean> OVERWRITE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(PutParameterRequest::overwrite)).setter(setter(Builder::overwrite))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overwrite").build()).build();

    private static final SdkField<String> ALLOWED_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::allowedPattern)).setter(setter(Builder::allowedPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedPattern").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(PutParameterRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<String> POLICIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::policies)).setter(setter(Builder::policies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build()).build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutParameterRequest::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            VALUE_FIELD, TYPE_FIELD, KEY_ID_FIELD, OVERWRITE_FIELD, ALLOWED_PATTERN_FIELD, TAGS_FIELD, TIER_FIELD,
            POLICIES_FIELD, DATA_TYPE_FIELD));

    private final String name;

    private final String description;

    private final String value;

    private final String type;

    private final String keyId;

    private final Boolean overwrite;

    private final String allowedPattern;

    private final List<Tag> tags;

    private final String tier;

    private final String policies;

    private final String dataType;

    private PutParameterRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.value = builder.value;
        this.type = builder.type;
        this.keyId = builder.keyId;
        this.overwrite = builder.overwrite;
        this.allowedPattern = builder.allowedPattern;
        this.tags = builder.tags;
        this.tier = builder.tier;
        this.policies = builder.policies;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * The fully qualified name of the parameter that you want to add to the system. The fully qualified name includes
     * the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must include a leading
     * forward slash character (/) when you create or reference a parameter. For example:
     * <code>/Dev/DBServer/MySQL/db-string13</code>
     * </p>
     * <p>
     * Naming Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Parameter names are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name must be unique within an AWS Region
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name can't be prefixed with "aws" or "ssm" (case-insensitive).
     * </p>
     * </li>
     * <li>
     * <p>
     * Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-/</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name can't include spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * Parameter hierarchies are limited to a maximum depth of fifteen levels.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For additional information about valid values for parameter names, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-parameter-name-constraints.html">About
     * requirements and constraints for parameter names</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The maximum length constraint listed below includes capacity for additional system attributes that are not part
     * of the name. The maximum length for a parameter name, including the full length of the parameter ARN, is 1011
     * characters. For example, the length of the following parameter name is 65 characters, not 20 characters:
     * </p>
     * <p>
     * <code>arn:aws:ssm:us-east-2:111122223333:parameter/ExampleParameterName</code>
     * </p>
     * </note>
     * 
     * @return The fully qualified name of the parameter that you want to add to the system. The fully qualified name
     *         includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must
     *         include a leading forward slash character (/) when you create or reference a parameter. For example:
     *         <code>/Dev/DBServer/MySQL/db-string13</code> </p>
     *         <p>
     *         Naming Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Parameter names are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name must be unique within an AWS Region
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name can't be prefixed with "aws" or "ssm" (case-insensitive).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-/</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name can't include spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Parameter hierarchies are limited to a maximum depth of fifteen levels.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For additional information about valid values for parameter names, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-parameter-name-constraints.html"
     *         >About requirements and constraints for parameter names</a> in the <i>AWS Systems Manager User Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         The maximum length constraint listed below includes capacity for additional system attributes that are
     *         not part of the name. The maximum length for a parameter name, including the full length of the parameter
     *         ARN, is 1011 characters. For example, the length of the following parameter name is 65 characters, not 20
     *         characters:
     *         </p>
     *         <p>
     *         <code>arn:aws:ssm:us-east-2:111122223333:parameter/ExampleParameterName</code>
     *         </p>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Information about the parameter that you want to add to the system. Optional but recommended.
     * </p>
     * <important>
     * <p>
     * Do not enter personally identifiable information in this field.
     * </p>
     * </important>
     * 
     * @return Information about the parameter that you want to add to the system. Optional but recommended.</p>
     *         <important>
     *         <p>
     *         Do not enter personally identifiable information in this field.
     *         </p>
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB. Advanced
     * parameters have a value limit of 8 KB.
     * </p>
     * 
     * @return The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB.
     *         Advanced parameters have a value limit of 8 KB.
     */
    public String value() {
        return value;
    }

    /**
     * <p>
     * The type of parameter that you want to add to the system.
     * </p>
     * <p>
     * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or special
     * character to escape items in the list. If you have a parameter value that requires a comma, then use the
     * <code>String</code> data type.
     * </p>
     * <note>
     * <p>
     * <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China Regions.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter that you want to add to the system.</p>
     *         <p>
     *         Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
     *         special character to escape items in the list. If you have a parameter value that requires a comma, then
     *         use the <code>String</code> data type.
     *         </p>
     *         <note>
     *         <p>
     *         <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
     *         Regions.
     *         </p>
     * @see ParameterType
     */
    public ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter that you want to add to the system.
     * </p>
     * <p>
     * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or special
     * character to escape items in the list. If you have a parameter value that requires a comma, then use the
     * <code>String</code> data type.
     * </p>
     * <note>
     * <p>
     * <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China Regions.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter that you want to add to the system.</p>
     *         <p>
     *         Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
     *         special character to escape items in the list. If you have a parameter value that requires a comma, then
     *         use the <code>String</code> data type.
     *         </p>
     *         <note>
     *         <p>
     *         <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
     *         Regions.
     *         </p>
     * @see ParameterType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The KMS Key ID that you want to use to encrypt a parameter. Either the default AWS Key Management Service (AWS
     * KMS) key automatically assigned to your AWS account or a custom key. Required for parameters that use the
     * <code>SecureString</code> data type.
     * </p>
     * <p>
     * If you don't specify a key ID, the system uses the default key associated with your AWS account.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To use your default AWS KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify the
     * <code>Key ID</code> when you create the parameter. The system automatically populates <code>Key ID</code> with
     * your default KMS key.
     * </p>
     * </li>
     * <li>
     * <p>
     * To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code> parameter.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The KMS Key ID that you want to use to encrypt a parameter. Either the default AWS Key Management Service
     *         (AWS KMS) key automatically assigned to your AWS account or a custom key. Required for parameters that
     *         use the <code>SecureString</code> data type.</p>
     *         <p>
     *         If you don't specify a key ID, the system uses the default key associated with your AWS account.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To use your default AWS KMS key, choose the <code>SecureString</code> data type, and do <i>not</i>
     *         specify the <code>Key ID</code> when you create the parameter. The system automatically populates
     *         <code>Key ID</code> with your default KMS key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
     *         parameter.
     *         </p>
     *         </li>
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Overwrite an existing parameter. If not specified, will default to "false".
     * </p>
     * 
     * @return Overwrite an existing parameter. If not specified, will default to "false".
     */
    public Boolean overwrite() {
        return overwrite;
    }

    /**
     * <p>
     * A regular expression used to validate the parameter value. For example, for String types with values restricted
     * to numbers, you can specify the following: AllowedPattern=^\d+$
     * </p>
     * 
     * @return A regular expression used to validate the parameter value. For example, for String types with values
     *         restricted to numbers, you can specify the following: AllowedPattern=^\d+$
     */
    public String allowedPattern() {
        return allowedPattern;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways, such
     * as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to identify
     * the type of resource to which it applies, the environment, or the type of configuration data referenced by the
     * parameter. In this case, you could specify the following key name/value pairs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Key=Resource,Value=S3bucket</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Key=OS,Value=Windows</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Key=ParameterType,Value=LicenseKey</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
     *         ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
     *         parameter to identify the type of resource to which it applies, the environment, or the type of
     *         configuration data referenced by the parameter. In this case, you could specify the following key
     *         name/value pairs:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Key=Resource,Value=S3bucket</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Key=OS,Value=Windows</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Key=ParameterType,Value=LicenseKey</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
     *         </p>
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The parameter tier to assign to a parameter.
     * </p>
     * <p>
     * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a content
     * size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of 10,000 standard
     * parameters for each Region in an AWS account. Standard parameters are offered at no additional cost.
     * </p>
     * <p>
     * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You can
     * create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced parameters incur a
     * charge. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     * >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced parameter
     * to a standard parameter. Reverting an advanced parameter to a standard parameter would result in data loss
     * because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also remove any
     * policies attached to the parameter. Lastly, advanced parameters use a different form of encryption than standard
     * parameters.
     * </p>
     * <p>
     * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced parameter,
     * you must delete it and recreate it as a new standard parameter.
     * </p>
     * <p>
     * <b>Using the Default Tier Configuration</b>
     * </p>
     * <p>
     * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you specify
     * a tier in the request, Parameter Store creates or updates the parameter according to that request. However, if
     * you do not specify a tier in a request, Parameter Store assigns the tier based on the current Parameter Store
     * default tier configuration.
     * </p>
     * <p>
     * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     * advanced-parameter tier, you can specify one of the following as the default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     * parameter is standard or advanced.
     * </p>
     * <p>
     * If the request doesn't include any options that require an advanced parameter, the parameter is created in the
     * standard-parameter tier. If one or more options requiring an advanced parameter are included in the request,
     * Parameter Store create a parameter in the advanced-parameter tier.
     * </p>
     * <p>
     * This approach helps control your parameter-related costs by always creating standard parameters unless an
     * advanced parameter is necessary.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Options that require an advanced parameter include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The content size of the parameter is more than 4 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The parameter uses a parameter policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * More than 10,000 parameters already exist in your AWS account in the current Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about configuring the default tier option, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
     * parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier to assign to a parameter.</p>
     *         <p>
     *         Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
     *         content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
     *         10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
     *         additional cost.
     *         </p>
     *         <p>
     *         Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
     *         You can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced
     *         parameters incur a charge. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     *         >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
     *         </p>
     *         <p>
     *         You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
     *         parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result
     *         in data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting
     *         would also remove any policies attached to the parameter. Lastly, advanced parameters use a different
     *         form of encryption than standard parameters.
     *         </p>
     *         <p>
     *         If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
     *         parameter, you must delete it and recreate it as a new standard parameter.
     *         </p>
     *         <p>
     *         <b>Using the Default Tier Configuration</b>
     *         </p>
     *         <p>
     *         In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
     *         specify a tier in the request, Parameter Store creates or updates the parameter according to that
     *         request. However, if you do not specify a tier in a request, Parameter Store assigns the tier based on
     *         the current Parameter Store default tier configuration.
     *         </p>
     *         <p>
     *         The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     *         advanced-parameter tier, you can specify one of the following as the default:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     *         parameter is standard or advanced.
     *         </p>
     *         <p>
     *         If the request doesn't include any options that require an advanced parameter, the parameter is created
     *         in the standard-parameter tier. If one or more options requiring an advanced parameter are included in
     *         the request, Parameter Store create a parameter in the advanced-parameter tier.
     *         </p>
     *         <p>
     *         This approach helps control your parameter-related costs by always creating standard parameters unless an
     *         advanced parameter is necessary.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Options that require an advanced parameter include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The content size of the parameter is more than 4 KB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The parameter uses a parameter policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than 10,000 parameters already exist in your AWS account in the current Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about configuring the default tier option, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
     *         default parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see ParameterTier
     */
    public ParameterTier tier() {
        return ParameterTier.fromValue(tier);
    }

    /**
     * <p>
     * The parameter tier to assign to a parameter.
     * </p>
     * <p>
     * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a content
     * size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of 10,000 standard
     * parameters for each Region in an AWS account. Standard parameters are offered at no additional cost.
     * </p>
     * <p>
     * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You can
     * create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced parameters incur a
     * charge. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     * >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced parameter
     * to a standard parameter. Reverting an advanced parameter to a standard parameter would result in data loss
     * because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also remove any
     * policies attached to the parameter. Lastly, advanced parameters use a different form of encryption than standard
     * parameters.
     * </p>
     * <p>
     * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced parameter,
     * you must delete it and recreate it as a new standard parameter.
     * </p>
     * <p>
     * <b>Using the Default Tier Configuration</b>
     * </p>
     * <p>
     * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you specify
     * a tier in the request, Parameter Store creates or updates the parameter according to that request. However, if
     * you do not specify a tier in a request, Parameter Store assigns the tier based on the current Parameter Store
     * default tier configuration.
     * </p>
     * <p>
     * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     * advanced-parameter tier, you can specify one of the following as the default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     * parameter is standard or advanced.
     * </p>
     * <p>
     * If the request doesn't include any options that require an advanced parameter, the parameter is created in the
     * standard-parameter tier. If one or more options requiring an advanced parameter are included in the request,
     * Parameter Store create a parameter in the advanced-parameter tier.
     * </p>
     * <p>
     * This approach helps control your parameter-related costs by always creating standard parameters unless an
     * advanced parameter is necessary.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Options that require an advanced parameter include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The content size of the parameter is more than 4 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The parameter uses a parameter policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * More than 10,000 parameters already exist in your AWS account in the current Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about configuring the default tier option, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
     * parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier to assign to a parameter.</p>
     *         <p>
     *         Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
     *         content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
     *         10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
     *         additional cost.
     *         </p>
     *         <p>
     *         Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
     *         You can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced
     *         parameters incur a charge. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     *         >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
     *         </p>
     *         <p>
     *         You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
     *         parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result
     *         in data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting
     *         would also remove any policies attached to the parameter. Lastly, advanced parameters use a different
     *         form of encryption than standard parameters.
     *         </p>
     *         <p>
     *         If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
     *         parameter, you must delete it and recreate it as a new standard parameter.
     *         </p>
     *         <p>
     *         <b>Using the Default Tier Configuration</b>
     *         </p>
     *         <p>
     *         In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
     *         specify a tier in the request, Parameter Store creates or updates the parameter according to that
     *         request. However, if you do not specify a tier in a request, Parameter Store assigns the tier based on
     *         the current Parameter Store default tier configuration.
     *         </p>
     *         <p>
     *         The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     *         advanced-parameter tier, you can specify one of the following as the default:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     *         parameter is standard or advanced.
     *         </p>
     *         <p>
     *         If the request doesn't include any options that require an advanced parameter, the parameter is created
     *         in the standard-parameter tier. If one or more options requiring an advanced parameter are included in
     *         the request, Parameter Store create a parameter in the advanced-parameter tier.
     *         </p>
     *         <p>
     *         This approach helps control your parameter-related costs by always creating standard parameters unless an
     *         advanced parameter is necessary.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Options that require an advanced parameter include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The content size of the parameter is more than 4 KB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The parameter uses a parameter policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than 10,000 parameters already exist in your AWS account in the current Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about configuring the default tier option, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
     *         default parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see ParameterTier
     */
    public String tierAsString() {
        return tier;
    }

    /**
     * <p>
     * One or more policies to apply to a parameter. This action takes a JSON array. Parameter Store supports the
     * following policy types:
     * </p>
     * <p>
     * Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify the
     * expiration date. You can update the expiration date and time by updating the policy. Updating the
     * <i>parameter</i> does not affect the expiration date and time. When the expiration time is reached, Parameter
     * Store deletes the parameter.
     * </p>
     * <p>
     * ExpirationNotification: This policy triggers an event in Amazon CloudWatch Events that notifies you about the
     * expiration. By using this policy, you can receive notification before or after the expiration time is reached, in
     * units of days or hours.
     * </p>
     * <p>
     * NoChangeNotification: This policy triggers a CloudWatch event if a parameter has not been modified for a
     * specified period of time. This policy type is useful when, for example, a secret needs to be changed within a
     * period of time, but it has not been changed.
     * </p>
     * <p>
     * All existing policies are preserved until you send new policies or an empty policy. For more information about
     * parameter policies, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
     * parameter policies</a>.
     * </p>
     * 
     * @return One or more policies to apply to a parameter. This action takes a JSON array. Parameter Store supports
     *         the following policy types:</p>
     *         <p>
     *         Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify
     *         the expiration date. You can update the expiration date and time by updating the policy. Updating the
     *         <i>parameter</i> does not affect the expiration date and time. When the expiration time is reached,
     *         Parameter Store deletes the parameter.
     *         </p>
     *         <p>
     *         ExpirationNotification: This policy triggers an event in Amazon CloudWatch Events that notifies you about
     *         the expiration. By using this policy, you can receive notification before or after the expiration time is
     *         reached, in units of days or hours.
     *         </p>
     *         <p>
     *         NoChangeNotification: This policy triggers a CloudWatch event if a parameter has not been modified for a
     *         specified period of time. This policy type is useful when, for example, a secret needs to be changed
     *         within a period of time, but it has not been changed.
     *         </p>
     *         <p>
     *         All existing policies are preserved until you send new policies or an empty policy. For more information
     *         about parameter policies, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html"
     *         >Assigning parameter policies</a>.
     */
    public String policies() {
        return policies;
    }

    /**
     * <p>
     * The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon Machine
     * Image IDs.
     * </p>
     * <p>
     * <b>The following data type values are supported.</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>text</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>aws:ec2:image</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Systems Manager validates
     * the parameter value is in the required format, such as <code>ami-12345abcdeEXAMPLE</code>, and that the specified
     * AMI is available in your AWS account. For more information, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html">Native
     * parameter support for Amazon Machine Image IDs</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * 
     * @return The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon
     *         Machine Image IDs.</p>
     *         <p>
     *         <b>The following data type values are supported.</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>text</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>aws:ec2:image</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Systems Manager
     *         validates the parameter value is in the required format, such as <code>ami-12345abcdeEXAMPLE</code>, and
     *         that the specified AMI is available in your AWS account. For more information, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html"
     *         >Native parameter support for Amazon Machine Image IDs</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(overwrite());
        hashCode = 31 * hashCode + Objects.hashCode(allowedPattern());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutParameterRequest)) {
            return false;
        }
        PutParameterRequest other = (PutParameterRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(value(), other.value()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(keyId(), other.keyId()) && Objects.equals(overwrite(), other.overwrite())
                && Objects.equals(allowedPattern(), other.allowedPattern()) && Objects.equals(tags(), other.tags())
                && Objects.equals(tierAsString(), other.tierAsString()) && Objects.equals(policies(), other.policies())
                && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutParameterRequest").add("Name", name()).add("Description", description())
                .add("Value", value()).add("Type", typeAsString()).add("KeyId", keyId()).add("Overwrite", overwrite())
                .add("AllowedPattern", allowedPattern()).add("Tags", tags()).add("Tier", tierAsString())
                .add("Policies", policies()).add("DataType", dataType()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "Overwrite":
            return Optional.ofNullable(clazz.cast(overwrite()));
        case "AllowedPattern":
            return Optional.ofNullable(clazz.cast(allowedPattern()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutParameterRequest, T> g) {
        return obj -> g.apply((PutParameterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutParameterRequest> {
        /**
         * <p>
         * The fully qualified name of the parameter that you want to add to the system. The fully qualified name
         * includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must
         * include a leading forward slash character (/) when you create or reference a parameter. For example:
         * <code>/Dev/DBServer/MySQL/db-string13</code>
         * </p>
         * <p>
         * Naming Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Parameter names are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name must be unique within an AWS Region
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name can't be prefixed with "aws" or "ssm" (case-insensitive).
         * </p>
         * </li>
         * <li>
         * <p>
         * Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-/</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name can't include spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * Parameter hierarchies are limited to a maximum depth of fifteen levels.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For additional information about valid values for parameter names, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-parameter-name-constraints.html"
         * >About requirements and constraints for parameter names</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <note>
         * <p>
         * The maximum length constraint listed below includes capacity for additional system attributes that are not
         * part of the name. The maximum length for a parameter name, including the full length of the parameter ARN, is
         * 1011 characters. For example, the length of the following parameter name is 65 characters, not 20 characters:
         * </p>
         * <p>
         * <code>arn:aws:ssm:us-east-2:111122223333:parameter/ExampleParameterName</code>
         * </p>
         * </note>
         * 
         * @param name
         *        The fully qualified name of the parameter that you want to add to the system. The fully qualified name
         *        includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you
         *        must include a leading forward slash character (/) when you create or reference a parameter. For
         *        example: <code>/Dev/DBServer/MySQL/db-string13</code> </p>
         *        <p>
         *        Naming Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Parameter names are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name must be unique within an AWS Region
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name can't be prefixed with "aws" or "ssm" (case-insensitive).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-/</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name can't include spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Parameter hierarchies are limited to a maximum depth of fifteen levels.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For additional information about valid values for parameter names, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-parameter-name-constraints.html"
         *        >About requirements and constraints for parameter names</a> in the <i>AWS Systems Manager User
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        The maximum length constraint listed below includes capacity for additional system attributes that are
         *        not part of the name. The maximum length for a parameter name, including the full length of the
         *        parameter ARN, is 1011 characters. For example, the length of the following parameter name is 65
         *        characters, not 20 characters:
         *        </p>
         *        <p>
         *        <code>arn:aws:ssm:us-east-2:111122223333:parameter/ExampleParameterName</code>
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Information about the parameter that you want to add to the system. Optional but recommended.
         * </p>
         * <important>
         * <p>
         * Do not enter personally identifiable information in this field.
         * </p>
         * </important>
         * 
         * @param description
         *        Information about the parameter that you want to add to the system. Optional but recommended.</p>
         *        <important>
         *        <p>
         *        Do not enter personally identifiable information in this field.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB.
         * Advanced parameters have a value limit of 8 KB.
         * </p>
         * 
         * @param value
         *        The parameter value that you want to add to the system. Standard parameters have a value limit of 4
         *        KB. Advanced parameters have a value limit of 8 KB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The type of parameter that you want to add to the system.
         * </p>
         * <p>
         * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
         * special character to escape items in the list. If you have a parameter value that requires a comma, then use
         * the <code>String</code> data type.
         * </p>
         * <note>
         * <p>
         * <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
         * Regions.
         * </p>
         * </note>
         * 
         * @param type
         *        The type of parameter that you want to add to the system.</p>
         *        <p>
         *        Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation
         *        or special character to escape items in the list. If you have a parameter value that requires a comma,
         *        then use the <code>String</code> data type.
         *        </p>
         *        <note>
         *        <p>
         *        <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
         *        Regions.
         *        </p>
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter that you want to add to the system.
         * </p>
         * <p>
         * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
         * special character to escape items in the list. If you have a parameter value that requires a comma, then use
         * the <code>String</code> data type.
         * </p>
         * <note>
         * <p>
         * <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
         * Regions.
         * </p>
         * </note>
         * 
         * @param type
         *        The type of parameter that you want to add to the system.</p>
         *        <p>
         *        Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation
         *        or special character to escape items in the list. If you have a parameter value that requires a comma,
         *        then use the <code>String</code> data type.
         *        </p>
         *        <note>
         *        <p>
         *        <code>SecureString</code> is not currently supported for AWS CloudFormation templates or in the China
         *        Regions.
         *        </p>
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The KMS Key ID that you want to use to encrypt a parameter. Either the default AWS Key Management Service
         * (AWS KMS) key automatically assigned to your AWS account or a custom key. Required for parameters that use
         * the <code>SecureString</code> data type.
         * </p>
         * <p>
         * If you don't specify a key ID, the system uses the default key associated with your AWS account.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To use your default AWS KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify
         * the <code>Key ID</code> when you create the parameter. The system automatically populates <code>Key ID</code>
         * with your default KMS key.
         * </p>
         * </li>
         * <li>
         * <p>
         * To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
         * parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyId
         *        The KMS Key ID that you want to use to encrypt a parameter. Either the default AWS Key Management
         *        Service (AWS KMS) key automatically assigned to your AWS account or a custom key. Required for
         *        parameters that use the <code>SecureString</code> data type.</p>
         *        <p>
         *        If you don't specify a key ID, the system uses the default key associated with your AWS account.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To use your default AWS KMS key, choose the <code>SecureString</code> data type, and do <i>not</i>
         *        specify the <code>Key ID</code> when you create the parameter. The system automatically populates
         *        <code>Key ID</code> with your default KMS key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
         *        parameter.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Overwrite an existing parameter. If not specified, will default to "false".
         * </p>
         * 
         * @param overwrite
         *        Overwrite an existing parameter. If not specified, will default to "false".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overwrite(Boolean overwrite);

        /**
         * <p>
         * A regular expression used to validate the parameter value. For example, for String types with values
         * restricted to numbers, you can specify the following: AllowedPattern=^\d+$
         * </p>
         * 
         * @param allowedPattern
         *        A regular expression used to validate the parameter value. For example, for String types with values
         *        restricted to numbers, you can specify the following: AllowedPattern=^\d+$
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedPattern(String allowedPattern);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key name/value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
         *        parameter to identify the type of resource to which it applies, the environment, or the type of
         *        configuration data referenced by the parameter. In this case, you could specify the following key
         *        name/value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=Resource,Value=S3bucket</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=ParameterType,Value=LicenseKey</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key name/value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
         *        parameter to identify the type of resource to which it applies, the environment, or the type of
         *        configuration data referenced by the parameter. In this case, you could specify the following key
         *        name/value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=Resource,Value=S3bucket</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=ParameterType,Value=LicenseKey</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key name/value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The parameter tier to assign to a parameter.
         * </p>
         * <p>
         * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         * content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
         * 10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
         * additional cost.
         * </p>
         * <p>
         * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You
         * can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced parameters
         * incur a charge. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         * >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <p>
         * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
         * parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result in
         * data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also
         * remove any policies attached to the parameter. Lastly, advanced parameters use a different form of encryption
         * than standard parameters.
         * </p>
         * <p>
         * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         * parameter, you must delete it and recreate it as a new standard parameter.
         * </p>
         * <p>
         * <b>Using the Default Tier Configuration</b>
         * </p>
         * <p>
         * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
         * specify a tier in the request, Parameter Store creates or updates the parameter according to that request.
         * However, if you do not specify a tier in a request, Parameter Store assigns the tier based on the current
         * Parameter Store default tier configuration.
         * </p>
         * <p>
         * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         * advanced-parameter tier, you can specify one of the following as the default:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
         * parameter is standard or advanced.
         * </p>
         * <p>
         * If the request doesn't include any options that require an advanced parameter, the parameter is created in
         * the standard-parameter tier. If one or more options requiring an advanced parameter are included in the
         * request, Parameter Store create a parameter in the advanced-parameter tier.
         * </p>
         * <p>
         * This approach helps control your parameter-related costs by always creating standard parameters unless an
         * advanced parameter is necessary.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Options that require an advanced parameter include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The content size of the parameter is more than 4 KB.
         * </p>
         * </li>
         * <li>
         * <p>
         * The parameter uses a parameter policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * More than 10,000 parameters already exist in your AWS account in the current Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about configuring the default tier option, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
         * parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param tier
         *        The parameter tier to assign to a parameter.</p>
         *        <p>
         *        Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         *        content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum
         *        of 10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
         *        additional cost.
         *        </p>
         *        <p>
         *        Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
         *        You can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced
         *        parameters incur a charge. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         *        >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
         *        </p>
         *        <p>
         *        You can change a standard parameter to an advanced parameter any time. But you can't revert an
         *        advanced parameter to a standard parameter. Reverting an advanced parameter to a standard parameter
         *        would result in data loss because the system would truncate the size of the parameter from 8 KB to 4
         *        KB. Reverting would also remove any policies attached to the parameter. Lastly, advanced parameters
         *        use a different form of encryption than standard parameters.
         *        </p>
         *        <p>
         *        If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         *        parameter, you must delete it and recreate it as a new standard parameter.
         *        </p>
         *        <p>
         *        <b>Using the Default Tier Configuration</b>
         *        </p>
         *        <p>
         *        In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever
         *        you specify a tier in the request, Parameter Store creates or updates the parameter according to that
         *        request. However, if you do not specify a tier in a request, Parameter Store assigns the tier based on
         *        the current Parameter Store default tier configuration.
         *        </p>
         *        <p>
         *        The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         *        advanced-parameter tier, you can specify one of the following as the default:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if
         *        the parameter is standard or advanced.
         *        </p>
         *        <p>
         *        If the request doesn't include any options that require an advanced parameter, the parameter is
         *        created in the standard-parameter tier. If one or more options requiring an advanced parameter are
         *        included in the request, Parameter Store create a parameter in the advanced-parameter tier.
         *        </p>
         *        <p>
         *        This approach helps control your parameter-related costs by always creating standard parameters unless
         *        an advanced parameter is necessary.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Options that require an advanced parameter include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The content size of the parameter is more than 4 KB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The parameter uses a parameter policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        More than 10,000 parameters already exist in your AWS account in the current Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about configuring the default tier option, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
         *        default parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The parameter tier to assign to a parameter.
         * </p>
         * <p>
         * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         * content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
         * 10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
         * additional cost.
         * </p>
         * <p>
         * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You
         * can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced parameters
         * incur a charge. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         * >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <p>
         * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
         * parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result in
         * data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also
         * remove any policies attached to the parameter. Lastly, advanced parameters use a different form of encryption
         * than standard parameters.
         * </p>
         * <p>
         * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         * parameter, you must delete it and recreate it as a new standard parameter.
         * </p>
         * <p>
         * <b>Using the Default Tier Configuration</b>
         * </p>
         * <p>
         * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
         * specify a tier in the request, Parameter Store creates or updates the parameter according to that request.
         * However, if you do not specify a tier in a request, Parameter Store assigns the tier based on the current
         * Parameter Store default tier configuration.
         * </p>
         * <p>
         * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         * advanced-parameter tier, you can specify one of the following as the default:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
         * parameter is standard or advanced.
         * </p>
         * <p>
         * If the request doesn't include any options that require an advanced parameter, the parameter is created in
         * the standard-parameter tier. If one or more options requiring an advanced parameter are included in the
         * request, Parameter Store create a parameter in the advanced-parameter tier.
         * </p>
         * <p>
         * This approach helps control your parameter-related costs by always creating standard parameters unless an
         * advanced parameter is necessary.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Options that require an advanced parameter include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The content size of the parameter is more than 4 KB.
         * </p>
         * </li>
         * <li>
         * <p>
         * The parameter uses a parameter policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * More than 10,000 parameters already exist in your AWS account in the current Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about configuring the default tier option, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
         * parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param tier
         *        The parameter tier to assign to a parameter.</p>
         *        <p>
         *        Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         *        content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum
         *        of 10,000 standard parameters for each Region in an AWS account. Standard parameters are offered at no
         *        additional cost.
         *        </p>
         *        <p>
         *        Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
         *        You can create a maximum of 100,000 advanced parameters for each Region in an AWS account. Advanced
         *        parameters incur a charge. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         *        >Standard and advanced parameter tiers</a> in the <i>AWS Systems Manager User Guide</i>.
         *        </p>
         *        <p>
         *        You can change a standard parameter to an advanced parameter any time. But you can't revert an
         *        advanced parameter to a standard parameter. Reverting an advanced parameter to a standard parameter
         *        would result in data loss because the system would truncate the size of the parameter from 8 KB to 4
         *        KB. Reverting would also remove any policies attached to the parameter. Lastly, advanced parameters
         *        use a different form of encryption than standard parameters.
         *        </p>
         *        <p>
         *        If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         *        parameter, you must delete it and recreate it as a new standard parameter.
         *        </p>
         *        <p>
         *        <b>Using the Default Tier Configuration</b>
         *        </p>
         *        <p>
         *        In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever
         *        you specify a tier in the request, Parameter Store creates or updates the parameter according to that
         *        request. However, if you do not specify a tier in a request, Parameter Store assigns the tier based on
         *        the current Parameter Store default tier configuration.
         *        </p>
         *        <p>
         *        The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         *        advanced-parameter tier, you can specify one of the following as the default:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if
         *        the parameter is standard or advanced.
         *        </p>
         *        <p>
         *        If the request doesn't include any options that require an advanced parameter, the parameter is
         *        created in the standard-parameter tier. If one or more options requiring an advanced parameter are
         *        included in the request, Parameter Store create a parameter in the advanced-parameter tier.
         *        </p>
         *        <p>
         *        This approach helps control your parameter-related costs by always creating standard parameters unless
         *        an advanced parameter is necessary.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Options that require an advanced parameter include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The content size of the parameter is more than 4 KB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The parameter uses a parameter policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        More than 10,000 parameters already exist in your AWS account in the current Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about configuring the default tier option, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
         *        default parameter tier</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(ParameterTier tier);

        /**
         * <p>
         * One or more policies to apply to a parameter. This action takes a JSON array. Parameter Store supports the
         * following policy types:
         * </p>
         * <p>
         * Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify the
         * expiration date. You can update the expiration date and time by updating the policy. Updating the
         * <i>parameter</i> does not affect the expiration date and time. When the expiration time is reached, Parameter
         * Store deletes the parameter.
         * </p>
         * <p>
         * ExpirationNotification: This policy triggers an event in Amazon CloudWatch Events that notifies you about the
         * expiration. By using this policy, you can receive notification before or after the expiration time is
         * reached, in units of days or hours.
         * </p>
         * <p>
         * NoChangeNotification: This policy triggers a CloudWatch event if a parameter has not been modified for a
         * specified period of time. This policy type is useful when, for example, a secret needs to be changed within a
         * period of time, but it has not been changed.
         * </p>
         * <p>
         * All existing policies are preserved until you send new policies or an empty policy. For more information
         * about parameter policies, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
         * parameter policies</a>.
         * </p>
         * 
         * @param policies
         *        One or more policies to apply to a parameter. This action takes a JSON array. Parameter Store supports
         *        the following policy types:</p>
         *        <p>
         *        Expiration: This policy deletes the parameter after it expires. When you create the policy, you
         *        specify the expiration date. You can update the expiration date and time by updating the policy.
         *        Updating the <i>parameter</i> does not affect the expiration date and time. When the expiration time
         *        is reached, Parameter Store deletes the parameter.
         *        </p>
         *        <p>
         *        ExpirationNotification: This policy triggers an event in Amazon CloudWatch Events that notifies you
         *        about the expiration. By using this policy, you can receive notification before or after the
         *        expiration time is reached, in units of days or hours.
         *        </p>
         *        <p>
         *        NoChangeNotification: This policy triggers a CloudWatch event if a parameter has not been modified for
         *        a specified period of time. This policy type is useful when, for example, a secret needs to be changed
         *        within a period of time, but it has not been changed.
         *        </p>
         *        <p>
         *        All existing policies are preserved until you send new policies or an empty policy. For more
         *        information about parameter policies, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html"
         *        >Assigning parameter policies</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(String policies);

        /**
         * <p>
         * The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon Machine
         * Image IDs.
         * </p>
         * <p>
         * <b>The following data type values are supported.</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>text</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws:ec2:image</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Systems Manager
         * validates the parameter value is in the required format, such as <code>ami-12345abcdeEXAMPLE</code>, and that
         * the specified AMI is available in your AWS account. For more information, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html">Native
         * parameter support for Amazon Machine Image IDs</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param dataType
         *        The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon
         *        Machine Image IDs.</p>
         *        <p>
         *        <b>The following data type values are supported.</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>text</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws:ec2:image</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Systems
         *        Manager validates the parameter value is in the required format, such as
         *        <code>ami-12345abcdeEXAMPLE</code>, and that the specified AMI is available in your AWS account. For
         *        more information, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html"
         *        >Native parameter support for Amazon Machine Image IDs</a> in the <i>AWS Systems Manager User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private String value;

        private String type;

        private String keyId;

        private Boolean overwrite;

        private String allowedPattern;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String tier;

        private String policies;

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(PutParameterRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            value(model.value);
            type(model.type);
            keyId(model.keyId);
            overwrite(model.overwrite);
            allowedPattern(model.allowedPattern);
            tags(model.tags);
            tier(model.tier);
            policies(model.policies);
            dataType(model.dataType);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getValue() {
            return value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final Boolean getOverwrite() {
            return overwrite;
        }

        @Override
        public final Builder overwrite(Boolean overwrite) {
            this.overwrite = overwrite;
            return this;
        }

        public final void setOverwrite(Boolean overwrite) {
            this.overwrite = overwrite;
        }

        public final String getAllowedPattern() {
            return allowedPattern;
        }

        @Override
        public final Builder allowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
            return this;
        }

        public final void setAllowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getTier() {
            return tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(ParameterTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        public final String getPolicies() {
            return policies;
        }

        @Override
        public final Builder policies(String policies) {
            this.policies = policies;
            return this;
        }

        public final void setPolicies(String policies) {
            this.policies = policies;
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutParameterRequest build() {
            return new PutParameterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
