/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the high-level patch compliance state for a managed instance, providing information about the number of
 * installed, missing, not applicable, and failed patches along with metadata about the operation when this information
 * was gathered for the instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstancePatchState implements SdkPojo, Serializable,
        ToCopyableBuilder<InstancePatchState.Builder, InstancePatchState> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> PATCH_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::patchGroup)).setter(setter(Builder::patchGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PatchGroup").build()).build();

    private static final SdkField<String> BASELINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::baselineId)).setter(setter(Builder::baselineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaselineId").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::snapshotId)).setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId").build()).build();

    private static final SdkField<String> INSTALL_OVERRIDE_LIST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::installOverrideList)).setter(setter(Builder::installOverrideList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstallOverrideList").build())
            .build();

    private static final SdkField<String> OWNER_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::ownerInformation)).setter(setter(Builder::ownerInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerInformation").build()).build();

    private static final SdkField<Integer> INSTALLED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::installedCount)).setter(setter(Builder::installedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledCount").build()).build();

    private static final SdkField<Integer> INSTALLED_OTHER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::installedOtherCount)).setter(setter(Builder::installedOtherCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledOtherCount").build())
            .build();

    private static final SdkField<Integer> INSTALLED_PENDING_REBOOT_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::installedPendingRebootCount))
            .setter(setter(Builder::installedPendingRebootCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledPendingRebootCount")
                    .build()).build();

    private static final SdkField<Integer> INSTALLED_REJECTED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::installedRejectedCount)).setter(setter(Builder::installedRejectedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledRejectedCount").build())
            .build();

    private static final SdkField<Integer> MISSING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::missingCount)).setter(setter(Builder::missingCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MissingCount").build()).build();

    private static final SdkField<Integer> FAILED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::failedCount)).setter(setter(Builder::failedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedCount").build()).build();

    private static final SdkField<Integer> UNREPORTED_NOT_APPLICABLE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::unreportedNotApplicableCount))
            .setter(setter(Builder::unreportedNotApplicableCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnreportedNotApplicableCount")
                    .build()).build();

    private static final SdkField<Integer> NOT_APPLICABLE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstancePatchState::notApplicableCount)).setter(setter(Builder::notApplicableCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotApplicableCount").build())
            .build();

    private static final SdkField<Instant> OPERATION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstancePatchState::operationStartTime)).setter(setter(Builder::operationStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationStartTime").build())
            .build();

    private static final SdkField<Instant> OPERATION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstancePatchState::operationEndTime)).setter(setter(Builder::operationEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationEndTime").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::operationAsString)).setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation").build()).build();

    private static final SdkField<Instant> LAST_NO_REBOOT_INSTALL_OPERATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstancePatchState::lastNoRebootInstallOperationTime))
            .setter(setter(Builder::lastNoRebootInstallOperationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastNoRebootInstallOperationTime")
                    .build()).build();

    private static final SdkField<String> REBOOT_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstancePatchState::rebootOptionAsString)).setter(setter(Builder::rebootOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RebootOption").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            PATCH_GROUP_FIELD, BASELINE_ID_FIELD, SNAPSHOT_ID_FIELD, INSTALL_OVERRIDE_LIST_FIELD, OWNER_INFORMATION_FIELD,
            INSTALLED_COUNT_FIELD, INSTALLED_OTHER_COUNT_FIELD, INSTALLED_PENDING_REBOOT_COUNT_FIELD,
            INSTALLED_REJECTED_COUNT_FIELD, MISSING_COUNT_FIELD, FAILED_COUNT_FIELD, UNREPORTED_NOT_APPLICABLE_COUNT_FIELD,
            NOT_APPLICABLE_COUNT_FIELD, OPERATION_START_TIME_FIELD, OPERATION_END_TIME_FIELD, OPERATION_FIELD,
            LAST_NO_REBOOT_INSTALL_OPERATION_TIME_FIELD, REBOOT_OPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final String patchGroup;

    private final String baselineId;

    private final String snapshotId;

    private final String installOverrideList;

    private final String ownerInformation;

    private final Integer installedCount;

    private final Integer installedOtherCount;

    private final Integer installedPendingRebootCount;

    private final Integer installedRejectedCount;

    private final Integer missingCount;

    private final Integer failedCount;

    private final Integer unreportedNotApplicableCount;

    private final Integer notApplicableCount;

    private final Instant operationStartTime;

    private final Instant operationEndTime;

    private final String operation;

    private final Instant lastNoRebootInstallOperationTime;

    private final String rebootOption;

    private InstancePatchState(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.patchGroup = builder.patchGroup;
        this.baselineId = builder.baselineId;
        this.snapshotId = builder.snapshotId;
        this.installOverrideList = builder.installOverrideList;
        this.ownerInformation = builder.ownerInformation;
        this.installedCount = builder.installedCount;
        this.installedOtherCount = builder.installedOtherCount;
        this.installedPendingRebootCount = builder.installedPendingRebootCount;
        this.installedRejectedCount = builder.installedRejectedCount;
        this.missingCount = builder.missingCount;
        this.failedCount = builder.failedCount;
        this.unreportedNotApplicableCount = builder.unreportedNotApplicableCount;
        this.notApplicableCount = builder.notApplicableCount;
        this.operationStartTime = builder.operationStartTime;
        this.operationEndTime = builder.operationEndTime;
        this.operation = builder.operation;
        this.lastNoRebootInstallOperationTime = builder.lastNoRebootInstallOperationTime;
        this.rebootOption = builder.rebootOption;
    }

    /**
     * <p>
     * The ID of the managed instance the high-level patch compliance information was collected for.
     * </p>
     * 
     * @return The ID of the managed instance the high-level patch compliance information was collected for.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The name of the patch group the managed instance belongs to.
     * </p>
     * 
     * @return The name of the patch group the managed instance belongs to.
     */
    public String patchGroup() {
        return patchGroup;
    }

    /**
     * <p>
     * The ID of the patch baseline used to patch the instance.
     * </p>
     * 
     * @return The ID of the patch baseline used to patch the instance.
     */
    public String baselineId() {
        return baselineId;
    }

    /**
     * <p>
     * The ID of the patch baseline snapshot used during the patching operation when this compliance data was collected.
     * </p>
     * 
     * @return The ID of the patch baseline snapshot used during the patching operation when this compliance data was
     *         collected.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * An https URL or an Amazon S3 path-style URL to a list of patches to be installed. This patch installation list,
     * which you maintain in an Amazon S3 bucket in YAML format and specify in the SSM document
     * <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch baseline.
     * </p>
     * <p>
     * For more information about the <code>InstallOverrideList</code> parameter, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html">About
     * the SSM Document AWS-RunPatchBaseline</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * 
     * @return An https URL or an Amazon S3 path-style URL to a list of patches to be installed. This patch installation
     *         list, which you maintain in an Amazon S3 bucket in YAML format and specify in the SSM document
     *         <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch baseline.</p>
     *         <p>
     *         For more information about the <code>InstallOverrideList</code> parameter, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
     *         >About the SSM Document AWS-RunPatchBaseline</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public String installOverrideList() {
        return installOverrideList;
    }

    /**
     * <p>
     * Placeholder information. This field will always be empty in the current release of the service.
     * </p>
     * 
     * @return Placeholder information. This field will always be empty in the current release of the service.
     */
    public String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are installed on the instance.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are installed on the instance.
     */
    public Integer installedCount() {
        return installedCount;
    }

    /**
     * <p>
     * The number of patches not specified in the patch baseline that are installed on the instance.
     * </p>
     * 
     * @return The number of patches not specified in the patch baseline that are installed on the instance.
     */
    public Integer installedOtherCount() {
        return installedOtherCount;
    }

    /**
     * <p>
     * The number of patches installed by Patch Manager since the last time the instance was rebooted.
     * </p>
     * 
     * @return The number of patches installed by Patch Manager since the last time the instance was rebooted.
     */
    public Integer installedPendingRebootCount() {
        return installedPendingRebootCount;
    }

    /**
     * <p>
     * The number of instances with patches installed that are specified in a RejectedPatches list. Patches with a
     * status of <i>InstalledRejected</i> were typically installed before they were added to a RejectedPatches list.
     * </p>
     * <note>
     * <p>
     * If ALLOW_AS_DEPENDENCY is the specified option for RejectedPatchesAction, the value of InstalledRejectedCount
     * will always be 0 (zero).
     * </p>
     * </note>
     * 
     * @return The number of instances with patches installed that are specified in a RejectedPatches list. Patches with
     *         a status of <i>InstalledRejected</i> were typically installed before they were added to a RejectedPatches
     *         list.</p> <note>
     *         <p>
     *         If ALLOW_AS_DEPENDENCY is the specified option for RejectedPatchesAction, the value of
     *         InstalledRejectedCount will always be 0 (zero).
     *         </p>
     */
    public Integer installedRejectedCount() {
        return installedRejectedCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are applicable for the instance but aren't currently
     * installed.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are applicable for the instance but aren't currently
     *         installed.
     */
    public Integer missingCount() {
        return missingCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that were attempted to be installed during the last patching
     * operation, but failed to install.
     * </p>
     * 
     * @return The number of patches from the patch baseline that were attempted to be installed during the last
     *         patching operation, but failed to install.
     */
    public Integer failedCount() {
        return failedCount;
    }

    /**
     * <p>
     * The number of patches beyond the supported limit of <code>NotApplicableCount</code> that are not reported by name
     * to Systems Manager Inventory.
     * </p>
     * 
     * @return The number of patches beyond the supported limit of <code>NotApplicableCount</code> that are not reported
     *         by name to Systems Manager Inventory.
     */
    public Integer unreportedNotApplicableCount() {
        return unreportedNotApplicableCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that aren't applicable for the instance and therefore aren't
     * installed on the instance. This number may be truncated if the list of patch names is very large. The number of
     * patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
     * </p>
     * 
     * @return The number of patches from the patch baseline that aren't applicable for the instance and therefore
     *         aren't installed on the instance. This number may be truncated if the list of patch names is very large.
     *         The number of patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
     */
    public Integer notApplicableCount() {
        return notApplicableCount;
    }

    /**
     * <p>
     * The time the most recent patching operation was started on the instance.
     * </p>
     * 
     * @return The time the most recent patching operation was started on the instance.
     */
    public Instant operationStartTime() {
        return operationStartTime;
    }

    /**
     * <p>
     * The time the most recent patching operation completed on the instance.
     * </p>
     * 
     * @return The time the most recent patching operation completed on the instance.
     */
    public Instant operationEndTime() {
        return operationEndTime;
    }

    /**
     * <p>
     * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
     * missing patches).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
     *         (install missing patches).
     * @see PatchOperationType
     */
    public PatchOperationType operation() {
        return PatchOperationType.fromValue(operation);
    }

    /**
     * <p>
     * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
     * missing patches).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
     *         (install missing patches).
     * @see PatchOperationType
     */
    public String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The time of the last attempt to patch the instance with <code>NoReboot</code> specified as the reboot option.
     * </p>
     * 
     * @return The time of the last attempt to patch the instance with <code>NoReboot</code> specified as the reboot
     *         option.
     */
    public Instant lastNoRebootInstallOperationTime() {
        return lastNoRebootInstallOperationTime;
    }

    /**
     * <p>
     * Indicates the reboot option specified in the patch baseline.
     * </p>
     * <note>
     * <p>
     * Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
     * <code>Scan</code> operations.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any patches
     * are detected with a status of <code>InstalledPendingReboot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system. Patches
     * installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These patches might not
     * be in effect until a reboot is performed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rebootOption} will
     * return {@link RebootOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rebootOptionAsString}.
     * </p>
     * 
     * @return Indicates the reboot option specified in the patch baseline.</p> <note>
     *         <p>
     *         Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
     *         <code>Scan</code> operations.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any
     *         patches are detected with a status of <code>InstalledPendingReboot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system.
     *         Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These
     *         patches might not be in effect until a reboot is performed.
     *         </p>
     *         </li>
     * @see RebootOption
     */
    public RebootOption rebootOption() {
        return RebootOption.fromValue(rebootOption);
    }

    /**
     * <p>
     * Indicates the reboot option specified in the patch baseline.
     * </p>
     * <note>
     * <p>
     * Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
     * <code>Scan</code> operations.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any patches
     * are detected with a status of <code>InstalledPendingReboot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system. Patches
     * installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These patches might not
     * be in effect until a reboot is performed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rebootOption} will
     * return {@link RebootOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rebootOptionAsString}.
     * </p>
     * 
     * @return Indicates the reboot option specified in the patch baseline.</p> <note>
     *         <p>
     *         Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
     *         <code>Scan</code> operations.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any
     *         patches are detected with a status of <code>InstalledPendingReboot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system.
     *         Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These
     *         patches might not be in effect until a reboot is performed.
     *         </p>
     *         </li>
     * @see RebootOption
     */
    public String rebootOptionAsString() {
        return rebootOption;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(patchGroup());
        hashCode = 31 * hashCode + Objects.hashCode(baselineId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(installOverrideList());
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(installedCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedOtherCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedPendingRebootCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedRejectedCount());
        hashCode = 31 * hashCode + Objects.hashCode(missingCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedCount());
        hashCode = 31 * hashCode + Objects.hashCode(unreportedNotApplicableCount());
        hashCode = 31 * hashCode + Objects.hashCode(notApplicableCount());
        hashCode = 31 * hashCode + Objects.hashCode(operationStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastNoRebootInstallOperationTime());
        hashCode = 31 * hashCode + Objects.hashCode(rebootOptionAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstancePatchState)) {
            return false;
        }
        InstancePatchState other = (InstancePatchState) obj;
        return Objects.equals(instanceId(), other.instanceId()) && Objects.equals(patchGroup(), other.patchGroup())
                && Objects.equals(baselineId(), other.baselineId()) && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(installOverrideList(), other.installOverrideList())
                && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(installedCount(), other.installedCount())
                && Objects.equals(installedOtherCount(), other.installedOtherCount())
                && Objects.equals(installedPendingRebootCount(), other.installedPendingRebootCount())
                && Objects.equals(installedRejectedCount(), other.installedRejectedCount())
                && Objects.equals(missingCount(), other.missingCount()) && Objects.equals(failedCount(), other.failedCount())
                && Objects.equals(unreportedNotApplicableCount(), other.unreportedNotApplicableCount())
                && Objects.equals(notApplicableCount(), other.notApplicableCount())
                && Objects.equals(operationStartTime(), other.operationStartTime())
                && Objects.equals(operationEndTime(), other.operationEndTime())
                && Objects.equals(operationAsString(), other.operationAsString())
                && Objects.equals(lastNoRebootInstallOperationTime(), other.lastNoRebootInstallOperationTime())
                && Objects.equals(rebootOptionAsString(), other.rebootOptionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstancePatchState").add("InstanceId", instanceId()).add("PatchGroup", patchGroup())
                .add("BaselineId", baselineId()).add("SnapshotId", snapshotId())
                .add("InstallOverrideList", installOverrideList())
                .add("OwnerInformation", ownerInformation() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InstalledCount", installedCount()).add("InstalledOtherCount", installedOtherCount())
                .add("InstalledPendingRebootCount", installedPendingRebootCount())
                .add("InstalledRejectedCount", installedRejectedCount()).add("MissingCount", missingCount())
                .add("FailedCount", failedCount()).add("UnreportedNotApplicableCount", unreportedNotApplicableCount())
                .add("NotApplicableCount", notApplicableCount()).add("OperationStartTime", operationStartTime())
                .add("OperationEndTime", operationEndTime()).add("Operation", operationAsString())
                .add("LastNoRebootInstallOperationTime", lastNoRebootInstallOperationTime())
                .add("RebootOption", rebootOptionAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "PatchGroup":
            return Optional.ofNullable(clazz.cast(patchGroup()));
        case "BaselineId":
            return Optional.ofNullable(clazz.cast(baselineId()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "InstallOverrideList":
            return Optional.ofNullable(clazz.cast(installOverrideList()));
        case "OwnerInformation":
            return Optional.ofNullable(clazz.cast(ownerInformation()));
        case "InstalledCount":
            return Optional.ofNullable(clazz.cast(installedCount()));
        case "InstalledOtherCount":
            return Optional.ofNullable(clazz.cast(installedOtherCount()));
        case "InstalledPendingRebootCount":
            return Optional.ofNullable(clazz.cast(installedPendingRebootCount()));
        case "InstalledRejectedCount":
            return Optional.ofNullable(clazz.cast(installedRejectedCount()));
        case "MissingCount":
            return Optional.ofNullable(clazz.cast(missingCount()));
        case "FailedCount":
            return Optional.ofNullable(clazz.cast(failedCount()));
        case "UnreportedNotApplicableCount":
            return Optional.ofNullable(clazz.cast(unreportedNotApplicableCount()));
        case "NotApplicableCount":
            return Optional.ofNullable(clazz.cast(notApplicableCount()));
        case "OperationStartTime":
            return Optional.ofNullable(clazz.cast(operationStartTime()));
        case "OperationEndTime":
            return Optional.ofNullable(clazz.cast(operationEndTime()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "LastNoRebootInstallOperationTime":
            return Optional.ofNullable(clazz.cast(lastNoRebootInstallOperationTime()));
        case "RebootOption":
            return Optional.ofNullable(clazz.cast(rebootOptionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstancePatchState, T> g) {
        return obj -> g.apply((InstancePatchState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstancePatchState> {
        /**
         * <p>
         * The ID of the managed instance the high-level patch compliance information was collected for.
         * </p>
         * 
         * @param instanceId
         *        The ID of the managed instance the high-level patch compliance information was collected for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The name of the patch group the managed instance belongs to.
         * </p>
         * 
         * @param patchGroup
         *        The name of the patch group the managed instance belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patchGroup(String patchGroup);

        /**
         * <p>
         * The ID of the patch baseline used to patch the instance.
         * </p>
         * 
         * @param baselineId
         *        The ID of the patch baseline used to patch the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineId(String baselineId);

        /**
         * <p>
         * The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         * collected.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         *        collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * An https URL or an Amazon S3 path-style URL to a list of patches to be installed. This patch installation
         * list, which you maintain in an Amazon S3 bucket in YAML format and specify in the SSM document
         * <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch baseline.
         * </p>
         * <p>
         * For more information about the <code>InstallOverrideList</code> parameter, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
         * >About the SSM Document AWS-RunPatchBaseline</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param installOverrideList
         *        An https URL or an Amazon S3 path-style URL to a list of patches to be installed. This patch
         *        installation list, which you maintain in an Amazon S3 bucket in YAML format and specify in the SSM
         *        document <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch
         *        baseline.</p>
         *        <p>
         *        For more information about the <code>InstallOverrideList</code> parameter, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
         *        >About the SSM Document AWS-RunPatchBaseline</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installOverrideList(String installOverrideList);

        /**
         * <p>
         * Placeholder information. This field will always be empty in the current release of the service.
         * </p>
         * 
         * @param ownerInformation
         *        Placeholder information. This field will always be empty in the current release of the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * The number of patches from the patch baseline that are installed on the instance.
         * </p>
         * 
         * @param installedCount
         *        The number of patches from the patch baseline that are installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedCount(Integer installedCount);

        /**
         * <p>
         * The number of patches not specified in the patch baseline that are installed on the instance.
         * </p>
         * 
         * @param installedOtherCount
         *        The number of patches not specified in the patch baseline that are installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedOtherCount(Integer installedOtherCount);

        /**
         * <p>
         * The number of patches installed by Patch Manager since the last time the instance was rebooted.
         * </p>
         * 
         * @param installedPendingRebootCount
         *        The number of patches installed by Patch Manager since the last time the instance was rebooted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedPendingRebootCount(Integer installedPendingRebootCount);

        /**
         * <p>
         * The number of instances with patches installed that are specified in a RejectedPatches list. Patches with a
         * status of <i>InstalledRejected</i> were typically installed before they were added to a RejectedPatches list.
         * </p>
         * <note>
         * <p>
         * If ALLOW_AS_DEPENDENCY is the specified option for RejectedPatchesAction, the value of InstalledRejectedCount
         * will always be 0 (zero).
         * </p>
         * </note>
         * 
         * @param installedRejectedCount
         *        The number of instances with patches installed that are specified in a RejectedPatches list. Patches
         *        with a status of <i>InstalledRejected</i> were typically installed before they were added to a
         *        RejectedPatches list.</p> <note>
         *        <p>
         *        If ALLOW_AS_DEPENDENCY is the specified option for RejectedPatchesAction, the value of
         *        InstalledRejectedCount will always be 0 (zero).
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedRejectedCount(Integer installedRejectedCount);

        /**
         * <p>
         * The number of patches from the patch baseline that are applicable for the instance but aren't currently
         * installed.
         * </p>
         * 
         * @param missingCount
         *        The number of patches from the patch baseline that are applicable for the instance but aren't
         *        currently installed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingCount(Integer missingCount);

        /**
         * <p>
         * The number of patches from the patch baseline that were attempted to be installed during the last patching
         * operation, but failed to install.
         * </p>
         * 
         * @param failedCount
         *        The number of patches from the patch baseline that were attempted to be installed during the last
         *        patching operation, but failed to install.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedCount(Integer failedCount);

        /**
         * <p>
         * The number of patches beyond the supported limit of <code>NotApplicableCount</code> that are not reported by
         * name to Systems Manager Inventory.
         * </p>
         * 
         * @param unreportedNotApplicableCount
         *        The number of patches beyond the supported limit of <code>NotApplicableCount</code> that are not
         *        reported by name to Systems Manager Inventory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unreportedNotApplicableCount(Integer unreportedNotApplicableCount);

        /**
         * <p>
         * The number of patches from the patch baseline that aren't applicable for the instance and therefore aren't
         * installed on the instance. This number may be truncated if the list of patch names is very large. The number
         * of patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
         * </p>
         * 
         * @param notApplicableCount
         *        The number of patches from the patch baseline that aren't applicable for the instance and therefore
         *        aren't installed on the instance. This number may be truncated if the list of patch names is very
         *        large. The number of patches beyond this limit are reported in
         *        <code>UnreportedNotApplicableCount</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notApplicableCount(Integer notApplicableCount);

        /**
         * <p>
         * The time the most recent patching operation was started on the instance.
         * </p>
         * 
         * @param operationStartTime
         *        The time the most recent patching operation was started on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationStartTime(Instant operationStartTime);

        /**
         * <p>
         * The time the most recent patching operation completed on the instance.
         * </p>
         * 
         * @param operationEndTime
         *        The time the most recent patching operation completed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationEndTime(Instant operationEndTime);

        /**
         * <p>
         * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
         * missing patches).
         * </p>
         * 
         * @param operation
         *        The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
         *        (install missing patches).
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(String operation);

        /**
         * <p>
         * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
         * missing patches).
         * </p>
         * 
         * @param operation
         *        The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
         *        (install missing patches).
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(PatchOperationType operation);

        /**
         * <p>
         * The time of the last attempt to patch the instance with <code>NoReboot</code> specified as the reboot option.
         * </p>
         * 
         * @param lastNoRebootInstallOperationTime
         *        The time of the last attempt to patch the instance with <code>NoReboot</code> specified as the reboot
         *        option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime);

        /**
         * <p>
         * Indicates the reboot option specified in the patch baseline.
         * </p>
         * <note>
         * <p>
         * Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
         * <code>Scan</code> operations.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any
         * patches are detected with a status of <code>InstalledPendingReboot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system.
         * Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These
         * patches might not be in effect until a reboot is performed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rebootOption
         *        Indicates the reboot option specified in the patch baseline.</p> <note>
         *        <p>
         *        Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch
         *        Manager <code>Scan</code> operations.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if
         *        any patches are detected with a status of <code>InstalledPendingReboot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the
         *        system. Patches installed with this option are assigned a status of
         *        <code>InstalledPendingReboot</code>. These patches might not be in effect until a reboot is performed.
         *        </p>
         *        </li>
         * @see RebootOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RebootOption
         */
        Builder rebootOption(String rebootOption);

        /**
         * <p>
         * Indicates the reboot option specified in the patch baseline.
         * </p>
         * <note>
         * <p>
         * Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch Manager
         * <code>Scan</code> operations.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if any
         * patches are detected with a status of <code>InstalledPendingReboot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the system.
         * Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These
         * patches might not be in effect until a reboot is performed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rebootOption
         *        Indicates the reboot option specified in the patch baseline.</p> <note>
         *        <p>
         *        Reboot options apply to <code>Install</code> operations only. Reboots are not attempted for Patch
         *        Manager <code>Scan</code> operations.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>RebootIfNeeded</b>: Patch Manager tries to reboot the instance if it installed any patches, or if
         *        any patches are detected with a status of <code>InstalledPendingReboot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>NoReboot</b>: Patch Manager attempts to install missing packages without trying to reboot the
         *        system. Patches installed with this option are assigned a status of
         *        <code>InstalledPendingReboot</code>. These patches might not be in effect until a reboot is performed.
         *        </p>
         *        </li>
         * @see RebootOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RebootOption
         */
        Builder rebootOption(RebootOption rebootOption);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String patchGroup;

        private String baselineId;

        private String snapshotId;

        private String installOverrideList;

        private String ownerInformation;

        private Integer installedCount;

        private Integer installedOtherCount;

        private Integer installedPendingRebootCount;

        private Integer installedRejectedCount;

        private Integer missingCount;

        private Integer failedCount;

        private Integer unreportedNotApplicableCount;

        private Integer notApplicableCount;

        private Instant operationStartTime;

        private Instant operationEndTime;

        private String operation;

        private Instant lastNoRebootInstallOperationTime;

        private String rebootOption;

        private BuilderImpl() {
        }

        private BuilderImpl(InstancePatchState model) {
            instanceId(model.instanceId);
            patchGroup(model.patchGroup);
            baselineId(model.baselineId);
            snapshotId(model.snapshotId);
            installOverrideList(model.installOverrideList);
            ownerInformation(model.ownerInformation);
            installedCount(model.installedCount);
            installedOtherCount(model.installedOtherCount);
            installedPendingRebootCount(model.installedPendingRebootCount);
            installedRejectedCount(model.installedRejectedCount);
            missingCount(model.missingCount);
            failedCount(model.failedCount);
            unreportedNotApplicableCount(model.unreportedNotApplicableCount);
            notApplicableCount(model.notApplicableCount);
            operationStartTime(model.operationStartTime);
            operationEndTime(model.operationEndTime);
            operation(model.operation);
            lastNoRebootInstallOperationTime(model.lastNoRebootInstallOperationTime);
            rebootOption(model.rebootOption);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getPatchGroup() {
            return patchGroup;
        }

        @Override
        public final Builder patchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
            return this;
        }

        public final void setPatchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
        }

        public final String getBaselineId() {
            return baselineId;
        }

        @Override
        public final Builder baselineId(String baselineId) {
            this.baselineId = baselineId;
            return this;
        }

        public final void setBaselineId(String baselineId) {
            this.baselineId = baselineId;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final String getInstallOverrideList() {
            return installOverrideList;
        }

        @Override
        public final Builder installOverrideList(String installOverrideList) {
            this.installOverrideList = installOverrideList;
            return this;
        }

        public final void setInstallOverrideList(String installOverrideList) {
            this.installOverrideList = installOverrideList;
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        public final Integer getInstalledCount() {
            return installedCount;
        }

        @Override
        public final Builder installedCount(Integer installedCount) {
            this.installedCount = installedCount;
            return this;
        }

        public final void setInstalledCount(Integer installedCount) {
            this.installedCount = installedCount;
        }

        public final Integer getInstalledOtherCount() {
            return installedOtherCount;
        }

        @Override
        public final Builder installedOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
            return this;
        }

        public final void setInstalledOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
        }

        public final Integer getInstalledPendingRebootCount() {
            return installedPendingRebootCount;
        }

        @Override
        public final Builder installedPendingRebootCount(Integer installedPendingRebootCount) {
            this.installedPendingRebootCount = installedPendingRebootCount;
            return this;
        }

        public final void setInstalledPendingRebootCount(Integer installedPendingRebootCount) {
            this.installedPendingRebootCount = installedPendingRebootCount;
        }

        public final Integer getInstalledRejectedCount() {
            return installedRejectedCount;
        }

        @Override
        public final Builder installedRejectedCount(Integer installedRejectedCount) {
            this.installedRejectedCount = installedRejectedCount;
            return this;
        }

        public final void setInstalledRejectedCount(Integer installedRejectedCount) {
            this.installedRejectedCount = installedRejectedCount;
        }

        public final Integer getMissingCount() {
            return missingCount;
        }

        @Override
        public final Builder missingCount(Integer missingCount) {
            this.missingCount = missingCount;
            return this;
        }

        public final void setMissingCount(Integer missingCount) {
            this.missingCount = missingCount;
        }

        public final Integer getFailedCount() {
            return failedCount;
        }

        @Override
        public final Builder failedCount(Integer failedCount) {
            this.failedCount = failedCount;
            return this;
        }

        public final void setFailedCount(Integer failedCount) {
            this.failedCount = failedCount;
        }

        public final Integer getUnreportedNotApplicableCount() {
            return unreportedNotApplicableCount;
        }

        @Override
        public final Builder unreportedNotApplicableCount(Integer unreportedNotApplicableCount) {
            this.unreportedNotApplicableCount = unreportedNotApplicableCount;
            return this;
        }

        public final void setUnreportedNotApplicableCount(Integer unreportedNotApplicableCount) {
            this.unreportedNotApplicableCount = unreportedNotApplicableCount;
        }

        public final Integer getNotApplicableCount() {
            return notApplicableCount;
        }

        @Override
        public final Builder notApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
            return this;
        }

        public final void setNotApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
        }

        public final Instant getOperationStartTime() {
            return operationStartTime;
        }

        @Override
        public final Builder operationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
            return this;
        }

        public final void setOperationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
        }

        public final Instant getOperationEndTime() {
            return operationEndTime;
        }

        @Override
        public final Builder operationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
            return this;
        }

        public final void setOperationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
        }

        public final String getOperation() {
            return operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(PatchOperationType operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        public final Instant getLastNoRebootInstallOperationTime() {
            return lastNoRebootInstallOperationTime;
        }

        @Override
        public final Builder lastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime) {
            this.lastNoRebootInstallOperationTime = lastNoRebootInstallOperationTime;
            return this;
        }

        public final void setLastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime) {
            this.lastNoRebootInstallOperationTime = lastNoRebootInstallOperationTime;
        }

        public final String getRebootOption() {
            return rebootOption;
        }

        @Override
        public final Builder rebootOption(String rebootOption) {
            this.rebootOption = rebootOption;
            return this;
        }

        @Override
        public final Builder rebootOption(RebootOption rebootOption) {
            this.rebootOption(rebootOption == null ? null : rebootOption.toString());
            return this;
        }

        public final void setRebootOption(String rebootOption) {
            this.rebootOption = rebootOption;
        }

        @Override
        public InstancePatchState build() {
            return new InstancePatchState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
