/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an Automation failure.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FailureDetails implements SdkPojo, Serializable, ToCopyableBuilder<FailureDetails.Builder, FailureDetails> {
    private static final SdkField<String> FAILURE_STAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(FailureDetails::failureStage)).setter(setter(Builder::failureStage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureStage").build()).build();

    private static final SdkField<String> FAILURE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(FailureDetails::failureType)).setter(setter(Builder::failureType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureType").build()).build();

    private static final SdkField<Map<String, List<String>>> DETAILS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(FailureDetails::details))
            .setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FAILURE_STAGE_FIELD,
            FAILURE_TYPE_FIELD, DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String failureStage;

    private final String failureType;

    private final Map<String, List<String>> details;

    private FailureDetails(BuilderImpl builder) {
        this.failureStage = builder.failureStage;
        this.failureType = builder.failureType;
        this.details = builder.details;
    }

    /**
     * <p>
     * The stage of the Automation execution when the failure occurred. The stages include the following:
     * InputValidation, PreVerification, Invocation, PostVerification.
     * </p>
     * 
     * @return The stage of the Automation execution when the failure occurred. The stages include the following:
     *         InputValidation, PreVerification, Invocation, PostVerification.
     */
    public String failureStage() {
        return failureStage;
    }

    /**
     * <p>
     * The type of Automation failure. Failure types include the following: Action, Permission, Throttling,
     * Verification, Internal.
     * </p>
     * 
     * @return The type of Automation failure. Failure types include the following: Action, Permission, Throttling,
     *         Verification, Internal.
     */
    public String failureType() {
        return failureType;
    }

    /**
     * Returns true if the Details property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDetails() {
        return details != null && !(details instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Detailed information about the Automation step failure.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Detailed information about the Automation step failure.
     */
    public Map<String, List<String>> details() {
        return details;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(failureStage());
        hashCode = 31 * hashCode + Objects.hashCode(failureType());
        hashCode = 31 * hashCode + Objects.hashCode(details());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FailureDetails)) {
            return false;
        }
        FailureDetails other = (FailureDetails) obj;
        return Objects.equals(failureStage(), other.failureStage()) && Objects.equals(failureType(), other.failureType())
                && Objects.equals(details(), other.details());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("FailureDetails").add("FailureStage", failureStage()).add("FailureType", failureType())
                .add("Details", details()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FailureStage":
            return Optional.ofNullable(clazz.cast(failureStage()));
        case "FailureType":
            return Optional.ofNullable(clazz.cast(failureType()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FailureDetails, T> g) {
        return obj -> g.apply((FailureDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FailureDetails> {
        /**
         * <p>
         * The stage of the Automation execution when the failure occurred. The stages include the following:
         * InputValidation, PreVerification, Invocation, PostVerification.
         * </p>
         * 
         * @param failureStage
         *        The stage of the Automation execution when the failure occurred. The stages include the following:
         *        InputValidation, PreVerification, Invocation, PostVerification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureStage(String failureStage);

        /**
         * <p>
         * The type of Automation failure. Failure types include the following: Action, Permission, Throttling,
         * Verification, Internal.
         * </p>
         * 
         * @param failureType
         *        The type of Automation failure. Failure types include the following: Action, Permission, Throttling,
         *        Verification, Internal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureType(String failureType);

        /**
         * <p>
         * Detailed information about the Automation step failure.
         * </p>
         * 
         * @param details
         *        Detailed information about the Automation step failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Map<String, ? extends Collection<String>> details);
    }

    static final class BuilderImpl implements Builder {
        private String failureStage;

        private String failureType;

        private Map<String, List<String>> details = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FailureDetails model) {
            failureStage(model.failureStage);
            failureType(model.failureType);
            details(model.details);
        }

        public final String getFailureStage() {
            return failureStage;
        }

        @Override
        public final Builder failureStage(String failureStage) {
            this.failureStage = failureStage;
            return this;
        }

        public final void setFailureStage(String failureStage) {
            this.failureStage = failureStage;
        }

        public final String getFailureType() {
            return failureType;
        }

        @Override
        public final Builder failureType(String failureType) {
            this.failureType = failureType;
            return this;
        }

        public final void setFailureType(String failureType) {
            this.failureType = failureType;
        }

        public final Map<String, ? extends Collection<String>> getDetails() {
            return details;
        }

        @Override
        public final Builder details(Map<String, ? extends Collection<String>> details) {
            this.details = AutomationParameterMapCopier.copy(details);
            return this;
        }

        public final void setDetails(Map<String, ? extends Collection<String>> details) {
            this.details = AutomationParameterMapCopier.copy(details);
        }

        @Override
        public FailureDetails build() {
            return new FailureDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
