/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the compliance as defined by the resource type. For example, for a patch resource type,
 * <code>Items</code> includes information about the PatchSeverity, Classification, and so on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComplianceItem implements SdkPojo, Serializable, ToCopyableBuilder<ComplianceItem.Builder, ComplianceItem> {
    private static final SdkField<String> COMPLIANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::complianceType)).setter(setter(Builder::complianceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceType").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComplianceItem::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<ComplianceExecutionSummary> EXECUTION_SUMMARY_FIELD = SdkField
            .<ComplianceExecutionSummary> builder(MarshallingType.SDK_POJO).getter(getter(ComplianceItem::executionSummary))
            .setter(setter(Builder::executionSummary)).constructor(ComplianceExecutionSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionSummary").build()).build();

    private static final SdkField<Map<String, String>> DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ComplianceItem::details))
            .setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPLIANCE_TYPE_FIELD,
            RESOURCE_TYPE_FIELD, RESOURCE_ID_FIELD, ID_FIELD, TITLE_FIELD, STATUS_FIELD, SEVERITY_FIELD, EXECUTION_SUMMARY_FIELD,
            DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String complianceType;

    private final String resourceType;

    private final String resourceId;

    private final String id;

    private final String title;

    private final String status;

    private final String severity;

    private final ComplianceExecutionSummary executionSummary;

    private final Map<String, String> details;

    private ComplianceItem(BuilderImpl builder) {
        this.complianceType = builder.complianceType;
        this.resourceType = builder.resourceType;
        this.resourceId = builder.resourceId;
        this.id = builder.id;
        this.title = builder.title;
        this.status = builder.status;
        this.severity = builder.severity;
        this.executionSummary = builder.executionSummary;
        this.details = builder.details;
    }

    /**
     * <p>
     * The compliance type. For example, Association (for a State Manager association), Patch, or Custom:
     * <code>string</code> are all valid compliance types.
     * </p>
     * 
     * @return The compliance type. For example, Association (for a State Manager association), Patch, or Custom:
     *         <code>string</code> are all valid compliance types.
     */
    public String complianceType() {
        return complianceType;
    }

    /**
     * <p>
     * The type of resource. <code>ManagedInstance</code> is currently the only supported resource type.
     * </p>
     * 
     * @return The type of resource. <code>ManagedInstance</code> is currently the only supported resource type.
     */
    public String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * An ID for the resource. For a managed instance, this is the instance ID.
     * </p>
     * 
     * @return An ID for the resource. For a managed instance, this is the instance ID.
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * An ID for the compliance item. For example, if the compliance item is a Windows patch, the ID could be the number
     * of the KB article; for example: KB4010320.
     * </p>
     * 
     * @return An ID for the compliance item. For example, if the compliance item is a Windows patch, the ID could be
     *         the number of the KB article; for example: KB4010320.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * A title for the compliance item. For example, if the compliance item is a Windows patch, the title could be the
     * title of the KB article for the patch; for example: Security Update for Active Directory Federation Services.
     * </p>
     * 
     * @return A title for the compliance item. For example, if the compliance item is a Windows patch, the title could
     *         be the title of the KB article for the patch; for example: Security Update for Active Directory
     *         Federation Services.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for Windows
     * patches that aren't applicable).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
     *         Windows patches that aren't applicable).
     * @see ComplianceStatus
     */
    public ComplianceStatus status() {
        return ComplianceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for Windows
     * patches that aren't applicable).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
     *         Windows patches that aren't applicable).
     * @see ComplianceStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     * Informational, Unspecified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     *         Informational, Unspecified.
     * @see ComplianceSeverity
     */
    public ComplianceSeverity severity() {
        return ComplianceSeverity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     * Informational, Unspecified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     *         Informational, Unspecified.
     * @see ComplianceSeverity
     */
    public String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * A summary for the compliance item. The summary includes an execution ID, the execution type (for example,
     * command), and the execution time.
     * </p>
     * 
     * @return A summary for the compliance item. The summary includes an execution ID, the execution type (for example,
     *         command), and the execution time.
     */
    public ComplianceExecutionSummary executionSummary() {
        return executionSummary;
    }

    /**
     * Returns true if the Details property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDetails() {
        return details != null && !(details instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A "Key": "Value" tag combination for the compliance item.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A "Key": "Value" tag combination for the compliance item.
     */
    public Map<String, String> details() {
        return details;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(complianceType());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(executionSummary());
        hashCode = 31 * hashCode + Objects.hashCode(details());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComplianceItem)) {
            return false;
        }
        ComplianceItem other = (ComplianceItem) obj;
        return Objects.equals(complianceType(), other.complianceType()) && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(id(), other.id())
                && Objects.equals(title(), other.title()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(severityAsString(), other.severityAsString())
                && Objects.equals(executionSummary(), other.executionSummary()) && Objects.equals(details(), other.details());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ComplianceItem").add("ComplianceType", complianceType()).add("ResourceType", resourceType())
                .add("ResourceId", resourceId()).add("Id", id()).add("Title", title()).add("Status", statusAsString())
                .add("Severity", severityAsString()).add("ExecutionSummary", executionSummary()).add("Details", details())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComplianceType":
            return Optional.ofNullable(clazz.cast(complianceType()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "ExecutionSummary":
            return Optional.ofNullable(clazz.cast(executionSummary()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComplianceItem, T> g) {
        return obj -> g.apply((ComplianceItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComplianceItem> {
        /**
         * <p>
         * The compliance type. For example, Association (for a State Manager association), Patch, or Custom:
         * <code>string</code> are all valid compliance types.
         * </p>
         * 
         * @param complianceType
         *        The compliance type. For example, Association (for a State Manager association), Patch, or Custom:
         *        <code>string</code> are all valid compliance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceType(String complianceType);

        /**
         * <p>
         * The type of resource. <code>ManagedInstance</code> is currently the only supported resource type.
         * </p>
         * 
         * @param resourceType
         *        The type of resource. <code>ManagedInstance</code> is currently the only supported resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * An ID for the resource. For a managed instance, this is the instance ID.
         * </p>
         * 
         * @param resourceId
         *        An ID for the resource. For a managed instance, this is the instance ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * An ID for the compliance item. For example, if the compliance item is a Windows patch, the ID could be the
         * number of the KB article; for example: KB4010320.
         * </p>
         * 
         * @param id
         *        An ID for the compliance item. For example, if the compliance item is a Windows patch, the ID could be
         *        the number of the KB article; for example: KB4010320.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A title for the compliance item. For example, if the compliance item is a Windows patch, the title could be
         * the title of the KB article for the patch; for example: Security Update for Active Directory Federation
         * Services.
         * </p>
         * 
         * @param title
         *        A title for the compliance item. For example, if the compliance item is a Windows patch, the title
         *        could be the title of the KB article for the patch; for example: Security Update for Active Directory
         *        Federation Services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
         * Windows patches that aren't applicable).
         * </p>
         * 
         * @param status
         *        The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
         *        Windows patches that aren't applicable).
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
         * Windows patches that aren't applicable).
         * </p>
         * 
         * @param status
         *        The status of the compliance item. An item is either COMPLIANT, NON_COMPLIANT, or an empty string (for
         *        Windows patches that aren't applicable).
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(ComplianceStatus status);

        /**
         * <p>
         * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
         * Informational, Unspecified.
         * </p>
         * 
         * @param severity
         *        The severity of the compliance status. Severity can be one of the following: Critical, High, Medium,
         *        Low, Informational, Unspecified.
         * @see ComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceSeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
         * Informational, Unspecified.
         * </p>
         * 
         * @param severity
         *        The severity of the compliance status. Severity can be one of the following: Critical, High, Medium,
         *        Low, Informational, Unspecified.
         * @see ComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceSeverity
         */
        Builder severity(ComplianceSeverity severity);

        /**
         * <p>
         * A summary for the compliance item. The summary includes an execution ID, the execution type (for example,
         * command), and the execution time.
         * </p>
         * 
         * @param executionSummary
         *        A summary for the compliance item. The summary includes an execution ID, the execution type (for
         *        example, command), and the execution time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionSummary(ComplianceExecutionSummary executionSummary);

        /**
         * <p>
         * A summary for the compliance item. The summary includes an execution ID, the execution type (for example,
         * command), and the execution time.
         * </p>
         * This is a convenience that creates an instance of the {@link ComplianceExecutionSummary.Builder} avoiding the
         * need to create one manually via {@link ComplianceExecutionSummary#builder()}.
         *
         * When the {@link Consumer} completes, {@link ComplianceExecutionSummary.Builder#build()} is called immediately
         * and its result is passed to {@link #executionSummary(ComplianceExecutionSummary)}.
         * 
         * @param executionSummary
         *        a consumer that will call methods on {@link ComplianceExecutionSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #executionSummary(ComplianceExecutionSummary)
         */
        default Builder executionSummary(Consumer<ComplianceExecutionSummary.Builder> executionSummary) {
            return executionSummary(ComplianceExecutionSummary.builder().applyMutation(executionSummary).build());
        }

        /**
         * <p>
         * A "Key": "Value" tag combination for the compliance item.
         * </p>
         * 
         * @param details
         *        A "Key": "Value" tag combination for the compliance item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Map<String, String> details);
    }

    static final class BuilderImpl implements Builder {
        private String complianceType;

        private String resourceType;

        private String resourceId;

        private String id;

        private String title;

        private String status;

        private String severity;

        private ComplianceExecutionSummary executionSummary;

        private Map<String, String> details = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComplianceItem model) {
            complianceType(model.complianceType);
            resourceType(model.resourceType);
            resourceId(model.resourceId);
            id(model.id);
            title(model.title);
            status(model.status);
            severity(model.severity);
            executionSummary(model.executionSummary);
            details(model.details);
        }

        public final String getComplianceType() {
            return complianceType;
        }

        @Override
        public final Builder complianceType(String complianceType) {
            this.complianceType = complianceType;
            return this;
        }

        public final void setComplianceType(String complianceType) {
            this.complianceType = complianceType;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComplianceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getSeverity() {
            return severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(ComplianceSeverity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        public final ComplianceExecutionSummary.Builder getExecutionSummary() {
            return executionSummary != null ? executionSummary.toBuilder() : null;
        }

        @Override
        public final Builder executionSummary(ComplianceExecutionSummary executionSummary) {
            this.executionSummary = executionSummary;
            return this;
        }

        public final void setExecutionSummary(ComplianceExecutionSummary.BuilderImpl executionSummary) {
            this.executionSummary = executionSummary != null ? executionSummary.build() : null;
        }

        public final Map<String, String> getDetails() {
            return details;
        }

        @Override
        public final Builder details(Map<String, String> details) {
            this.details = ComplianceItemDetailsCopier.copy(details);
            return this;
        }

        public final void setDetails(Map<String, String> details) {
            this.details = ComplianceItemDetailsCopier.copy(details);
        }

        @Override
        public ComplianceItem build() {
            return new ComplianceItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
