/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inventory result item.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InventoryResultItem implements SdkPojo, Serializable,
        ToCopyableBuilder<InventoryResultItem.Builder, InventoryResultItem> {
    private static final SdkField<String> TYPE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TypeName").getter(getter(InventoryResultItem::typeName)).setter(setter(Builder::typeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TypeName").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchemaVersion").getter(getter(InventoryResultItem::schemaVersion))
            .setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> CAPTURE_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CaptureTime").getter(getter(InventoryResultItem::captureTime)).setter(setter(Builder::captureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureTime").build()).build();

    private static final SdkField<String> CONTENT_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentHash").getter(getter(InventoryResultItem::contentHash)).setter(setter(Builder::contentHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentHash").build()).build();

    private static final SdkField<List<Map<String, String>>> CONTENT_FIELD = SdkField
            .<List<Map<String, String>>> builder(MarshallingType.LIST)
            .memberName("Content")
            .getter(getter(InventoryResultItem::content))
            .setter(setter(Builder::content))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Content").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, String>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_NAME_FIELD,
            SCHEMA_VERSION_FIELD, CAPTURE_TIME_FIELD, CONTENT_HASH_FIELD, CONTENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String typeName;

    private final String schemaVersion;

    private final String captureTime;

    private final String contentHash;

    private final List<Map<String, String>> content;

    private InventoryResultItem(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.schemaVersion = builder.schemaVersion;
        this.captureTime = builder.captureTime;
        this.contentHash = builder.contentHash;
        this.content = builder.content;
    }

    /**
     * <p>
     * The name of the inventory result item type.
     * </p>
     * 
     * @return The name of the inventory result item type.
     */
    public String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory result item/
     * </p>
     * 
     * @return The schema version for the inventory result item/
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The time inventory item data was captured.
     * </p>
     * 
     * @return The time inventory item data was captured.
     */
    public String captureTime() {
        return captureTime;
    }

    /**
     * <p>
     * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update inventory
     * information. The PutInventory API does not update the inventory item type contents if the MD5 hash has not
     * changed since last update.
     * </p>
     * 
     * @return MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
     *         inventory information. The PutInventory API does not update the inventory item type contents if the MD5
     *         hash has not changed since last update.
     */
    public String contentHash() {
        return contentHash;
    }

    /**
     * Returns true if the Content property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasContent() {
        return content != null && !(content instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains all the inventory data of the item type. Results include attribute names and values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContent()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Contains all the inventory data of the item type. Results include attribute names and values.
     */
    public List<Map<String, String>> content() {
        return content;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(captureTime());
        hashCode = 31 * hashCode + Objects.hashCode(contentHash());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryResultItem)) {
            return false;
        }
        InventoryResultItem other = (InventoryResultItem) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(captureTime(), other.captureTime()) && Objects.equals(contentHash(), other.contentHash())
                && Objects.equals(content(), other.content());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InventoryResultItem").add("TypeName", typeName()).add("SchemaVersion", schemaVersion())
                .add("CaptureTime", captureTime()).add("ContentHash", contentHash()).add("Content", content()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.ofNullable(clazz.cast(typeName()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "CaptureTime":
            return Optional.ofNullable(clazz.cast(captureTime()));
        case "ContentHash":
            return Optional.ofNullable(clazz.cast(contentHash()));
        case "Content":
            return Optional.ofNullable(clazz.cast(content()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InventoryResultItem, T> g) {
        return obj -> g.apply((InventoryResultItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InventoryResultItem> {
        /**
         * <p>
         * The name of the inventory result item type.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory result item type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory result item/
         * </p>
         * 
         * @param schemaVersion
         *        The schema version for the inventory result item/
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The time inventory item data was captured.
         * </p>
         * 
         * @param captureTime
         *        The time inventory item data was captured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureTime(String captureTime);

        /**
         * <p>
         * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         * inventory information. The PutInventory API does not update the inventory item type contents if the MD5 hash
         * has not changed since last update.
         * </p>
         * 
         * @param contentHash
         *        MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         *        inventory information. The PutInventory API does not update the inventory item type contents if the
         *        MD5 hash has not changed since last update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentHash(String contentHash);

        /**
         * <p>
         * Contains all the inventory data of the item type. Results include attribute names and values.
         * </p>
         * 
         * @param content
         *        Contains all the inventory data of the item type. Results include attribute names and values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Collection<? extends Map<String, String>> content);

        /**
         * <p>
         * Contains all the inventory data of the item type. Results include attribute names and values.
         * </p>
         * 
         * @param content
         *        Contains all the inventory data of the item type. Results include attribute names and values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Map<String, String>... content);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String schemaVersion;

        private String captureTime;

        private String contentHash;

        private List<Map<String, String>> content = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryResultItem model) {
            typeName(model.typeName);
            schemaVersion(model.schemaVersion);
            captureTime(model.captureTime);
            contentHash(model.contentHash);
            content(model.content);
        }

        public final String getTypeName() {
            return typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final String getCaptureTime() {
            return captureTime;
        }

        @Override
        public final Builder captureTime(String captureTime) {
            this.captureTime = captureTime;
            return this;
        }

        public final void setCaptureTime(String captureTime) {
            this.captureTime = captureTime;
        }

        public final String getContentHash() {
            return contentHash;
        }

        @Override
        public final Builder contentHash(String contentHash) {
            this.contentHash = contentHash;
            return this;
        }

        public final void setContentHash(String contentHash) {
            this.contentHash = contentHash;
        }

        public final Collection<? extends Map<String, String>> getContent() {
            return content;
        }

        @Override
        public final Builder content(Collection<? extends Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder content(Map<String, String>... content) {
            content(Arrays.asList(content));
            return this;
        }

        public final void setContent(Collection<? extends Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
        }

        @Override
        public InventoryResultItem build() {
            return new InventoryResultItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
