/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about parameter usage.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParameterHistory implements SdkPojo, Serializable,
        ToCopyableBuilder<ParameterHistory.Builder, ParameterHistory> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ParameterHistory::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ParameterHistory::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(ParameterHistory::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedDate").getter(getter(ParameterHistory::lastModifiedDate))
            .setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final SdkField<String> LAST_MODIFIED_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModifiedUser").getter(getter(ParameterHistory::lastModifiedUser))
            .setter(setter(Builder::lastModifiedUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedUser").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ParameterHistory::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Value")
            .getter(getter(ParameterHistory::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> ALLOWED_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllowedPattern").getter(getter(ParameterHistory::allowedPattern))
            .setter(setter(Builder::allowedPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedPattern").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Version")
            .getter(getter(ParameterHistory::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Labels")
            .getter(getter(ParameterHistory::labels))
            .setter(setter(Builder::labels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Labels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Tier")
            .getter(getter(ParameterHistory::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<List<ParameterInlinePolicy>> POLICIES_FIELD = SdkField
            .<List<ParameterInlinePolicy>> builder(MarshallingType.LIST)
            .memberName("Policies")
            .getter(getter(ParameterHistory::policies))
            .setter(setter(Builder::policies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ParameterInlinePolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterInlinePolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataType").getter(getter(ParameterHistory::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            KEY_ID_FIELD, LAST_MODIFIED_DATE_FIELD, LAST_MODIFIED_USER_FIELD, DESCRIPTION_FIELD, VALUE_FIELD,
            ALLOWED_PATTERN_FIELD, VERSION_FIELD, LABELS_FIELD, TIER_FIELD, POLICIES_FIELD, DATA_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final String keyId;

    private final Instant lastModifiedDate;

    private final String lastModifiedUser;

    private final String description;

    private final String value;

    private final String allowedPattern;

    private final Long version;

    private final List<String> labels;

    private final String tier;

    private final List<ParameterInlinePolicy> policies;

    private final String dataType;

    private ParameterHistory(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.keyId = builder.keyId;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.lastModifiedUser = builder.lastModifiedUser;
        this.description = builder.description;
        this.value = builder.value;
        this.allowedPattern = builder.allowedPattern;
        this.version = builder.version;
        this.labels = builder.labels;
        this.tier = builder.tier;
        this.policies = builder.policies;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * The name of the parameter.
     * </p>
     * 
     * @return The name of the parameter.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The type of parameter used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter used.
     * @see ParameterType
     */
    public ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter used.
     * @see ParameterType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The ID of the query key used for this parameter.
     * </p>
     * 
     * @return The ID of the query key used for this parameter.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Date the parameter was last changed or updated.
     * </p>
     * 
     * @return Date the parameter was last changed or updated.
     */
    public Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the AWS user who last changed the parameter.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the AWS user who last changed the parameter.
     */
    public String lastModifiedUser() {
        return lastModifiedUser;
    }

    /**
     * <p>
     * Information about the parameter.
     * </p>
     * 
     * @return Information about the parameter.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The parameter value.
     * </p>
     * 
     * @return The parameter value.
     */
    public String value() {
        return value;
    }

    /**
     * <p>
     * Parameter names can include the following letters and symbols.
     * </p>
     * <p>
     * a-zA-Z0-9_.-
     * </p>
     * 
     * @return Parameter names can include the following letters and symbols.</p>
     *         <p>
     *         a-zA-Z0-9_.-
     */
    public String allowedPattern() {
        return allowedPattern;
    }

    /**
     * <p>
     * The parameter version.
     * </p>
     * 
     * @return The parameter version.
     */
    public Long version() {
        return version;
    }

    /**
     * Returns true if the Labels property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasLabels() {
        return labels != null && !(labels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Labels assigned to the parameter version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLabels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Labels assigned to the parameter version.
     */
    public List<String> labels() {
        return labels;
    }

    /**
     * <p>
     * The parameter tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier.
     * @see ParameterTier
     */
    public ParameterTier tier() {
        return ParameterTier.fromValue(tier);
    }

    /**
     * <p>
     * The parameter tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier.
     * @see ParameterTier
     */
    public String tierAsString() {
        return tier;
    }

    /**
     * Returns true if the Policies property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasPolicies() {
        return policies != null && !(policies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the policies assigned to a parameter.
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
     * parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPolicies()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Information about the policies assigned to a parameter.</p>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">
     *         Assigning parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public List<ParameterInlinePolicy> policies() {
        return policies;
    }

    /**
     * <p>
     * The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
     * <code>text</code>.
     * </p>
     * 
     * @return The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
     *         <code>text</code>.
     */
    public String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedUser());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(allowedPattern());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasLabels() ? labels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPolicies() ? policies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParameterHistory)) {
            return false;
        }
        ParameterHistory other = (ParameterHistory) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(keyId(), other.keyId()) && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(lastModifiedUser(), other.lastModifiedUser())
                && Objects.equals(description(), other.description()) && Objects.equals(value(), other.value())
                && Objects.equals(allowedPattern(), other.allowedPattern()) && Objects.equals(version(), other.version())
                && hasLabels() == other.hasLabels() && Objects.equals(labels(), other.labels())
                && Objects.equals(tierAsString(), other.tierAsString()) && hasPolicies() == other.hasPolicies()
                && Objects.equals(policies(), other.policies()) && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ParameterHistory").add("Name", name()).add("Type", typeAsString()).add("KeyId", keyId())
                .add("LastModifiedDate", lastModifiedDate()).add("LastModifiedUser", lastModifiedUser())
                .add("Description", description()).add("Value", value()).add("AllowedPattern", allowedPattern())
                .add("Version", version()).add("Labels", hasLabels() ? labels() : null).add("Tier", tierAsString())
                .add("Policies", hasPolicies() ? policies() : null).add("DataType", dataType()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "LastModifiedUser":
            return Optional.ofNullable(clazz.cast(lastModifiedUser()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "AllowedPattern":
            return Optional.ofNullable(clazz.cast(allowedPattern()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Labels":
            return Optional.ofNullable(clazz.cast(labels()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ParameterHistory, T> g) {
        return obj -> g.apply((ParameterHistory) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParameterHistory> {
        /**
         * <p>
         * The name of the parameter.
         * </p>
         * 
         * @param name
         *        The name of the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of parameter used.
         * </p>
         * 
         * @param type
         *        The type of parameter used.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter used.
         * </p>
         * 
         * @param type
         *        The type of parameter used.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The ID of the query key used for this parameter.
         * </p>
         * 
         * @param keyId
         *        The ID of the query key used for this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Date the parameter was last changed or updated.
         * </p>
         * 
         * @param lastModifiedDate
         *        Date the parameter was last changed or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * Amazon Resource Name (ARN) of the AWS user who last changed the parameter.
         * </p>
         * 
         * @param lastModifiedUser
         *        Amazon Resource Name (ARN) of the AWS user who last changed the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedUser(String lastModifiedUser);

        /**
         * <p>
         * Information about the parameter.
         * </p>
         * 
         * @param description
         *        Information about the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The parameter value.
         * </p>
         * 
         * @param value
         *        The parameter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * Parameter names can include the following letters and symbols.
         * </p>
         * <p>
         * a-zA-Z0-9_.-
         * </p>
         * 
         * @param allowedPattern
         *        Parameter names can include the following letters and symbols.</p>
         *        <p>
         *        a-zA-Z0-9_.-
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedPattern(String allowedPattern);

        /**
         * <p>
         * The parameter version.
         * </p>
         * 
         * @param version
         *        The parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);

        /**
         * <p>
         * Labels assigned to the parameter version.
         * </p>
         * 
         * @param labels
         *        Labels assigned to the parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(Collection<String> labels);

        /**
         * <p>
         * Labels assigned to the parameter version.
         * </p>
         * 
         * @param labels
         *        Labels assigned to the parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(String... labels);

        /**
         * <p>
         * The parameter tier.
         * </p>
         * 
         * @param tier
         *        The parameter tier.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The parameter tier.
         * </p>
         * 
         * @param tier
         *        The parameter tier.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(ParameterTier tier);

        /**
         * <p>
         * Information about the policies assigned to a parameter.
         * </p>
         * <p>
         * <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
         * parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param policies
         *        Information about the policies assigned to a parameter.</p>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">
         *        Assigning parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(Collection<ParameterInlinePolicy> policies);

        /**
         * <p>
         * Information about the policies assigned to a parameter.
         * </p>
         * <p>
         * <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
         * parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param policies
         *        Information about the policies assigned to a parameter.</p>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">
         *        Assigning parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(ParameterInlinePolicy... policies);

        /**
         * <p>
         * Information about the policies assigned to a parameter.
         * </p>
         * <p>
         * <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
         * parameter policies</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ParameterInlinePolicy>.Builder} avoiding
         * the need to create one manually via {@link List<ParameterInlinePolicy>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ParameterInlinePolicy>.Builder#build()} is called
         * immediately and its result is passed to {@link #policies(List<ParameterInlinePolicy>)}.
         * 
         * @param policies
         *        a consumer that will call methods on {@link List<ParameterInlinePolicy>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(List<ParameterInlinePolicy>)
         */
        Builder policies(Consumer<ParameterInlinePolicy.Builder>... policies);

        /**
         * <p>
         * The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
         * <code>text</code>.
         * </p>
         * 
         * @param dataType
         *        The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default
         *        is <code>text</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private String keyId;

        private Instant lastModifiedDate;

        private String lastModifiedUser;

        private String description;

        private String value;

        private String allowedPattern;

        private Long version;

        private List<String> labels = DefaultSdkAutoConstructList.getInstance();

        private String tier;

        private List<ParameterInlinePolicy> policies = DefaultSdkAutoConstructList.getInstance();

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(ParameterHistory model) {
            name(model.name);
            type(model.type);
            keyId(model.keyId);
            lastModifiedDate(model.lastModifiedDate);
            lastModifiedUser(model.lastModifiedUser);
            description(model.description);
            value(model.value);
            allowedPattern(model.allowedPattern);
            version(model.version);
            labels(model.labels);
            tier(model.tier);
            policies(model.policies);
            dataType(model.dataType);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        public final String getLastModifiedUser() {
            return lastModifiedUser;
        }

        @Override
        public final Builder lastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
            return this;
        }

        public final void setLastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getValue() {
            return value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        public final String getAllowedPattern() {
            return allowedPattern;
        }

        @Override
        public final Builder allowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
            return this;
        }

        public final void setAllowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
        }

        public final Long getVersion() {
            return version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        public final Collection<String> getLabels() {
            if (labels instanceof SdkAutoConstructList) {
                return null;
            }
            return labels;
        }

        @Override
        public final Builder labels(Collection<String> labels) {
            this.labels = ParameterLabelListCopier.copy(labels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder labels(String... labels) {
            labels(Arrays.asList(labels));
            return this;
        }

        public final void setLabels(Collection<String> labels) {
            this.labels = ParameterLabelListCopier.copy(labels);
        }

        public final String getTier() {
            return tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(ParameterTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        public final Collection<ParameterInlinePolicy.Builder> getPolicies() {
            if (policies instanceof SdkAutoConstructList) {
                return null;
            }
            return policies != null ? policies.stream().map(ParameterInlinePolicy::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder policies(Collection<ParameterInlinePolicy> policies) {
            this.policies = ParameterPolicyListCopier.copy(policies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(ParameterInlinePolicy... policies) {
            policies(Arrays.asList(policies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(Consumer<ParameterInlinePolicy.Builder>... policies) {
            policies(Stream.of(policies).map(c -> ParameterInlinePolicy.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPolicies(Collection<ParameterInlinePolicy.BuilderImpl> policies) {
            this.policies = ParameterPolicyListCopier.copyFromBuilder(policies);
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public ParameterHistory build() {
            return new ParameterHistory(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
