/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetParametersByPathRequest extends SsmRequest implements
        ToCopyableBuilder<GetParametersByPathRequest.Builder, GetParametersByPathRequest> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(GetParametersByPathRequest::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<Boolean> RECURSIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Recursive").getter(getter(GetParametersByPathRequest::recursive)).setter(setter(Builder::recursive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recursive").build()).build();

    private static final SdkField<List<ParameterStringFilter>> PARAMETER_FILTERS_FIELD = SdkField
            .<List<ParameterStringFilter>> builder(MarshallingType.LIST)
            .memberName("ParameterFilters")
            .getter(getter(GetParametersByPathRequest::parameterFilters))
            .setter(setter(Builder::parameterFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ParameterStringFilter> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterStringFilter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> WITH_DECRYPTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("WithDecryption").getter(getter(GetParametersByPathRequest::withDecryption))
            .setter(setter(Builder::withDecryption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WithDecryption").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResults").getter(getter(GetParametersByPathRequest::maxResults)).setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxResults").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(GetParametersByPathRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD, RECURSIVE_FIELD,
            PARAMETER_FILTERS_FIELD, WITH_DECRYPTION_FIELD, MAX_RESULTS_FIELD, NEXT_TOKEN_FIELD));

    private final String path;

    private final Boolean recursive;

    private final List<ParameterStringFilter> parameterFilters;

    private final Boolean withDecryption;

    private final Integer maxResults;

    private final String nextToken;

    private GetParametersByPathRequest(BuilderImpl builder) {
        super(builder);
        this.path = builder.path;
        this.recursive = builder.recursive;
        this.parameterFilters = builder.parameterFilters;
        this.withDecryption = builder.withDecryption;
        this.maxResults = builder.maxResults;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * The hierarchy for the parameter. Hierarchies start with a forward slash (/). The hierachy is the parameter name
     * except the last part of the parameter. For the API call to succeeed, the last part of the parameter name can't be
     * in the path. A parameter name hierarchy can have a maximum of 15 levels. Here is an example of a hierarchy:
     * <code>/Finance/Prod/IAD/WinServ2016/license33 </code>
     * </p>
     * 
     * @return The hierarchy for the parameter. Hierarchies start with a forward slash (/). The hierachy is the
     *         parameter name except the last part of the parameter. For the API call to succeeed, the last part of the
     *         parameter name can't be in the path. A parameter name hierarchy can have a maximum of 15 levels. Here is
     *         an example of a hierarchy: <code>/Finance/Prod/IAD/WinServ2016/license33 </code>
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * Retrieve all parameters within a hierarchy.
     * </p>
     * <important>
     * <p>
     * If a user has access to a path, then the user can access all levels of that path. For example, if a user has
     * permission to access path <code>/a</code>, then the user can also access <code>/a/b</code>. Even if a user has
     * explicitly been denied access in IAM for parameter <code>/a/b</code>, they can still call the GetParametersByPath
     * API operation recursively for <code>/a</code> and view <code>/a/b</code>.
     * </p>
     * </important>
     * 
     * @return Retrieve all parameters within a hierarchy.</p> <important>
     *         <p>
     *         If a user has access to a path, then the user can access all levels of that path. For example, if a user
     *         has permission to access path <code>/a</code>, then the user can also access <code>/a/b</code>. Even if a
     *         user has explicitly been denied access in IAM for parameter <code>/a/b</code>, they can still call the
     *         GetParametersByPath API operation recursively for <code>/a</code> and view <code>/a/b</code>.
     *         </p>
     */
    public final Boolean recursive() {
        return recursive;
    }

    /**
     * Returns true if the ParameterFilters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasParameterFilters() {
        return parameterFilters != null && !(parameterFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Filters to limit the request results.
     * </p>
     * <note>
     * <p>
     * For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
     * <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
     * </p>
     * <p>
     * The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>: <code>tag</code>,
     * <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameterFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Filters to limit the request results.</p> <note>
     *         <p>
     *         For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
     *         <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
     *         </p>
     *         <p>
     *         The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
     *         <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
     *         </p>
     */
    public final List<ParameterStringFilter> parameterFilters() {
        return parameterFilters;
    }

    /**
     * <p>
     * Retrieve all parameters in a hierarchy with their value decrypted.
     * </p>
     * 
     * @return Retrieve all parameters in a hierarchy with their value decrypted.
     */
    public final Boolean withDecryption() {
        return withDecryption;
    }

    /**
     * <p>
     * The maximum number of items to return for this call. The call also returns a token that you can specify in a
     * subsequent call to get the next set of results.
     * </p>
     * 
     * @return The maximum number of items to return for this call. The call also returns a token that you can specify
     *         in a subsequent call to get the next set of results.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    /**
     * <p>
     * A token to start the list. Use this token to get the next set of results.
     * </p>
     * 
     * @return A token to start the list. Use this token to get the next set of results.
     */
    public final String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(recursive());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameterFilters() ? parameterFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(withDecryption());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetParametersByPathRequest)) {
            return false;
        }
        GetParametersByPathRequest other = (GetParametersByPathRequest) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(recursive(), other.recursive())
                && hasParameterFilters() == other.hasParameterFilters()
                && Objects.equals(parameterFilters(), other.parameterFilters())
                && Objects.equals(withDecryption(), other.withDecryption()) && Objects.equals(maxResults(), other.maxResults())
                && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetParametersByPathRequest").add("Path", path()).add("Recursive", recursive())
                .add("ParameterFilters", hasParameterFilters() ? parameterFilters() : null)
                .add("WithDecryption", withDecryption()).add("MaxResults", maxResults()).add("NextToken", nextToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Recursive":
            return Optional.ofNullable(clazz.cast(recursive()));
        case "ParameterFilters":
            return Optional.ofNullable(clazz.cast(parameterFilters()));
        case "WithDecryption":
            return Optional.ofNullable(clazz.cast(withDecryption()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetParametersByPathRequest, T> g) {
        return obj -> g.apply((GetParametersByPathRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetParametersByPathRequest> {
        /**
         * <p>
         * The hierarchy for the parameter. Hierarchies start with a forward slash (/). The hierachy is the parameter
         * name except the last part of the parameter. For the API call to succeeed, the last part of the parameter name
         * can't be in the path. A parameter name hierarchy can have a maximum of 15 levels. Here is an example of a
         * hierarchy: <code>/Finance/Prod/IAD/WinServ2016/license33 </code>
         * </p>
         * 
         * @param path
         *        The hierarchy for the parameter. Hierarchies start with a forward slash (/). The hierachy is the
         *        parameter name except the last part of the parameter. For the API call to succeeed, the last part of
         *        the parameter name can't be in the path. A parameter name hierarchy can have a maximum of 15 levels.
         *        Here is an example of a hierarchy: <code>/Finance/Prod/IAD/WinServ2016/license33 </code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Retrieve all parameters within a hierarchy.
         * </p>
         * <important>
         * <p>
         * If a user has access to a path, then the user can access all levels of that path. For example, if a user has
         * permission to access path <code>/a</code>, then the user can also access <code>/a/b</code>. Even if a user
         * has explicitly been denied access in IAM for parameter <code>/a/b</code>, they can still call the
         * GetParametersByPath API operation recursively for <code>/a</code> and view <code>/a/b</code>.
         * </p>
         * </important>
         * 
         * @param recursive
         *        Retrieve all parameters within a hierarchy.</p> <important>
         *        <p>
         *        If a user has access to a path, then the user can access all levels of that path. For example, if a
         *        user has permission to access path <code>/a</code>, then the user can also access <code>/a/b</code>.
         *        Even if a user has explicitly been denied access in IAM for parameter <code>/a/b</code>, they can
         *        still call the GetParametersByPath API operation recursively for <code>/a</code> and view
         *        <code>/a/b</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recursive(Boolean recursive);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * <note>
         * <p>
         * For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
         * <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
         * </p>
         * <p>
         * The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
         * <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
         * </p>
         * </note>
         * 
         * @param parameterFilters
         *        Filters to limit the request results.</p> <note>
         *        <p>
         *        For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
         *        <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
         *        </p>
         *        <p>
         *        The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
         *        <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterFilters(Collection<ParameterStringFilter> parameterFilters);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * <note>
         * <p>
         * For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
         * <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
         * </p>
         * <p>
         * The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
         * <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
         * </p>
         * </note>
         * 
         * @param parameterFilters
         *        Filters to limit the request results.</p> <note>
         *        <p>
         *        For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
         *        <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
         *        </p>
         *        <p>
         *        The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
         *        <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterFilters(ParameterStringFilter... parameterFilters);

        /**
         * <p>
         * Filters to limit the request results.
         * </p>
         * <note>
         * <p>
         * For <code>GetParametersByPath</code>, the following filter <code>Key</code> names are supported:
         * <code>Type</code>, <code>KeyId</code>, <code>Label</code>, and <code>DataType</code>.
         * </p>
         * <p>
         * The following <code>Key</code> values are not supported for <code>GetParametersByPath</code>:
         * <code>tag</code>, <code>Name</code>, <code>Path</code>, and <code>Tier</code>.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<ParameterStringFilter>.Builder}
         * avoiding the need to create one manually via {@link List<ParameterStringFilter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ParameterStringFilter>.Builder#build()} is called
         * immediately and its result is passed to {@link #parameterFilters(List<ParameterStringFilter>)}.
         * 
         * @param parameterFilters
         *        a consumer that will call methods on {@link List<ParameterStringFilter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterFilters(List<ParameterStringFilter>)
         */
        Builder parameterFilters(Consumer<ParameterStringFilter.Builder>... parameterFilters);

        /**
         * <p>
         * Retrieve all parameters in a hierarchy with their value decrypted.
         * </p>
         * 
         * @param withDecryption
         *        Retrieve all parameters in a hierarchy with their value decrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder withDecryption(Boolean withDecryption);

        /**
         * <p>
         * The maximum number of items to return for this call. The call also returns a token that you can specify in a
         * subsequent call to get the next set of results.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of items to return for this call. The call also returns a token that you can
         *        specify in a subsequent call to get the next set of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        /**
         * <p>
         * A token to start the list. Use this token to get the next set of results.
         * </p>
         * 
         * @param nextToken
         *        A token to start the list. Use this token to get the next set of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String path;

        private Boolean recursive;

        private List<ParameterStringFilter> parameterFilters = DefaultSdkAutoConstructList.getInstance();

        private Boolean withDecryption;

        private Integer maxResults;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(GetParametersByPathRequest model) {
            super(model);
            path(model.path);
            recursive(model.recursive);
            parameterFilters(model.parameterFilters);
            withDecryption(model.withDecryption);
            maxResults(model.maxResults);
            nextToken(model.nextToken);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final Boolean getRecursive() {
            return recursive;
        }

        @Override
        public final Builder recursive(Boolean recursive) {
            this.recursive = recursive;
            return this;
        }

        public final void setRecursive(Boolean recursive) {
            this.recursive = recursive;
        }

        public final List<ParameterStringFilter.Builder> getParameterFilters() {
            List<ParameterStringFilter.Builder> result = ParameterStringFilterListCopier.copyToBuilder(this.parameterFilters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder parameterFilters(Collection<ParameterStringFilter> parameterFilters) {
            this.parameterFilters = ParameterStringFilterListCopier.copy(parameterFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterFilters(ParameterStringFilter... parameterFilters) {
            parameterFilters(Arrays.asList(parameterFilters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterFilters(Consumer<ParameterStringFilter.Builder>... parameterFilters) {
            parameterFilters(Stream.of(parameterFilters).map(c -> ParameterStringFilter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameterFilters(Collection<ParameterStringFilter.BuilderImpl> parameterFilters) {
            this.parameterFilters = ParameterStringFilterListCopier.copyFromBuilder(parameterFilters);
        }

        public final Boolean getWithDecryption() {
            return withDecryption;
        }

        @Override
        public final Builder withDecryption(Boolean withDecryption) {
            this.withDecryption = withDecryption;
            return this;
        }

        public final void setWithDecryption(Boolean withDecryption) {
            this.withDecryption = withDecryption;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetParametersByPathRequest build() {
            return new GetParametersByPathRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
