/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An array of search criteria that targets instances using a Key,Value combination that you specify.
 * </p>
 * <note>
 * <p>
 * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task, targets
 * are optional for other maintenance window task types (Automation, AWS Lambda, and AWS Step Functions). For more
 * information about running tasks that do not specify targets, see <a
 * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
 * >Registering maintenance window tasks without targets</a> in the <i>AWS Systems Manager User Guide</i>.
 * </p>
 * </note>
 * <p>
 * Supported formats include the following.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=<i>instance-id-1</i>,<i>instance-id-2</i>,<i>instance-id-3</i> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag:<i>my-tag-key</i>,Values=<i>my-tag-value-1</i>,<i>my-tag-value-2</i> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag-key,Values=<i>my-tag-key-1</i>,<i>my-tag-key-2</i> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Run Command and Maintenance window targets only</b>:
 * <code>Key=resource-groups:Name,Values=<i>resource-group-name</i> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Maintenance window targets only</b>:
 * <code>Key=resource-groups:ResourceTypeFilters,Values=<i>resource-type-1</i>,<i>resource-type-2</i> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Automation targets only</b>: <code>Key=ResourceGroup;Values=<i>resource-group-name</i> </code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * For example:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=i-02573cafcfEXAMPLE,i-0471e04240EXAMPLE,i-07782c72faEXAMPLE</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag:CostCenter,Values=CostCenter1,CostCenter2,CostCenter3</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag-key,Values=Name,Instance-Type,CostCenter</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Run Command and Maintenance window targets only</b>:
 * <code>Key=resource-groups:Name,Values=ProductionResourceGroup</code>
 * </p>
 * <p>
 * This example demonstrates how to target all resources in the resource group <b>ProductionResourceGroup</b> in your
 * maintenance window.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Maintenance window targets only</b>:
 * <code>Key=resource-groups:ResourceTypeFilters,Values=<i>AWS::EC2::INSTANCE</i>,<i>AWS::EC2::VPC</i> </code>
 * </p>
 * <p>
 * This example demonstrates how to target only EC2 instances and VPCs in your maintenance window.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Automation targets only</b>: <code>Key=ResourceGroup,Values=MyResourceGroup</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>State Manager association targets only</b>: <code>Key=InstanceIds,Values=<i>*</i> </code>
 * </p>
 * <p>
 * This example demonstrates how to target all managed instances in the AWS Region where the association was created.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information about how to send commands that target instances using <code>Key,Value</code> parameters, see <a
 * href
 * ="https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-targeting"
 * >Targeting multiple instances</a> in the <i>AWS Systems Manager User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Target implements SdkPojo, Serializable, ToCopyableBuilder<Target.Builder, Target> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(Target::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Key").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Target::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String key;

    private final List<String> values;

    private Target(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
    }

    /**
     * <p>
     * User-defined criteria for sending commands that target instances that meet the criteria.
     * </p>
     * 
     * @return User-defined criteria for sending commands that target instances that meet the criteria.
     */
    public final String key() {
        return key;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * User-defined criteria that maps to <code>Key</code>. For example, if you specified <code>tag:ServerRole</code>,
     * you could specify <code>value:WebServer</code> to run a command on instances that include EC2 tags of
     * <code>ServerRole,WebServer</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return User-defined criteria that maps to <code>Key</code>. For example, if you specified
     *         <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
     *         that include EC2 tags of <code>ServerRole,WebServer</code>.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Target)) {
            return false;
        }
        Target other = (Target) obj;
        return Objects.equals(key(), other.key()) && hasValues() == other.hasValues() && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Target").add("Key", key()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Target, T> g) {
        return obj -> g.apply((Target) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Target> {
        /**
         * <p>
         * User-defined criteria for sending commands that target instances that meet the criteria.
         * </p>
         * 
         * @param key
         *        User-defined criteria for sending commands that target instances that meet the criteria.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * User-defined criteria that maps to <code>Key</code>. For example, if you specified
         * <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
         * that include EC2 tags of <code>ServerRole,WebServer</code>.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to <code>Key</code>. For example, if you specified
         *        <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on
         *        instances that include EC2 tags of <code>ServerRole,WebServer</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * User-defined criteria that maps to <code>Key</code>. For example, if you specified
         * <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
         * that include EC2 tags of <code>ServerRole,WebServer</code>.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to <code>Key</code>. For example, if you specified
         *        <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on
         *        instances that include EC2 tags of <code>ServerRole,WebServer</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Target model) {
            key(model.key);
            values(model.values);
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
        }

        @Override
        public Target build() {
            return new Target(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
