/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the source of the data included in the resource data sync.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDataSyncSource implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceDataSyncSource.Builder, ResourceDataSyncSource> {
    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(ResourceDataSyncSource::sourceType)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<ResourceDataSyncAwsOrganizationsSource> AWS_ORGANIZATIONS_SOURCE_FIELD = SdkField
            .<ResourceDataSyncAwsOrganizationsSource> builder(MarshallingType.SDK_POJO).memberName("AwsOrganizationsSource")
            .getter(getter(ResourceDataSyncSource::awsOrganizationsSource)).setter(setter(Builder::awsOrganizationsSource))
            .constructor(ResourceDataSyncAwsOrganizationsSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsOrganizationsSource").build())
            .build();

    private static final SdkField<List<String>> SOURCE_REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SourceRegions")
            .getter(getter(ResourceDataSyncSource::sourceRegions))
            .setter(setter(Builder::sourceRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INCLUDE_FUTURE_REGIONS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeFutureRegions").getter(getter(ResourceDataSyncSource::includeFutureRegions))
            .setter(setter(Builder::includeFutureRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeFutureRegions").build())
            .build();

    private static final SdkField<Boolean> ENABLE_ALL_OPS_DATA_SOURCES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("EnableAllOpsDataSources")
            .getter(getter(ResourceDataSyncSource::enableAllOpsDataSources)).setter(setter(Builder::enableAllOpsDataSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableAllOpsDataSources").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(SOURCE_TYPE_FIELD, AWS_ORGANIZATIONS_SOURCE_FIELD, SOURCE_REGIONS_FIELD, INCLUDE_FUTURE_REGIONS_FIELD,
                    ENABLE_ALL_OPS_DATA_SOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sourceType;

    private final ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource;

    private final List<String> sourceRegions;

    private final Boolean includeFutureRegions;

    private final Boolean enableAllOpsDataSources;

    private ResourceDataSyncSource(BuilderImpl builder) {
        this.sourceType = builder.sourceType;
        this.awsOrganizationsSource = builder.awsOrganizationsSource;
        this.sourceRegions = builder.sourceRegions;
        this.includeFutureRegions = builder.includeFutureRegions;
        this.enableAllOpsDataSources = builder.enableAllOpsDataSources;
    }

    /**
     * <p>
     * The type of data source for the resource data sync. <code>SourceType</code> is either
     * <code>AwsOrganizations</code> (if an organization is present in AWS Organizations) or
     * <code>singleAccountMultiRegions</code>.
     * </p>
     * 
     * @return The type of data source for the resource data sync. <code>SourceType</code> is either
     *         <code>AwsOrganizations</code> (if an organization is present in AWS Organizations) or
     *         <code>singleAccountMultiRegions</code>.
     */
    public final String sourceType() {
        return sourceType;
    }

    /**
     * <p>
     * Information about the AwsOrganizationsSource resource data sync source. A sync source of this type can
     * synchronize data from AWS Organizations.
     * </p>
     * 
     * @return Information about the AwsOrganizationsSource resource data sync source. A sync source of this type can
     *         synchronize data from AWS Organizations.
     */
    public final ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource() {
        return awsOrganizationsSource;
    }

    /**
     * Returns true if the SourceRegions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSourceRegions() {
        return sourceRegions != null && !(sourceRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>SyncSource</code> AWS Regions included in the resource data sync.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourceRegions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The <code>SyncSource</code> AWS Regions included in the resource data sync.
     */
    public final List<String> sourceRegions() {
        return sourceRegions;
    }

    /**
     * <p>
     * Whether to automatically synchronize and aggregate data from new AWS Regions when those Regions come online.
     * </p>
     * 
     * @return Whether to automatically synchronize and aggregate data from new AWS Regions when those Regions come
     *         online.
     */
    public final Boolean includeFutureRegions() {
        return includeFutureRegions;
    }

    /**
     * <p>
     * When you create a resource data sync, if you choose one of the AWS Organizations options, then Systems Manager
     * automatically enables all OpsData sources in the selected AWS Regions for all AWS accounts in your organization
     * (or in the selected organization units). For more information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resouce-data-sync-multiple-accounts-and-regions.html"
     * >About multiple account and Region resource data syncs</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * 
     * @return When you create a resource data sync, if you choose one of the AWS Organizations options, then Systems
     *         Manager automatically enables all OpsData sources in the selected AWS Regions for all AWS accounts in
     *         your organization (or in the selected organization units). For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resouce-data-sync-multiple-accounts-and-regions.html"
     *         >About multiple account and Region resource data syncs</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public final Boolean enableAllOpsDataSources() {
        return enableAllOpsDataSources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceType());
        hashCode = 31 * hashCode + Objects.hashCode(awsOrganizationsSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceRegions() ? sourceRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(includeFutureRegions());
        hashCode = 31 * hashCode + Objects.hashCode(enableAllOpsDataSources());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncSource)) {
            return false;
        }
        ResourceDataSyncSource other = (ResourceDataSyncSource) obj;
        return Objects.equals(sourceType(), other.sourceType())
                && Objects.equals(awsOrganizationsSource(), other.awsOrganizationsSource())
                && hasSourceRegions() == other.hasSourceRegions() && Objects.equals(sourceRegions(), other.sourceRegions())
                && Objects.equals(includeFutureRegions(), other.includeFutureRegions())
                && Objects.equals(enableAllOpsDataSources(), other.enableAllOpsDataSources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceDataSyncSource").add("SourceType", sourceType())
                .add("AwsOrganizationsSource", awsOrganizationsSource())
                .add("SourceRegions", hasSourceRegions() ? sourceRegions() : null)
                .add("IncludeFutureRegions", includeFutureRegions()).add("EnableAllOpsDataSources", enableAllOpsDataSources())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceType()));
        case "AwsOrganizationsSource":
            return Optional.ofNullable(clazz.cast(awsOrganizationsSource()));
        case "SourceRegions":
            return Optional.ofNullable(clazz.cast(sourceRegions()));
        case "IncludeFutureRegions":
            return Optional.ofNullable(clazz.cast(includeFutureRegions()));
        case "EnableAllOpsDataSources":
            return Optional.ofNullable(clazz.cast(enableAllOpsDataSources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceDataSyncSource, T> g) {
        return obj -> g.apply((ResourceDataSyncSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDataSyncSource> {
        /**
         * <p>
         * The type of data source for the resource data sync. <code>SourceType</code> is either
         * <code>AwsOrganizations</code> (if an organization is present in AWS Organizations) or
         * <code>singleAccountMultiRegions</code>.
         * </p>
         * 
         * @param sourceType
         *        The type of data source for the resource data sync. <code>SourceType</code> is either
         *        <code>AwsOrganizations</code> (if an organization is present in AWS Organizations) or
         *        <code>singleAccountMultiRegions</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * Information about the AwsOrganizationsSource resource data sync source. A sync source of this type can
         * synchronize data from AWS Organizations.
         * </p>
         * 
         * @param awsOrganizationsSource
         *        Information about the AwsOrganizationsSource resource data sync source. A sync source of this type can
         *        synchronize data from AWS Organizations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource);

        /**
         * <p>
         * Information about the AwsOrganizationsSource resource data sync source. A sync source of this type can
         * synchronize data from AWS Organizations.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceDataSyncAwsOrganizationsSource.Builder}
         * avoiding the need to create one manually via {@link ResourceDataSyncAwsOrganizationsSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceDataSyncAwsOrganizationsSource.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource)}.
         * 
         * @param awsOrganizationsSource
         *        a consumer that will call methods on {@link ResourceDataSyncAwsOrganizationsSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource)
         */
        default Builder awsOrganizationsSource(Consumer<ResourceDataSyncAwsOrganizationsSource.Builder> awsOrganizationsSource) {
            return awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource.builder().applyMutation(awsOrganizationsSource)
                    .build());
        }

        /**
         * <p>
         * The <code>SyncSource</code> AWS Regions included in the resource data sync.
         * </p>
         * 
         * @param sourceRegions
         *        The <code>SyncSource</code> AWS Regions included in the resource data sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegions(Collection<String> sourceRegions);

        /**
         * <p>
         * The <code>SyncSource</code> AWS Regions included in the resource data sync.
         * </p>
         * 
         * @param sourceRegions
         *        The <code>SyncSource</code> AWS Regions included in the resource data sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegions(String... sourceRegions);

        /**
         * <p>
         * Whether to automatically synchronize and aggregate data from new AWS Regions when those Regions come online.
         * </p>
         * 
         * @param includeFutureRegions
         *        Whether to automatically synchronize and aggregate data from new AWS Regions when those Regions come
         *        online.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeFutureRegions(Boolean includeFutureRegions);

        /**
         * <p>
         * When you create a resource data sync, if you choose one of the AWS Organizations options, then Systems
         * Manager automatically enables all OpsData sources in the selected AWS Regions for all AWS accounts in your
         * organization (or in the selected organization units). For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resouce-data-sync-multiple-accounts-and-regions.html"
         * >About multiple account and Region resource data syncs</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param enableAllOpsDataSources
         *        When you create a resource data sync, if you choose one of the AWS Organizations options, then Systems
         *        Manager automatically enables all OpsData sources in the selected AWS Regions for all AWS accounts in
         *        your organization (or in the selected organization units). For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resouce-data-sync-multiple-accounts-and-regions.html"
         *        >About multiple account and Region resource data syncs</a> in the <i>AWS Systems Manager User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableAllOpsDataSources(Boolean enableAllOpsDataSources);
    }

    static final class BuilderImpl implements Builder {
        private String sourceType;

        private ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource;

        private List<String> sourceRegions = DefaultSdkAutoConstructList.getInstance();

        private Boolean includeFutureRegions;

        private Boolean enableAllOpsDataSources;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncSource model) {
            sourceType(model.sourceType);
            awsOrganizationsSource(model.awsOrganizationsSource);
            sourceRegions(model.sourceRegions);
            includeFutureRegions(model.includeFutureRegions);
            enableAllOpsDataSources(model.enableAllOpsDataSources);
        }

        public final String getSourceType() {
            return sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        public final ResourceDataSyncAwsOrganizationsSource.Builder getAwsOrganizationsSource() {
            return awsOrganizationsSource != null ? awsOrganizationsSource.toBuilder() : null;
        }

        @Override
        public final Builder awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource) {
            this.awsOrganizationsSource = awsOrganizationsSource;
            return this;
        }

        public final void setAwsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource.BuilderImpl awsOrganizationsSource) {
            this.awsOrganizationsSource = awsOrganizationsSource != null ? awsOrganizationsSource.build() : null;
        }

        public final Collection<String> getSourceRegions() {
            if (sourceRegions instanceof SdkAutoConstructList) {
                return null;
            }
            return sourceRegions;
        }

        @Override
        public final Builder sourceRegions(Collection<String> sourceRegions) {
            this.sourceRegions = ResourceDataSyncSourceRegionListCopier.copy(sourceRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRegions(String... sourceRegions) {
            sourceRegions(Arrays.asList(sourceRegions));
            return this;
        }

        public final void setSourceRegions(Collection<String> sourceRegions) {
            this.sourceRegions = ResourceDataSyncSourceRegionListCopier.copy(sourceRegions);
        }

        public final Boolean getIncludeFutureRegions() {
            return includeFutureRegions;
        }

        @Override
        public final Builder includeFutureRegions(Boolean includeFutureRegions) {
            this.includeFutureRegions = includeFutureRegions;
            return this;
        }

        public final void setIncludeFutureRegions(Boolean includeFutureRegions) {
            this.includeFutureRegions = includeFutureRegions;
        }

        public final Boolean getEnableAllOpsDataSources() {
            return enableAllOpsDataSources;
        }

        @Override
        public final Builder enableAllOpsDataSources(Boolean enableAllOpsDataSources) {
            this.enableAllOpsDataSources = enableAllOpsDataSources;
            return this;
        }

        public final void setEnableAllOpsDataSources(Boolean enableAllOpsDataSources) {
            this.enableAllOpsDataSources = enableAllOpsDataSources;
        }

        @Override
        public ResourceDataSyncSource build() {
            return new ResourceDataSyncSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
