/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the association version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssociationVersionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<AssociationVersionInfo.Builder, AssociationVersionInfo> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationId").getter(getter(AssociationVersionInfo::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> ASSOCIATION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationVersion").getter(getter(AssociationVersionInfo::associationVersion))
            .setter(setter(Builder::associationVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationVersion").build())
            .build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedDate").getter(getter(AssociationVersionInfo::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AssociationVersionInfo::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(AssociationVersionInfo::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(AssociationVersionInfo::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(AssociationVersionInfo::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduleExpression").getter(getter(AssociationVersionInfo::scheduleExpression))
            .setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<InstanceAssociationOutputLocation> OUTPUT_LOCATION_FIELD = SdkField
            .<InstanceAssociationOutputLocation> builder(MarshallingType.SDK_POJO).memberName("OutputLocation")
            .getter(getter(AssociationVersionInfo::outputLocation)).setter(setter(Builder::outputLocation))
            .constructor(InstanceAssociationOutputLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocation").build()).build();

    private static final SdkField<String> ASSOCIATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationName").getter(getter(AssociationVersionInfo::associationName))
            .setter(setter(Builder::associationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationName").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxErrors").getter(getter(AssociationVersionInfo::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxConcurrency").getter(getter(AssociationVersionInfo::maxConcurrency))
            .setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> COMPLIANCE_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComplianceSeverity").getter(getter(AssociationVersionInfo::complianceSeverityAsString))
            .setter(setter(Builder::complianceSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceSeverity").build())
            .build();

    private static final SdkField<String> SYNC_COMPLIANCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SyncCompliance").getter(getter(AssociationVersionInfo::syncComplianceAsString))
            .setter(setter(Builder::syncCompliance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncCompliance").build()).build();

    private static final SdkField<Boolean> APPLY_ONLY_AT_CRON_INTERVAL_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("ApplyOnlyAtCronInterval")
            .getter(getter(AssociationVersionInfo::applyOnlyAtCronInterval)).setter(setter(Builder::applyOnlyAtCronInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplyOnlyAtCronInterval").build())
            .build();

    private static final SdkField<List<String>> CALENDAR_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CalendarNames")
            .getter(getter(AssociationVersionInfo::calendarNames))
            .setter(setter(Builder::calendarNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CalendarNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TargetLocation>> TARGET_LOCATIONS_FIELD = SdkField
            .<List<TargetLocation>> builder(MarshallingType.LIST)
            .memberName("TargetLocations")
            .getter(getter(AssociationVersionInfo::targetLocations))
            .setter(setter(Builder::targetLocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            ASSOCIATION_VERSION_FIELD, CREATED_DATE_FIELD, NAME_FIELD, DOCUMENT_VERSION_FIELD, PARAMETERS_FIELD, TARGETS_FIELD,
            SCHEDULE_EXPRESSION_FIELD, OUTPUT_LOCATION_FIELD, ASSOCIATION_NAME_FIELD, MAX_ERRORS_FIELD, MAX_CONCURRENCY_FIELD,
            COMPLIANCE_SEVERITY_FIELD, SYNC_COMPLIANCE_FIELD, APPLY_ONLY_AT_CRON_INTERVAL_FIELD, CALENDAR_NAMES_FIELD,
            TARGET_LOCATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String associationVersion;

    private final Instant createdDate;

    private final String name;

    private final String documentVersion;

    private final Map<String, List<String>> parameters;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final String associationName;

    private final String maxErrors;

    private final String maxConcurrency;

    private final String complianceSeverity;

    private final String syncCompliance;

    private final Boolean applyOnlyAtCronInterval;

    private final List<String> calendarNames;

    private final List<TargetLocation> targetLocations;

    private AssociationVersionInfo(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.associationVersion = builder.associationVersion;
        this.createdDate = builder.createdDate;
        this.name = builder.name;
        this.documentVersion = builder.documentVersion;
        this.parameters = builder.parameters;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.associationName = builder.associationName;
        this.maxErrors = builder.maxErrors;
        this.maxConcurrency = builder.maxConcurrency;
        this.complianceSeverity = builder.complianceSeverity;
        this.syncCompliance = builder.syncCompliance;
        this.applyOnlyAtCronInterval = builder.applyOnlyAtCronInterval;
        this.calendarNames = builder.calendarNames;
        this.targetLocations = builder.targetLocations;
    }

    /**
     * <p>
     * The ID created by the system when the association was created.
     * </p>
     * 
     * @return The ID created by the system when the association was created.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public final String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The date the association version was created.
     * </p>
     * 
     * @return The date the association version was created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The name specified when the association was created.
     * </p>
     * 
     * @return The name specified when the association was created.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The version of an Amazon Web Services Systems Manager document (SSM document) used when the association version
     * was created.
     * </p>
     * 
     * @return The version of an Amazon Web Services Systems Manager document (SSM document) used when the association
     *         version was created.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Parameters specified when the association version was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Parameters specified when the association version was created.
     */
    public final Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * Returns true if the Targets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets specified for the association when the association version was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The targets specified for the association when the association version was created.
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The cron or rate schedule specified for the association when the association version was created.
     * </p>
     * 
     * @return The cron or rate schedule specified for the association when the association version was created.
     */
    public final String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The location in Amazon S3 specified for the association when the association version was created.
     * </p>
     * 
     * @return The location in Amazon S3 specified for the association when the association version was created.
     */
    public final InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * The name specified for the association version when the association version was created.
     * </p>
     * 
     * @return The name specified for the association version when the association version was created.
     */
    public final String associationName() {
        return associationName;
    }

    /**
     * <p>
     * The number of errors that are allowed before the system stops sending requests to run the association on
     * additional targets. You can specify either an absolute number of errors, for example 10, or a percentage of the
     * target set, for example 10%. If you specify 3, for example, the system stops sending requests when the fourth
     * error is received. If you specify 0, then the system stops sending requests after the first error is returned. If
     * you run an association on 50 instances and set <code>MaxError</code> to 10%, then the system stops sending the
     * request when the sixth error is received.
     * </p>
     * <p>
     * Executions that are already running an association when <code>MaxErrors</code> is reached are allowed to
     * complete, but some of these executions may fail as well. If you need to ensure that there won't be more than
     * max-errors failed executions, set <code>MaxConcurrency</code> to 1 so that executions proceed one at a time.
     * </p>
     * 
     * @return The number of errors that are allowed before the system stops sending requests to run the association on
     *         additional targets. You can specify either an absolute number of errors, for example 10, or a percentage
     *         of the target set, for example 10%. If you specify 3, for example, the system stops sending requests when
     *         the fourth error is received. If you specify 0, then the system stops sending requests after the first
     *         error is returned. If you run an association on 50 instances and set <code>MaxError</code> to 10%, then
     *         the system stops sending the request when the sixth error is received.</p>
     *         <p>
     *         Executions that are already running an association when <code>MaxErrors</code> is reached are allowed to
     *         complete, but some of these executions may fail as well. If you need to ensure that there won't be more
     *         than max-errors failed executions, set <code>MaxConcurrency</code> to 1 so that executions proceed one at
     *         a time.
     */
    public final String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The maximum number of targets allowed to run the association at the same time. You can specify a number, for
     * example 10, or a percentage of the target set, for example 10%. The default value is 100%, which means all
     * targets run the association at the same time.
     * </p>
     * <p>
     * If a new instance starts and attempts to run an association while Systems Manager is running
     * <code>MaxConcurrency</code> associations, the association is allowed to run. During the next association
     * interval, the new instance will process its association within the limit specified for
     * <code>MaxConcurrency</code>.
     * </p>
     * 
     * @return The maximum number of targets allowed to run the association at the same time. You can specify a number,
     *         for example 10, or a percentage of the target set, for example 10%. The default value is 100%, which
     *         means all targets run the association at the same time.</p>
     *         <p>
     *         If a new instance starts and attempts to run an association while Systems Manager is running
     *         <code>MaxConcurrency</code> associations, the association is allowed to run. During the next association
     *         interval, the new instance will process its association within the limit specified for
     *         <code>MaxConcurrency</code>.
     */
    public final String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The severity level that is assigned to the association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #complianceSeverity} will return {@link AssociationComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #complianceSeverityAsString}.
     * </p>
     * 
     * @return The severity level that is assigned to the association.
     * @see AssociationComplianceSeverity
     */
    public final AssociationComplianceSeverity complianceSeverity() {
        return AssociationComplianceSeverity.fromValue(complianceSeverity);
    }

    /**
     * <p>
     * The severity level that is assigned to the association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #complianceSeverity} will return {@link AssociationComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #complianceSeverityAsString}.
     * </p>
     * 
     * @return The severity level that is assigned to the association.
     * @see AssociationComplianceSeverity
     */
    public final String complianceSeverityAsString() {
        return complianceSeverity;
    }

    /**
     * <p>
     * The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>. In
     * <code>AUTO</code> mode, the system uses the status of the association execution to determine the compliance
     * status. If the association execution runs successfully, then the association is <code>COMPLIANT</code>. If the
     * association execution doesn't run successfully, the association is <code>NON-COMPLIANT</code>.
     * </p>
     * <p>
     * In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
     * <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
     * capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
     * <a>PutComplianceItems</a> API operation.
     * </p>
     * <p>
     * By default, all associations use <code>AUTO</code> mode.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #syncCompliance}
     * will return {@link AssociationSyncCompliance#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #syncComplianceAsString}.
     * </p>
     * 
     * @return The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>.
     *         In <code>AUTO</code> mode, the system uses the status of the association execution to determine the
     *         compliance status. If the association execution runs successfully, then the association is
     *         <code>COMPLIANT</code>. If the association execution doesn't run successfully, the association is
     *         <code>NON-COMPLIANT</code>.</p>
     *         <p>
     *         In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
     *         <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
     *         capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
     *         <a>PutComplianceItems</a> API operation.
     *         </p>
     *         <p>
     *         By default, all associations use <code>AUTO</code> mode.
     * @see AssociationSyncCompliance
     */
    public final AssociationSyncCompliance syncCompliance() {
        return AssociationSyncCompliance.fromValue(syncCompliance);
    }

    /**
     * <p>
     * The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>. In
     * <code>AUTO</code> mode, the system uses the status of the association execution to determine the compliance
     * status. If the association execution runs successfully, then the association is <code>COMPLIANT</code>. If the
     * association execution doesn't run successfully, the association is <code>NON-COMPLIANT</code>.
     * </p>
     * <p>
     * In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
     * <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
     * capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
     * <a>PutComplianceItems</a> API operation.
     * </p>
     * <p>
     * By default, all associations use <code>AUTO</code> mode.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #syncCompliance}
     * will return {@link AssociationSyncCompliance#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #syncComplianceAsString}.
     * </p>
     * 
     * @return The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>.
     *         In <code>AUTO</code> mode, the system uses the status of the association execution to determine the
     *         compliance status. If the association execution runs successfully, then the association is
     *         <code>COMPLIANT</code>. If the association execution doesn't run successfully, the association is
     *         <code>NON-COMPLIANT</code>.</p>
     *         <p>
     *         In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
     *         <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
     *         capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
     *         <a>PutComplianceItems</a> API operation.
     *         </p>
     *         <p>
     *         By default, all associations use <code>AUTO</code> mode.
     * @see AssociationSyncCompliance
     */
    public final String syncComplianceAsString() {
        return syncCompliance;
    }

    /**
     * <p>
     * By default, when you create a new associations, the system runs it immediately after it is created and then
     * according to the schedule you specified. Specify this option if you don't want an association to run immediately
     * after you create it. This parameter isn't supported for rate expressions.
     * </p>
     * 
     * @return By default, when you create a new associations, the system runs it immediately after it is created and
     *         then according to the schedule you specified. Specify this option if you don't want an association to run
     *         immediately after you create it. This parameter isn't supported for rate expressions.
     */
    public final Boolean applyOnlyAtCronInterval() {
        return applyOnlyAtCronInterval;
    }

    /**
     * Returns true if the CalendarNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasCalendarNames() {
        return calendarNames != null && !(calendarNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are gated
     * under. The associations for this version only run when that Change Calendar is open. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar">Amazon Web
     * Services Systems Manager Change Calendar</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCalendarNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are
     *         gated under. The associations for this version only run when that Change Calendar is open. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar"
     *         >Amazon Web Services Systems Manager Change Calendar</a>.
     */
    public final List<String> calendarNames() {
        return calendarNames;
    }

    /**
     * Returns true if the TargetLocations property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasTargetLocations() {
        return targetLocations != null && !(targetLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to run the
     * association when this association version was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetLocations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to run
     *         the association when this association version was created.
     */
    public final List<TargetLocation> targetLocations() {
        return targetLocations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(complianceSeverityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(syncComplianceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(applyOnlyAtCronInterval());
        hashCode = 31 * hashCode + Objects.hashCode(hasCalendarNames() ? calendarNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetLocations() ? targetLocations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociationVersionInfo)) {
            return false;
        }
        AssociationVersionInfo other = (AssociationVersionInfo) obj;
        return Objects.equals(associationId(), other.associationId())
                && Objects.equals(associationVersion(), other.associationVersion())
                && Objects.equals(createdDate(), other.createdDate()) && Objects.equals(name(), other.name())
                && Objects.equals(documentVersion(), other.documentVersion()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(associationName(), other.associationName()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(maxConcurrency(), other.maxConcurrency())
                && Objects.equals(complianceSeverityAsString(), other.complianceSeverityAsString())
                && Objects.equals(syncComplianceAsString(), other.syncComplianceAsString())
                && Objects.equals(applyOnlyAtCronInterval(), other.applyOnlyAtCronInterval())
                && hasCalendarNames() == other.hasCalendarNames() && Objects.equals(calendarNames(), other.calendarNames())
                && hasTargetLocations() == other.hasTargetLocations()
                && Objects.equals(targetLocations(), other.targetLocations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssociationVersionInfo").add("AssociationId", associationId())
                .add("AssociationVersion", associationVersion()).add("CreatedDate", createdDate()).add("Name", name())
                .add("DocumentVersion", documentVersion()).add("Parameters", hasParameters() ? parameters() : null)
                .add("Targets", hasTargets() ? targets() : null).add("ScheduleExpression", scheduleExpression())
                .add("OutputLocation", outputLocation()).add("AssociationName", associationName()).add("MaxErrors", maxErrors())
                .add("MaxConcurrency", maxConcurrency()).add("ComplianceSeverity", complianceSeverityAsString())
                .add("SyncCompliance", syncComplianceAsString()).add("ApplyOnlyAtCronInterval", applyOnlyAtCronInterval())
                .add("CalendarNames", hasCalendarNames() ? calendarNames() : null)
                .add("TargetLocations", hasTargetLocations() ? targetLocations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "AssociationVersion":
            return Optional.ofNullable(clazz.cast(associationVersion()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "AssociationName":
            return Optional.ofNullable(clazz.cast(associationName()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "ComplianceSeverity":
            return Optional.ofNullable(clazz.cast(complianceSeverityAsString()));
        case "SyncCompliance":
            return Optional.ofNullable(clazz.cast(syncComplianceAsString()));
        case "ApplyOnlyAtCronInterval":
            return Optional.ofNullable(clazz.cast(applyOnlyAtCronInterval()));
        case "CalendarNames":
            return Optional.ofNullable(clazz.cast(calendarNames()));
        case "TargetLocations":
            return Optional.ofNullable(clazz.cast(targetLocations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssociationVersionInfo, T> g) {
        return obj -> g.apply((AssociationVersionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssociationVersionInfo> {
        /**
         * <p>
         * The ID created by the system when the association was created.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when the association was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The date the association version was created.
         * </p>
         * 
         * @param createdDate
         *        The date the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The name specified when the association was created.
         * </p>
         * 
         * @param name
         *        The name specified when the association was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of an Amazon Web Services Systems Manager document (SSM document) used when the association
         * version was created.
         * </p>
         * 
         * @param documentVersion
         *        The version of an Amazon Web Services Systems Manager document (SSM document) used when the
         *        association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * Parameters specified when the association version was created.
         * </p>
         * 
         * @param parameters
         *        Parameters specified when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * 
         * @param targets
         *        The targets specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * 
         * @param targets
         *        The targets specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The cron or rate schedule specified for the association when the association version was created.
         * </p>
         * 
         * @param scheduleExpression
         *        The cron or rate schedule specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The location in Amazon S3 specified for the association when the association version was created.
         * </p>
         * 
         * @param outputLocation
         *        The location in Amazon S3 specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * The location in Amazon S3 specified for the association when the association version was created.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().applyMutation(outputLocation).build());
        }

        /**
         * <p>
         * The name specified for the association version when the association version was created.
         * </p>
         * 
         * @param associationName
         *        The name specified for the association version when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);

        /**
         * <p>
         * The number of errors that are allowed before the system stops sending requests to run the association on
         * additional targets. You can specify either an absolute number of errors, for example 10, or a percentage of
         * the target set, for example 10%. If you specify 3, for example, the system stops sending requests when the
         * fourth error is received. If you specify 0, then the system stops sending requests after the first error is
         * returned. If you run an association on 50 instances and set <code>MaxError</code> to 10%, then the system
         * stops sending the request when the sixth error is received.
         * </p>
         * <p>
         * Executions that are already running an association when <code>MaxErrors</code> is reached are allowed to
         * complete, but some of these executions may fail as well. If you need to ensure that there won't be more than
         * max-errors failed executions, set <code>MaxConcurrency</code> to 1 so that executions proceed one at a time.
         * </p>
         * 
         * @param maxErrors
         *        The number of errors that are allowed before the system stops sending requests to run the association
         *        on additional targets. You can specify either an absolute number of errors, for example 10, or a
         *        percentage of the target set, for example 10%. If you specify 3, for example, the system stops sending
         *        requests when the fourth error is received. If you specify 0, then the system stops sending requests
         *        after the first error is returned. If you run an association on 50 instances and set
         *        <code>MaxError</code> to 10%, then the system stops sending the request when the sixth error is
         *        received.</p>
         *        <p>
         *        Executions that are already running an association when <code>MaxErrors</code> is reached are allowed
         *        to complete, but some of these executions may fail as well. If you need to ensure that there won't be
         *        more than max-errors failed executions, set <code>MaxConcurrency</code> to 1 so that executions
         *        proceed one at a time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The maximum number of targets allowed to run the association at the same time. You can specify a number, for
         * example 10, or a percentage of the target set, for example 10%. The default value is 100%, which means all
         * targets run the association at the same time.
         * </p>
         * <p>
         * If a new instance starts and attempts to run an association while Systems Manager is running
         * <code>MaxConcurrency</code> associations, the association is allowed to run. During the next association
         * interval, the new instance will process its association within the limit specified for
         * <code>MaxConcurrency</code>.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of targets allowed to run the association at the same time. You can specify a
         *        number, for example 10, or a percentage of the target set, for example 10%. The default value is 100%,
         *        which means all targets run the association at the same time.</p>
         *        <p>
         *        If a new instance starts and attempts to run an association while Systems Manager is running
         *        <code>MaxConcurrency</code> associations, the association is allowed to run. During the next
         *        association interval, the new instance will process its association within the limit specified for
         *        <code>MaxConcurrency</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The severity level that is assigned to the association.
         * </p>
         * 
         * @param complianceSeverity
         *        The severity level that is assigned to the association.
         * @see AssociationComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationComplianceSeverity
         */
        Builder complianceSeverity(String complianceSeverity);

        /**
         * <p>
         * The severity level that is assigned to the association.
         * </p>
         * 
         * @param complianceSeverity
         *        The severity level that is assigned to the association.
         * @see AssociationComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationComplianceSeverity
         */
        Builder complianceSeverity(AssociationComplianceSeverity complianceSeverity);

        /**
         * <p>
         * The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>. In
         * <code>AUTO</code> mode, the system uses the status of the association execution to determine the compliance
         * status. If the association execution runs successfully, then the association is <code>COMPLIANT</code>. If
         * the association execution doesn't run successfully, the association is <code>NON-COMPLIANT</code>.
         * </p>
         * <p>
         * In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
         * <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
         * capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
         * <a>PutComplianceItems</a> API operation.
         * </p>
         * <p>
         * By default, all associations use <code>AUTO</code> mode.
         * </p>
         * 
         * @param syncCompliance
         *        The mode for generating association compliance. You can specify <code>AUTO</code> or
         *        <code>MANUAL</code>. In <code>AUTO</code> mode, the system uses the status of the association
         *        execution to determine the compliance status. If the association execution runs successfully, then the
         *        association is <code>COMPLIANT</code>. If the association execution doesn't run successfully, the
         *        association is <code>NON-COMPLIANT</code>.</p>
         *        <p>
         *        In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
         *        <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager,
         *        a capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
         *        <a>PutComplianceItems</a> API operation.
         *        </p>
         *        <p>
         *        By default, all associations use <code>AUTO</code> mode.
         * @see AssociationSyncCompliance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationSyncCompliance
         */
        Builder syncCompliance(String syncCompliance);

        /**
         * <p>
         * The mode for generating association compliance. You can specify <code>AUTO</code> or <code>MANUAL</code>. In
         * <code>AUTO</code> mode, the system uses the status of the association execution to determine the compliance
         * status. If the association execution runs successfully, then the association is <code>COMPLIANT</code>. If
         * the association execution doesn't run successfully, the association is <code>NON-COMPLIANT</code>.
         * </p>
         * <p>
         * In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
         * <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager, a
         * capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
         * <a>PutComplianceItems</a> API operation.
         * </p>
         * <p>
         * By default, all associations use <code>AUTO</code> mode.
         * </p>
         * 
         * @param syncCompliance
         *        The mode for generating association compliance. You can specify <code>AUTO</code> or
         *        <code>MANUAL</code>. In <code>AUTO</code> mode, the system uses the status of the association
         *        execution to determine the compliance status. If the association execution runs successfully, then the
         *        association is <code>COMPLIANT</code>. If the association execution doesn't run successfully, the
         *        association is <code>NON-COMPLIANT</code>.</p>
         *        <p>
         *        In <code>MANUAL</code> mode, you must specify the <code>AssociationId</code> as a parameter for the
         *        <a>PutComplianceItems</a> API operation. In this case, compliance data isn't managed by State Manager,
         *        a capability of Amazon Web Services Systems Manager. It is managed by your direct call to the
         *        <a>PutComplianceItems</a> API operation.
         *        </p>
         *        <p>
         *        By default, all associations use <code>AUTO</code> mode.
         * @see AssociationSyncCompliance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationSyncCompliance
         */
        Builder syncCompliance(AssociationSyncCompliance syncCompliance);

        /**
         * <p>
         * By default, when you create a new associations, the system runs it immediately after it is created and then
         * according to the schedule you specified. Specify this option if you don't want an association to run
         * immediately after you create it. This parameter isn't supported for rate expressions.
         * </p>
         * 
         * @param applyOnlyAtCronInterval
         *        By default, when you create a new associations, the system runs it immediately after it is created and
         *        then according to the schedule you specified. Specify this option if you don't want an association to
         *        run immediately after you create it. This parameter isn't supported for rate expressions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applyOnlyAtCronInterval(Boolean applyOnlyAtCronInterval);

        /**
         * <p>
         * The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are gated
         * under. The associations for this version only run when that Change Calendar is open. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar">Amazon
         * Web Services Systems Manager Change Calendar</a>.
         * </p>
         * 
         * @param calendarNames
         *        The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are
         *        gated under. The associations for this version only run when that Change Calendar is open. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar"
         *        >Amazon Web Services Systems Manager Change Calendar</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calendarNames(Collection<String> calendarNames);

        /**
         * <p>
         * The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are gated
         * under. The associations for this version only run when that Change Calendar is open. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar">Amazon
         * Web Services Systems Manager Change Calendar</a>.
         * </p>
         * 
         * @param calendarNames
         *        The names or Amazon Resource Names (ARNs) of the Change Calendar type documents your associations are
         *        gated under. The associations for this version only run when that Change Calendar is open. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-change-calendar"
         *        >Amazon Web Services Systems Manager Change Calendar</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calendarNames(String... calendarNames);

        /**
         * <p>
         * The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to run the
         * association when this association version was created.
         * </p>
         * 
         * @param targetLocations
         *        The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to
         *        run the association when this association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocations(Collection<TargetLocation> targetLocations);

        /**
         * <p>
         * The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to run the
         * association when this association version was created.
         * </p>
         * 
         * @param targetLocations
         *        The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to
         *        run the association when this association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocations(TargetLocation... targetLocations);

        /**
         * <p>
         * The combination of Amazon Web Services Regions and Amazon Web Services accounts where you wanted to run the
         * association when this association version was created.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TargetLocation>.Builder} avoiding the need
         * to create one manually via {@link List<TargetLocation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TargetLocation>.Builder#build()} is called immediately and
         * its result is passed to {@link #targetLocations(List<TargetLocation>)}.
         * 
         * @param targetLocations
         *        a consumer that will call methods on {@link List<TargetLocation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetLocations(List<TargetLocation>)
         */
        Builder targetLocations(Consumer<TargetLocation.Builder>... targetLocations);
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String associationVersion;

        private Instant createdDate;

        private String name;

        private String documentVersion;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private String associationName;

        private String maxErrors;

        private String maxConcurrency;

        private String complianceSeverity;

        private String syncCompliance;

        private Boolean applyOnlyAtCronInterval;

        private List<String> calendarNames = DefaultSdkAutoConstructList.getInstance();

        private List<TargetLocation> targetLocations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AssociationVersionInfo model) {
            associationId(model.associationId);
            associationVersion(model.associationVersion);
            createdDate(model.createdDate);
            name(model.name);
            documentVersion(model.documentVersion);
            parameters(model.parameters);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            associationName(model.associationName);
            maxErrors(model.maxErrors);
            maxConcurrency(model.maxConcurrency);
            complianceSeverity(model.complianceSeverity);
            syncCompliance(model.syncCompliance);
            applyOnlyAtCronInterval(model.applyOnlyAtCronInterval);
            calendarNames(model.calendarNames);
            targetLocations(model.targetLocations);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        @Transient
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        @Override
        @Transient
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        @Transient
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        @Transient
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        @Override
        @Transient
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        @Transient
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        @Override
        @Transient
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        @Override
        @Transient
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final String getAssociationName() {
            return associationName;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        @Transient
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        @Override
        @Transient
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        @Override
        @Transient
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final String getComplianceSeverity() {
            return complianceSeverity;
        }

        public final void setComplianceSeverity(String complianceSeverity) {
            this.complianceSeverity = complianceSeverity;
        }

        @Override
        @Transient
        public final Builder complianceSeverity(String complianceSeverity) {
            this.complianceSeverity = complianceSeverity;
            return this;
        }

        @Override
        @Transient
        public final Builder complianceSeverity(AssociationComplianceSeverity complianceSeverity) {
            this.complianceSeverity(complianceSeverity == null ? null : complianceSeverity.toString());
            return this;
        }

        public final String getSyncCompliance() {
            return syncCompliance;
        }

        public final void setSyncCompliance(String syncCompliance) {
            this.syncCompliance = syncCompliance;
        }

        @Override
        @Transient
        public final Builder syncCompliance(String syncCompliance) {
            this.syncCompliance = syncCompliance;
            return this;
        }

        @Override
        @Transient
        public final Builder syncCompliance(AssociationSyncCompliance syncCompliance) {
            this.syncCompliance(syncCompliance == null ? null : syncCompliance.toString());
            return this;
        }

        public final Boolean getApplyOnlyAtCronInterval() {
            return applyOnlyAtCronInterval;
        }

        public final void setApplyOnlyAtCronInterval(Boolean applyOnlyAtCronInterval) {
            this.applyOnlyAtCronInterval = applyOnlyAtCronInterval;
        }

        @Override
        @Transient
        public final Builder applyOnlyAtCronInterval(Boolean applyOnlyAtCronInterval) {
            this.applyOnlyAtCronInterval = applyOnlyAtCronInterval;
            return this;
        }

        public final Collection<String> getCalendarNames() {
            if (calendarNames instanceof SdkAutoConstructList) {
                return null;
            }
            return calendarNames;
        }

        public final void setCalendarNames(Collection<String> calendarNames) {
            this.calendarNames = CalendarNameOrARNListCopier.copy(calendarNames);
        }

        @Override
        @Transient
        public final Builder calendarNames(Collection<String> calendarNames) {
            this.calendarNames = CalendarNameOrARNListCopier.copy(calendarNames);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder calendarNames(String... calendarNames) {
            calendarNames(Arrays.asList(calendarNames));
            return this;
        }

        public final List<TargetLocation.Builder> getTargetLocations() {
            List<TargetLocation.Builder> result = TargetLocationsCopier.copyToBuilder(this.targetLocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargetLocations(Collection<TargetLocation.BuilderImpl> targetLocations) {
            this.targetLocations = TargetLocationsCopier.copyFromBuilder(targetLocations);
        }

        @Override
        @Transient
        public final Builder targetLocations(Collection<TargetLocation> targetLocations) {
            this.targetLocations = TargetLocationsCopier.copy(targetLocations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targetLocations(TargetLocation... targetLocations) {
            targetLocations(Arrays.asList(targetLocations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targetLocations(Consumer<TargetLocation.Builder>... targetLocations) {
            targetLocations(Stream.of(targetLocations).map(c -> TargetLocation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public AssociationVersionInfo build() {
            return new AssociationVersionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
