/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An invocation is a copy of a command sent to a specific managed node. A command can apply to one or more managed
 * nodes. A command invocation applies to one managed node. For example, if a user runs <code>SendCommand</code> against
 * three managed nodes, then a command invocation is created for each requested managed node ID. A command invocation
 * returns status and detail information about a command you ran.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CommandInvocation implements SdkPojo, Serializable,
        ToCopyableBuilder<CommandInvocation.Builder, CommandInvocation> {
    private static final SdkField<String> COMMAND_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CommandId").getter(getter(CommandInvocation::commandId)).setter(setter(Builder::commandId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CommandId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(CommandInvocation::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceName").getter(getter(CommandInvocation::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceName").build()).build();

    private static final SdkField<String> COMMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Comment")
            .getter(getter(CommandInvocation::comment)).setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment").build()).build();

    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentName").getter(getter(CommandInvocation::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(CommandInvocation::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<Instant> REQUESTED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RequestedDateTime").getter(getter(CommandInvocation::requestedDateTime))
            .setter(setter(Builder::requestedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedDateTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(CommandInvocation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusDetails").getter(getter(CommandInvocation::statusDetails)).setter(setter(Builder::statusDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final SdkField<String> TRACE_OUTPUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TraceOutput").getter(getter(CommandInvocation::traceOutput)).setter(setter(Builder::traceOutput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TraceOutput").build()).build();

    private static final SdkField<String> STANDARD_OUTPUT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandardOutputUrl").getter(getter(CommandInvocation::standardOutputUrl))
            .setter(setter(Builder::standardOutputUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardOutputUrl").build()).build();

    private static final SdkField<String> STANDARD_ERROR_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandardErrorUrl").getter(getter(CommandInvocation::standardErrorUrl))
            .setter(setter(Builder::standardErrorUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardErrorUrl").build()).build();

    private static final SdkField<List<CommandPlugin>> COMMAND_PLUGINS_FIELD = SdkField
            .<List<CommandPlugin>> builder(MarshallingType.LIST)
            .memberName("CommandPlugins")
            .getter(getter(CommandInvocation::commandPlugins))
            .setter(setter(Builder::commandPlugins))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CommandPlugins").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CommandPlugin> builder(MarshallingType.SDK_POJO)
                                            .constructor(CommandPlugin::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceRole").getter(getter(CommandInvocation::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRole").build()).build();

    private static final SdkField<NotificationConfig> NOTIFICATION_CONFIG_FIELD = SdkField
            .<NotificationConfig> builder(MarshallingType.SDK_POJO).memberName("NotificationConfig")
            .getter(getter(CommandInvocation::notificationConfig)).setter(setter(Builder::notificationConfig))
            .constructor(NotificationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationConfig").build())
            .build();

    private static final SdkField<CloudWatchOutputConfig> CLOUD_WATCH_OUTPUT_CONFIG_FIELD = SdkField
            .<CloudWatchOutputConfig> builder(MarshallingType.SDK_POJO).memberName("CloudWatchOutputConfig")
            .getter(getter(CommandInvocation::cloudWatchOutputConfig)).setter(setter(Builder::cloudWatchOutputConfig))
            .constructor(CloudWatchOutputConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchOutputConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_ID_FIELD,
            INSTANCE_ID_FIELD, INSTANCE_NAME_FIELD, COMMENT_FIELD, DOCUMENT_NAME_FIELD, DOCUMENT_VERSION_FIELD,
            REQUESTED_DATE_TIME_FIELD, STATUS_FIELD, STATUS_DETAILS_FIELD, TRACE_OUTPUT_FIELD, STANDARD_OUTPUT_URL_FIELD,
            STANDARD_ERROR_URL_FIELD, COMMAND_PLUGINS_FIELD, SERVICE_ROLE_FIELD, NOTIFICATION_CONFIG_FIELD,
            CLOUD_WATCH_OUTPUT_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String commandId;

    private final String instanceId;

    private final String instanceName;

    private final String comment;

    private final String documentName;

    private final String documentVersion;

    private final Instant requestedDateTime;

    private final String status;

    private final String statusDetails;

    private final String traceOutput;

    private final String standardOutputUrl;

    private final String standardErrorUrl;

    private final List<CommandPlugin> commandPlugins;

    private final String serviceRole;

    private final NotificationConfig notificationConfig;

    private final CloudWatchOutputConfig cloudWatchOutputConfig;

    private CommandInvocation(BuilderImpl builder) {
        this.commandId = builder.commandId;
        this.instanceId = builder.instanceId;
        this.instanceName = builder.instanceName;
        this.comment = builder.comment;
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.requestedDateTime = builder.requestedDateTime;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.traceOutput = builder.traceOutput;
        this.standardOutputUrl = builder.standardOutputUrl;
        this.standardErrorUrl = builder.standardErrorUrl;
        this.commandPlugins = builder.commandPlugins;
        this.serviceRole = builder.serviceRole;
        this.notificationConfig = builder.notificationConfig;
        this.cloudWatchOutputConfig = builder.cloudWatchOutputConfig;
    }

    /**
     * <p>
     * The command against which this invocation was requested.
     * </p>
     * 
     * @return The command against which this invocation was requested.
     */
    public final String commandId() {
        return commandId;
    }

    /**
     * <p>
     * The managed node ID in which this invocation was requested.
     * </p>
     * 
     * @return The managed node ID in which this invocation was requested.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The fully qualified host name of the managed node.
     * </p>
     * 
     * @return The fully qualified host name of the managed node.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * User-specified information about the command, such as a brief description of what the command should do.
     * </p>
     * 
     * @return User-specified information about the command, such as a brief description of what the command should do.
     */
    public final String comment() {
        return comment;
    }

    /**
     * <p>
     * The document name that was requested for execution.
     * </p>
     * 
     * @return The document name that was requested for execution.
     */
    public final String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The Systems Manager document (SSM document) version.
     * </p>
     * 
     * @return The Systems Manager document (SSM document) version.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The time and date the request was sent to this managed node.
     * </p>
     * 
     * @return The time and date the request was sent to this managed node.
     */
    public final Instant requestedDateTime() {
        return requestedDateTime;
    }

    /**
     * <p>
     * Whether or not the invocation succeeded, failed, or is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandInvocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not the invocation succeeded, failed, or is pending.
     * @see CommandInvocationStatus
     */
    public final CommandInvocationStatus status() {
        return CommandInvocationStatus.fromValue(status);
    }

    /**
     * <p>
     * Whether or not the invocation succeeded, failed, or is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandInvocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not the invocation succeeded, failed, or is pending.
     * @see CommandInvocationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A detailed status of the command execution for each invocation (each managed node targeted by the command).
     * StatusDetails includes more information than Status because it includes states resulting from error and
     * concurrency control parameters. StatusDetails can show different results than Status. For more information about
     * these statuses, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding command
     * statuses</a> in the <i>Amazon Web Services Systems Manager User Guide</i>. StatusDetails can be one of the
     * following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The command hasn't been sent to the managed node.
     * </p>
     * </li>
     * <li>
     * <p>
     * In Progress: The command has been sent to the managed node but hasn't reached a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery Timed Out: The command wasn't delivered to the managed node before the delivery timeout expired.
     * Delivery timeouts don't count against the parent command's <code>MaxErrors</code> limit, but they do contribute
     * to whether the parent command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Execution Timed Out: Command execution started on the managed node, but the execution wasn't complete before the
     * execution timeout expired. Execution timeouts count against the <code>MaxErrors</code> limit of the parent
     * command. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The command wasn't successful on the managed node. For a plugin, this indicates that the result code
     * wasn't zero. For a command invocation, this indicates that the result code for one or more plugins wasn't zero.
     * Invocation failures count against the <code>MaxErrors</code> limit of the parent command. This is a terminal
     * state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cancelled: The command was terminated before it was completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Undeliverable: The command can't be delivered to the managed node. The managed node might not exist or might not
     * be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit and don't
     * contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled by
     * the system. This is a terminal state.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed status of the command execution for each invocation (each managed node targeted by the
     *         command). StatusDetails includes more information than Status because it includes states resulting from
     *         error and concurrency control parameters. StatusDetails can show different results than Status. For more
     *         information about these statuses, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
     *         command statuses</a> in the <i>Amazon Web Services Systems Manager User Guide</i>. StatusDetails can be
     *         one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The command hasn't been sent to the managed node.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In Progress: The command has been sent to the managed node but hasn't reached a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delivery Timed Out: The command wasn't delivered to the managed node before the delivery timeout expired.
     *         Delivery timeouts don't count against the parent command's <code>MaxErrors</code> limit, but they do
     *         contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Execution Timed Out: Command execution started on the managed node, but the execution wasn't complete
     *         before the execution timeout expired. Execution timeouts count against the <code>MaxErrors</code> limit
     *         of the parent command. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The command wasn't successful on the managed node. For a plugin, this indicates that the result
     *         code wasn't zero. For a command invocation, this indicates that the result code for one or more plugins
     *         wasn't zero. Invocation failures count against the <code>MaxErrors</code> limit of the parent command.
     *         This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cancelled: The command was terminated before it was completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Undeliverable: The command can't be delivered to the managed node. The managed node might not exist or
     *         might not be responding. Undeliverable invocations don't count against the parent command's MaxErrors
     *         limit and don't contribute to whether the parent command status is Success or Incomplete. This is a
     *         terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
     *         canceled by the system. This is a terminal state.
     *         </p>
     *         </li>
     */
    public final String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * Gets the trace output sent by the agent.
     * </p>
     * 
     * @return Gets the trace output sent by the agent.
     */
    public final String traceOutput() {
        return traceOutput;
    }

    /**
     * <p>
     * The URL to the plugin's StdOut file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was defined
     * for the parent command. For an invocation, <code>StandardOutputUrl</code> is populated if there is just one
     * plugin defined for the command, and the S3 bucket was defined for the command.
     * </p>
     * 
     * @return The URL to the plugin's StdOut file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
     *         defined for the parent command. For an invocation, <code>StandardOutputUrl</code> is populated if there
     *         is just one plugin defined for the command, and the S3 bucket was defined for the command.
     */
    public final String standardOutputUrl() {
        return standardOutputUrl;
    }

    /**
     * <p>
     * The URL to the plugin's StdErr file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was defined
     * for the parent command. For an invocation, <code>StandardErrorUrl</code> is populated if there is just one plugin
     * defined for the command, and the S3 bucket was defined for the command.
     * </p>
     * 
     * @return The URL to the plugin's StdErr file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
     *         defined for the parent command. For an invocation, <code>StandardErrorUrl</code> is populated if there is
     *         just one plugin defined for the command, and the S3 bucket was defined for the command.
     */
    public final String standardErrorUrl() {
        return standardErrorUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the CommandPlugins property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCommandPlugins() {
        return commandPlugins != null && !(commandPlugins instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Plugins processed by the command.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommandPlugins} method.
     * </p>
     * 
     * @return Plugins processed by the command.
     */
    public final List<CommandPlugin> commandPlugins() {
        return commandPlugins;
    }

    /**
     * <p>
     * The Identity and Access Management (IAM) service role that Run Command, a capability of Amazon Web Services
     * Systems Manager, uses to act on your behalf when sending notifications about command status changes on a per
     * managed node basis.
     * </p>
     * 
     * @return The Identity and Access Management (IAM) service role that Run Command, a capability of Amazon Web
     *         Services Systems Manager, uses to act on your behalf when sending notifications about command status
     *         changes on a per managed node basis.
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Configurations for sending notifications about command status changes on a per managed node basis.
     * </p>
     * 
     * @return Configurations for sending notifications about command status changes on a per managed node basis.
     */
    public final NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    /**
     * <p>
     * Amazon CloudWatch Logs information where you want Amazon Web Services Systems Manager to send the command output.
     * </p>
     * 
     * @return Amazon CloudWatch Logs information where you want Amazon Web Services Systems Manager to send the command
     *         output.
     */
    public final CloudWatchOutputConfig cloudWatchOutputConfig() {
        return cloudWatchOutputConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(commandId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(requestedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(traceOutput());
        hashCode = 31 * hashCode + Objects.hashCode(standardOutputUrl());
        hashCode = 31 * hashCode + Objects.hashCode(standardErrorUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommandPlugins() ? commandPlugins() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchOutputConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CommandInvocation)) {
            return false;
        }
        CommandInvocation other = (CommandInvocation) obj;
        return Objects.equals(commandId(), other.commandId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(instanceName(), other.instanceName()) && Objects.equals(comment(), other.comment())
                && Objects.equals(documentName(), other.documentName())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(requestedDateTime(), other.requestedDateTime())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetails(), other.statusDetails()) && Objects.equals(traceOutput(), other.traceOutput())
                && Objects.equals(standardOutputUrl(), other.standardOutputUrl())
                && Objects.equals(standardErrorUrl(), other.standardErrorUrl())
                && hasCommandPlugins() == other.hasCommandPlugins() && Objects.equals(commandPlugins(), other.commandPlugins())
                && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(notificationConfig(), other.notificationConfig())
                && Objects.equals(cloudWatchOutputConfig(), other.cloudWatchOutputConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CommandInvocation").add("CommandId", commandId()).add("InstanceId", instanceId())
                .add("InstanceName", instanceName()).add("Comment", comment()).add("DocumentName", documentName())
                .add("DocumentVersion", documentVersion()).add("RequestedDateTime", requestedDateTime())
                .add("Status", statusAsString()).add("StatusDetails", statusDetails()).add("TraceOutput", traceOutput())
                .add("StandardOutputUrl", standardOutputUrl()).add("StandardErrorUrl", standardErrorUrl())
                .add("CommandPlugins", hasCommandPlugins() ? commandPlugins() : null).add("ServiceRole", serviceRole())
                .add("NotificationConfig", notificationConfig()).add("CloudWatchOutputConfig", cloudWatchOutputConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CommandId":
            return Optional.ofNullable(clazz.cast(commandId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "InstanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "RequestedDateTime":
            return Optional.ofNullable(clazz.cast(requestedDateTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "TraceOutput":
            return Optional.ofNullable(clazz.cast(traceOutput()));
        case "StandardOutputUrl":
            return Optional.ofNullable(clazz.cast(standardOutputUrl()));
        case "StandardErrorUrl":
            return Optional.ofNullable(clazz.cast(standardErrorUrl()));
        case "CommandPlugins":
            return Optional.ofNullable(clazz.cast(commandPlugins()));
        case "ServiceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "NotificationConfig":
            return Optional.ofNullable(clazz.cast(notificationConfig()));
        case "CloudWatchOutputConfig":
            return Optional.ofNullable(clazz.cast(cloudWatchOutputConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CommandInvocation, T> g) {
        return obj -> g.apply((CommandInvocation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CommandInvocation> {
        /**
         * <p>
         * The command against which this invocation was requested.
         * </p>
         * 
         * @param commandId
         *        The command against which this invocation was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandId(String commandId);

        /**
         * <p>
         * The managed node ID in which this invocation was requested.
         * </p>
         * 
         * @param instanceId
         *        The managed node ID in which this invocation was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The fully qualified host name of the managed node.
         * </p>
         * 
         * @param instanceName
         *        The fully qualified host name of the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * User-specified information about the command, such as a brief description of what the command should do.
         * </p>
         * 
         * @param comment
         *        User-specified information about the command, such as a brief description of what the command should
         *        do.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * The document name that was requested for execution.
         * </p>
         * 
         * @param documentName
         *        The document name that was requested for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The Systems Manager document (SSM document) version.
         * </p>
         * 
         * @param documentVersion
         *        The Systems Manager document (SSM document) version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The time and date the request was sent to this managed node.
         * </p>
         * 
         * @param requestedDateTime
         *        The time and date the request was sent to this managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDateTime(Instant requestedDateTime);

        /**
         * <p>
         * Whether or not the invocation succeeded, failed, or is pending.
         * </p>
         * 
         * @param status
         *        Whether or not the invocation succeeded, failed, or is pending.
         * @see CommandInvocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandInvocationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Whether or not the invocation succeeded, failed, or is pending.
         * </p>
         * 
         * @param status
         *        Whether or not the invocation succeeded, failed, or is pending.
         * @see CommandInvocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandInvocationStatus
         */
        Builder status(CommandInvocationStatus status);

        /**
         * <p>
         * A detailed status of the command execution for each invocation (each managed node targeted by the command).
         * StatusDetails includes more information than Status because it includes states resulting from error and
         * concurrency control parameters. StatusDetails can show different results than Status. For more information
         * about these statuses, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
         * command statuses</a> in the <i>Amazon Web Services Systems Manager User Guide</i>. StatusDetails can be one
         * of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The command hasn't been sent to the managed node.
         * </p>
         * </li>
         * <li>
         * <p>
         * In Progress: The command has been sent to the managed node but hasn't reached a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Delivery Timed Out: The command wasn't delivered to the managed node before the delivery timeout expired.
         * Delivery timeouts don't count against the parent command's <code>MaxErrors</code> limit, but they do
         * contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Execution Timed Out: Command execution started on the managed node, but the execution wasn't complete before
         * the execution timeout expired. Execution timeouts count against the <code>MaxErrors</code> limit of the
         * parent command. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The command wasn't successful on the managed node. For a plugin, this indicates that the result code
         * wasn't zero. For a command invocation, this indicates that the result code for one or more plugins wasn't
         * zero. Invocation failures count against the <code>MaxErrors</code> limit of the parent command. This is a
         * terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cancelled: The command was terminated before it was completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Undeliverable: The command can't be delivered to the managed node. The managed node might not exist or might
         * not be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit and
         * don't contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled
         * by the system. This is a terminal state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetails
         *        A detailed status of the command execution for each invocation (each managed node targeted by the
         *        command). StatusDetails includes more information than Status because it includes states resulting
         *        from error and concurrency control parameters. StatusDetails can show different results than Status.
         *        For more information about these statuses, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html"
         *        >Understanding command statuses</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         *        StatusDetails can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The command hasn't been sent to the managed node.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In Progress: The command has been sent to the managed node but hasn't reached a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delivery Timed Out: The command wasn't delivered to the managed node before the delivery timeout
         *        expired. Delivery timeouts don't count against the parent command's <code>MaxErrors</code> limit, but
         *        they do contribute to whether the parent command status is Success or Incomplete. This is a terminal
         *        state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Execution Timed Out: Command execution started on the managed node, but the execution wasn't complete
         *        before the execution timeout expired. Execution timeouts count against the <code>MaxErrors</code>
         *        limit of the parent command. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The command wasn't successful on the managed node. For a plugin, this indicates that the
         *        result code wasn't zero. For a command invocation, this indicates that the result code for one or more
         *        plugins wasn't zero. Invocation failures count against the <code>MaxErrors</code> limit of the parent
         *        command. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cancelled: The command was terminated before it was completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Undeliverable: The command can't be delivered to the managed node. The managed node might not exist or
         *        might not be responding. Undeliverable invocations don't count against the parent command's MaxErrors
         *        limit and don't contribute to whether the parent command status is Success or Incomplete. This is a
         *        terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
         *        canceled by the system. This is a terminal state.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * Gets the trace output sent by the agent.
         * </p>
         * 
         * @param traceOutput
         *        Gets the trace output sent by the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traceOutput(String traceOutput);

        /**
         * <p>
         * The URL to the plugin's StdOut file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
         * defined for the parent command. For an invocation, <code>StandardOutputUrl</code> is populated if there is
         * just one plugin defined for the command, and the S3 bucket was defined for the command.
         * </p>
         * 
         * @param standardOutputUrl
         *        The URL to the plugin's StdOut file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
         *        defined for the parent command. For an invocation, <code>StandardOutputUrl</code> is populated if
         *        there is just one plugin defined for the command, and the S3 bucket was defined for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardOutputUrl(String standardOutputUrl);

        /**
         * <p>
         * The URL to the plugin's StdErr file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
         * defined for the parent command. For an invocation, <code>StandardErrorUrl</code> is populated if there is
         * just one plugin defined for the command, and the S3 bucket was defined for the command.
         * </p>
         * 
         * @param standardErrorUrl
         *        The URL to the plugin's StdErr file in Amazon Simple Storage Service (Amazon S3), if the S3 bucket was
         *        defined for the parent command. For an invocation, <code>StandardErrorUrl</code> is populated if there
         *        is just one plugin defined for the command, and the S3 bucket was defined for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardErrorUrl(String standardErrorUrl);

        /**
         * <p>
         * Plugins processed by the command.
         * </p>
         * 
         * @param commandPlugins
         *        Plugins processed by the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandPlugins(Collection<CommandPlugin> commandPlugins);

        /**
         * <p>
         * Plugins processed by the command.
         * </p>
         * 
         * @param commandPlugins
         *        Plugins processed by the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandPlugins(CommandPlugin... commandPlugins);

        /**
         * <p>
         * Plugins processed by the command.
         * </p>
         * This is a convenience that creates an instance of the {@link List<CommandPlugin>.Builder} avoiding the need
         * to create one manually via {@link List<CommandPlugin>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CommandPlugin>.Builder#build()} is called immediately and
         * its result is passed to {@link #commandPlugins(List<CommandPlugin>)}.
         * 
         * @param commandPlugins
         *        a consumer that will call methods on {@link List<CommandPlugin>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #commandPlugins(List<CommandPlugin>)
         */
        Builder commandPlugins(Consumer<CommandPlugin.Builder>... commandPlugins);

        /**
         * <p>
         * The Identity and Access Management (IAM) service role that Run Command, a capability of Amazon Web Services
         * Systems Manager, uses to act on your behalf when sending notifications about command status changes on a per
         * managed node basis.
         * </p>
         * 
         * @param serviceRole
         *        The Identity and Access Management (IAM) service role that Run Command, a capability of Amazon Web
         *        Services Systems Manager, uses to act on your behalf when sending notifications about command status
         *        changes on a per managed node basis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per managed node basis.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications about command status changes on a per managed node basis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per managed node basis.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationConfig.Builder} avoiding the need to
         * create one manually via {@link NotificationConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #notificationConfig(NotificationConfig)}.
         * 
         * @param notificationConfig
         *        a consumer that will call methods on {@link NotificationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfig(NotificationConfig)
         */
        default Builder notificationConfig(Consumer<NotificationConfig.Builder> notificationConfig) {
            return notificationConfig(NotificationConfig.builder().applyMutation(notificationConfig).build());
        }

        /**
         * <p>
         * Amazon CloudWatch Logs information where you want Amazon Web Services Systems Manager to send the command
         * output.
         * </p>
         * 
         * @param cloudWatchOutputConfig
         *        Amazon CloudWatch Logs information where you want Amazon Web Services Systems Manager to send the
         *        command output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig);

        /**
         * <p>
         * Amazon CloudWatch Logs information where you want Amazon Web Services Systems Manager to send the command
         * output.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchOutputConfig.Builder} avoiding the
         * need to create one manually via {@link CloudWatchOutputConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchOutputConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchOutputConfig(CloudWatchOutputConfig)}.
         * 
         * @param cloudWatchOutputConfig
         *        a consumer that will call methods on {@link CloudWatchOutputConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchOutputConfig(CloudWatchOutputConfig)
         */
        default Builder cloudWatchOutputConfig(Consumer<CloudWatchOutputConfig.Builder> cloudWatchOutputConfig) {
            return cloudWatchOutputConfig(CloudWatchOutputConfig.builder().applyMutation(cloudWatchOutputConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String commandId;

        private String instanceId;

        private String instanceName;

        private String comment;

        private String documentName;

        private String documentVersion;

        private Instant requestedDateTime;

        private String status;

        private String statusDetails;

        private String traceOutput;

        private String standardOutputUrl;

        private String standardErrorUrl;

        private List<CommandPlugin> commandPlugins = DefaultSdkAutoConstructList.getInstance();

        private String serviceRole;

        private NotificationConfig notificationConfig;

        private CloudWatchOutputConfig cloudWatchOutputConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(CommandInvocation model) {
            commandId(model.commandId);
            instanceId(model.instanceId);
            instanceName(model.instanceName);
            comment(model.comment);
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            requestedDateTime(model.requestedDateTime);
            status(model.status);
            statusDetails(model.statusDetails);
            traceOutput(model.traceOutput);
            standardOutputUrl(model.standardOutputUrl);
            standardErrorUrl(model.standardErrorUrl);
            commandPlugins(model.commandPlugins);
            serviceRole(model.serviceRole);
            notificationConfig(model.notificationConfig);
            cloudWatchOutputConfig(model.cloudWatchOutputConfig);
        }

        public final String getCommandId() {
            return commandId;
        }

        public final void setCommandId(String commandId) {
            this.commandId = commandId;
        }

        @Override
        @Transient
        public final Builder commandId(String commandId) {
            this.commandId = commandId;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        @Transient
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        @Transient
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        @Transient
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final String getDocumentName() {
            return documentName;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        @Override
        @Transient
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        @Transient
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final Instant getRequestedDateTime() {
            return requestedDateTime;
        }

        public final void setRequestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
        }

        @Override
        @Transient
        public final Builder requestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(CommandInvocationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        @Override
        @Transient
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final String getTraceOutput() {
            return traceOutput;
        }

        public final void setTraceOutput(String traceOutput) {
            this.traceOutput = traceOutput;
        }

        @Override
        @Transient
        public final Builder traceOutput(String traceOutput) {
            this.traceOutput = traceOutput;
            return this;
        }

        public final String getStandardOutputUrl() {
            return standardOutputUrl;
        }

        public final void setStandardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
        }

        @Override
        @Transient
        public final Builder standardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
            return this;
        }

        public final String getStandardErrorUrl() {
            return standardErrorUrl;
        }

        public final void setStandardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
        }

        @Override
        @Transient
        public final Builder standardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
            return this;
        }

        public final List<CommandPlugin.Builder> getCommandPlugins() {
            List<CommandPlugin.Builder> result = CommandPluginListCopier.copyToBuilder(this.commandPlugins);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCommandPlugins(Collection<CommandPlugin.BuilderImpl> commandPlugins) {
            this.commandPlugins = CommandPluginListCopier.copyFromBuilder(commandPlugins);
        }

        @Override
        @Transient
        public final Builder commandPlugins(Collection<CommandPlugin> commandPlugins) {
            this.commandPlugins = CommandPluginListCopier.copy(commandPlugins);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder commandPlugins(CommandPlugin... commandPlugins) {
            commandPlugins(Arrays.asList(commandPlugins));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder commandPlugins(Consumer<CommandPlugin.Builder>... commandPlugins) {
            commandPlugins(Stream.of(commandPlugins).map(c -> CommandPlugin.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        @Override
        @Transient
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final NotificationConfig.Builder getNotificationConfig() {
            return notificationConfig != null ? notificationConfig.toBuilder() : null;
        }

        public final void setNotificationConfig(NotificationConfig.BuilderImpl notificationConfig) {
            this.notificationConfig = notificationConfig != null ? notificationConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final CloudWatchOutputConfig.Builder getCloudWatchOutputConfig() {
            return cloudWatchOutputConfig != null ? cloudWatchOutputConfig.toBuilder() : null;
        }

        public final void setCloudWatchOutputConfig(CloudWatchOutputConfig.BuilderImpl cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig != null ? cloudWatchOutputConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig;
            return this;
        }

        @Override
        public CommandInvocation build() {
            return new CommandInvocation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
