/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutParameterRequest extends SsmRequest implements
        ToCopyableBuilder<PutParameterRequest.Builder, PutParameterRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PutParameterRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(PutParameterRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Value")
            .getter(getter(PutParameterRequest::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(PutParameterRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(PutParameterRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<Boolean> OVERWRITE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Overwrite").getter(getter(PutParameterRequest::overwrite)).setter(setter(Builder::overwrite))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overwrite").build()).build();

    private static final SdkField<String> ALLOWED_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllowedPattern").getter(getter(PutParameterRequest::allowedPattern))
            .setter(setter(Builder::allowedPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedPattern").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(PutParameterRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Tier")
            .getter(getter(PutParameterRequest::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<String> POLICIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Policies").getter(getter(PutParameterRequest::policies)).setter(setter(Builder::policies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build()).build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataType").getter(getter(PutParameterRequest::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            VALUE_FIELD, TYPE_FIELD, KEY_ID_FIELD, OVERWRITE_FIELD, ALLOWED_PATTERN_FIELD, TAGS_FIELD, TIER_FIELD,
            POLICIES_FIELD, DATA_TYPE_FIELD));

    private final String name;

    private final String description;

    private final String value;

    private final String type;

    private final String keyId;

    private final Boolean overwrite;

    private final String allowedPattern;

    private final List<Tag> tags;

    private final String tier;

    private final String policies;

    private final String dataType;

    private PutParameterRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.value = builder.value;
        this.type = builder.type;
        this.keyId = builder.keyId;
        this.overwrite = builder.overwrite;
        this.allowedPattern = builder.allowedPattern;
        this.tags = builder.tags;
        this.tier = builder.tier;
        this.policies = builder.policies;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * The fully qualified name of the parameter that you want to add to the system. The fully qualified name includes
     * the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must include a leading
     * forward slash character (/) when you create or reference a parameter. For example:
     * <code>/Dev/DBServer/MySQL/db-string13</code>
     * </p>
     * <p>
     * Naming Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Parameter names are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name must be unique within an Amazon Web Services Region
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name can't be prefixed with "<code>aws</code>" or "<code>ssm</code>" (case-insensitive).
     * </p>
     * </li>
     * <li>
     * <p>
     * Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-</code>
     * </p>
     * <p>
     * In addition, the slash character ( / ) is used to delineate hierarchies in parameter names. For example:
     * <code>/Dev/Production/East/Project-ABC/MyParameter</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * A parameter name can't include spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * Parameter hierarchies are limited to a maximum depth of fifteen levels.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For additional information about valid values for parameter names, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-paramstore-su-create.html">Creating
     * Systems Manager parameters</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The maximum length constraint of 2048 characters listed below includes 1037 characters reserved for internal use
     * by Systems Manager. The maximum length for a parameter name that you create is 1011 characters. This includes the
     * characters in the ARN that precede the name you specify, such as
     * <code>arn:aws:ssm:us-east-2:111122223333:parameter/</code>.
     * </p>
     * </note>
     * 
     * @return The fully qualified name of the parameter that you want to add to the system. The fully qualified name
     *         includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must
     *         include a leading forward slash character (/) when you create or reference a parameter. For example:
     *         <code>/Dev/DBServer/MySQL/db-string13</code> </p>
     *         <p>
     *         Naming Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Parameter names are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name must be unique within an Amazon Web Services Region
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name can't be prefixed with "<code>aws</code>" or "<code>ssm</code>" (case-insensitive).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-</code>
     *         </p>
     *         <p>
     *         In addition, the slash character ( / ) is used to delineate hierarchies in parameter names. For example:
     *         <code>/Dev/Production/East/Project-ABC/MyParameter</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A parameter name can't include spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Parameter hierarchies are limited to a maximum depth of fifteen levels.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For additional information about valid values for parameter names, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-paramstore-su-create.html"
     *         >Creating Systems Manager parameters</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         The maximum length constraint of 2048 characters listed below includes 1037 characters reserved for
     *         internal use by Systems Manager. The maximum length for a parameter name that you create is 1011
     *         characters. This includes the characters in the ARN that precede the name you specify, such as
     *         <code>arn:aws:ssm:us-east-2:111122223333:parameter/</code>.
     *         </p>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Information about the parameter that you want to add to the system. Optional but recommended.
     * </p>
     * <important>
     * <p>
     * Don't enter personally identifiable information in this field.
     * </p>
     * </important>
     * 
     * @return Information about the parameter that you want to add to the system. Optional but recommended.</p>
     *         <important>
     *         <p>
     *         Don't enter personally identifiable information in this field.
     *         </p>
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB. Advanced
     * parameters have a value limit of 8 KB.
     * </p>
     * <note>
     * <p>
     * Parameters can't be referenced or nested in the values of other parameters. You can't include <code>{{}}</code>
     * or <code>{{ssm:<i>parameter-name</i>}}</code> in a parameter value.
     * </p>
     * </note>
     * 
     * @return The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB.
     *         Advanced parameters have a value limit of 8 KB.</p> <note>
     *         <p>
     *         Parameters can't be referenced or nested in the values of other parameters. You can't include
     *         <code>{{}}</code> or <code>{{ssm:<i>parameter-name</i>}}</code> in a parameter value.
     *         </p>
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * The type of parameter that you want to add to the system.
     * </p>
     * <note>
     * <p>
     * <code>SecureString</code> isn't currently supported for CloudFormation templates.
     * </p>
     * </note>
     * <p>
     * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or special
     * character to escape items in the list. If you have a parameter value that requires a comma, then use the
     * <code>String</code> data type.
     * </p>
     * <important>
     * <p>
     * Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type when
     * creating a parameter.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter that you want to add to the system.</p> <note>
     *         <p>
     *         <code>SecureString</code> isn't currently supported for CloudFormation templates.
     *         </p>
     *         </note>
     *         <p>
     *         Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
     *         special character to escape items in the list. If you have a parameter value that requires a comma, then
     *         use the <code>String</code> data type.
     *         </p>
     *         <important>
     *         <p>
     *         Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type
     *         when creating a parameter.
     *         </p>
     * @see ParameterType
     */
    public final ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter that you want to add to the system.
     * </p>
     * <note>
     * <p>
     * <code>SecureString</code> isn't currently supported for CloudFormation templates.
     * </p>
     * </note>
     * <p>
     * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or special
     * character to escape items in the list. If you have a parameter value that requires a comma, then use the
     * <code>String</code> data type.
     * </p>
     * <important>
     * <p>
     * Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type when
     * creating a parameter.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter that you want to add to the system.</p> <note>
     *         <p>
     *         <code>SecureString</code> isn't currently supported for CloudFormation templates.
     *         </p>
     *         </note>
     *         <p>
     *         Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
     *         special character to escape items in the list. If you have a parameter value that requires a comma, then
     *         use the <code>String</code> data type.
     *         </p>
     *         <important>
     *         <p>
     *         Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type
     *         when creating a parameter.
     *         </p>
     * @see ParameterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Key Management Service (KMS) ID that you want to use to encrypt a parameter. Either the default KMS key
     * automatically assigned to your Amazon Web Services account or a custom key. Required for parameters that use the
     * <code>SecureString</code> data type.
     * </p>
     * <p>
     * If you don't specify a key ID, the system uses the default key associated with your Amazon Web Services account.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To use your default KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify the
     * <code>Key ID</code> when you create the parameter. The system automatically populates <code>Key ID</code> with
     * your default KMS key.
     * </p>
     * </li>
     * <li>
     * <p>
     * To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code> parameter.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Key Management Service (KMS) ID that you want to use to encrypt a parameter. Either the default KMS
     *         key automatically assigned to your Amazon Web Services account or a custom key. Required for parameters
     *         that use the <code>SecureString</code> data type.</p>
     *         <p>
     *         If you don't specify a key ID, the system uses the default key associated with your Amazon Web Services
     *         account.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To use your default KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify
     *         the <code>Key ID</code> when you create the parameter. The system automatically populates
     *         <code>Key ID</code> with your default KMS key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
     *         parameter.
     *         </p>
     *         </li>
     */
    public final String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Overwrite an existing parameter. The default value is <code>false</code>.
     * </p>
     * 
     * @return Overwrite an existing parameter. The default value is <code>false</code>.
     */
    public final Boolean overwrite() {
        return overwrite;
    }

    /**
     * <p>
     * A regular expression used to validate the parameter value. For example, for String types with values restricted
     * to numbers, you can specify the following: AllowedPattern=^\d+$
     * </p>
     * 
     * @return A regular expression used to validate the parameter value. For example, for String types with values
     *         restricted to numbers, you can specify the following: AllowedPattern=^\d+$
     */
    public final String allowedPattern() {
        return allowedPattern;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways, such
     * as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to identify
     * the type of resource to which it applies, the environment, or the type of configuration data referenced by the
     * parameter. In this case, you could specify the following key-value pairs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Key=Resource,Value=S3bucket</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Key=OS,Value=Windows</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Key=ParameterType,Value=LicenseKey</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
     *         ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
     *         parameter to identify the type of resource to which it applies, the environment, or the type of
     *         configuration data referenced by the parameter. In this case, you could specify the following key-value
     *         pairs:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Key=Resource,Value=S3bucket</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Key=OS,Value=Windows</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Key=ParameterType,Value=LicenseKey</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
     *         </p>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The parameter tier to assign to a parameter.
     * </p>
     * <p>
     * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a content
     * size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of 10,000 standard
     * parameters for each Region in an Amazon Web Services account. Standard parameters are offered at no additional
     * cost.
     * </p>
     * <p>
     * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You can
     * create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services account. Advanced
     * parameters incur a charge. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     * >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced parameter
     * to a standard parameter. Reverting an advanced parameter to a standard parameter would result in data loss
     * because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also remove any
     * policies attached to the parameter. Lastly, advanced parameters use a different form of encryption than standard
     * parameters.
     * </p>
     * <p>
     * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced parameter,
     * you must delete it and recreate it as a new standard parameter.
     * </p>
     * <p>
     * <b>Using the Default Tier Configuration</b>
     * </p>
     * <p>
     * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you specify
     * a tier in the request, Parameter Store creates or updates the parameter according to that request. However, if
     * you don't specify a tier in a request, Parameter Store assigns the tier based on the current Parameter Store
     * default tier configuration.
     * </p>
     * <p>
     * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     * advanced-parameter tier, you can specify one of the following as the default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     * parameter is standard or advanced.
     * </p>
     * <p>
     * If the request doesn't include any options that require an advanced parameter, the parameter is created in the
     * standard-parameter tier. If one or more options requiring an advanced parameter are included in the request,
     * Parameter Store create a parameter in the advanced-parameter tier.
     * </p>
     * <p>
     * This approach helps control your parameter-related costs by always creating standard parameters unless an
     * advanced parameter is necessary.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Options that require an advanced parameter include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The content size of the parameter is more than 4 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The parameter uses a parameter policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web Services
     * Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about configuring the default tier option, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
     * parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier to assign to a parameter.</p>
     *         <p>
     *         Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
     *         content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
     *         10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters are
     *         offered at no additional cost.
     *         </p>
     *         <p>
     *         Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
     *         You can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services
     *         account. Advanced parameters incur a charge. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     *         >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     *         </p>
     *         <p>
     *         You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
     *         parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result
     *         in data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting
     *         would also remove any policies attached to the parameter. Lastly, advanced parameters use a different
     *         form of encryption than standard parameters.
     *         </p>
     *         <p>
     *         If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
     *         parameter, you must delete it and recreate it as a new standard parameter.
     *         </p>
     *         <p>
     *         <b>Using the Default Tier Configuration</b>
     *         </p>
     *         <p>
     *         In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
     *         specify a tier in the request, Parameter Store creates or updates the parameter according to that
     *         request. However, if you don't specify a tier in a request, Parameter Store assigns the tier based on the
     *         current Parameter Store default tier configuration.
     *         </p>
     *         <p>
     *         The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     *         advanced-parameter tier, you can specify one of the following as the default:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     *         parameter is standard or advanced.
     *         </p>
     *         <p>
     *         If the request doesn't include any options that require an advanced parameter, the parameter is created
     *         in the standard-parameter tier. If one or more options requiring an advanced parameter are included in
     *         the request, Parameter Store create a parameter in the advanced-parameter tier.
     *         </p>
     *         <p>
     *         This approach helps control your parameter-related costs by always creating standard parameters unless an
     *         advanced parameter is necessary.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Options that require an advanced parameter include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The content size of the parameter is more than 4 KB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The parameter uses a parameter policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web
     *         Services Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about configuring the default tier option, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
     *         default parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see ParameterTier
     */
    public final ParameterTier tier() {
        return ParameterTier.fromValue(tier);
    }

    /**
     * <p>
     * The parameter tier to assign to a parameter.
     * </p>
     * <p>
     * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a content
     * size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of 10,000 standard
     * parameters for each Region in an Amazon Web Services account. Standard parameters are offered at no additional
     * cost.
     * </p>
     * <p>
     * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You can
     * create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services account. Advanced
     * parameters incur a charge. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     * >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced parameter
     * to a standard parameter. Reverting an advanced parameter to a standard parameter would result in data loss
     * because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also remove any
     * policies attached to the parameter. Lastly, advanced parameters use a different form of encryption than standard
     * parameters.
     * </p>
     * <p>
     * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced parameter,
     * you must delete it and recreate it as a new standard parameter.
     * </p>
     * <p>
     * <b>Using the Default Tier Configuration</b>
     * </p>
     * <p>
     * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you specify
     * a tier in the request, Parameter Store creates or updates the parameter according to that request. However, if
     * you don't specify a tier in a request, Parameter Store assigns the tier based on the current Parameter Store
     * default tier configuration.
     * </p>
     * <p>
     * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     * advanced-parameter tier, you can specify one of the following as the default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     * parameter is standard or advanced.
     * </p>
     * <p>
     * If the request doesn't include any options that require an advanced parameter, the parameter is created in the
     * standard-parameter tier. If one or more options requiring an advanced parameter are included in the request,
     * Parameter Store create a parameter in the advanced-parameter tier.
     * </p>
     * <p>
     * This approach helps control your parameter-related costs by always creating standard parameters unless an
     * advanced parameter is necessary.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Options that require an advanced parameter include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The content size of the parameter is more than 4 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The parameter uses a parameter policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web Services
     * Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about configuring the default tier option, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
     * parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier to assign to a parameter.</p>
     *         <p>
     *         Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
     *         content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
     *         10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters are
     *         offered at no additional cost.
     *         </p>
     *         <p>
     *         Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
     *         You can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services
     *         account. Advanced parameters incur a charge. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
     *         >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     *         </p>
     *         <p>
     *         You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
     *         parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result
     *         in data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting
     *         would also remove any policies attached to the parameter. Lastly, advanced parameters use a different
     *         form of encryption than standard parameters.
     *         </p>
     *         <p>
     *         If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
     *         parameter, you must delete it and recreate it as a new standard parameter.
     *         </p>
     *         <p>
     *         <b>Using the Default Tier Configuration</b>
     *         </p>
     *         <p>
     *         In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
     *         specify a tier in the request, Parameter Store creates or updates the parameter according to that
     *         request. However, if you don't specify a tier in a request, Parameter Store assigns the tier based on the
     *         current Parameter Store default tier configuration.
     *         </p>
     *         <p>
     *         The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
     *         advanced-parameter tier, you can specify one of the following as the default:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
     *         parameter is standard or advanced.
     *         </p>
     *         <p>
     *         If the request doesn't include any options that require an advanced parameter, the parameter is created
     *         in the standard-parameter tier. If one or more options requiring an advanced parameter are included in
     *         the request, Parameter Store create a parameter in the advanced-parameter tier.
     *         </p>
     *         <p>
     *         This approach helps control your parameter-related costs by always creating standard parameters unless an
     *         advanced parameter is necessary.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Options that require an advanced parameter include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The content size of the parameter is more than 4 KB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The parameter uses a parameter policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web
     *         Services Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about configuring the default tier option, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
     *         default parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see ParameterTier
     */
    public final String tierAsString() {
        return tier;
    }

    /**
     * <p>
     * One or more policies to apply to a parameter. This operation takes a JSON array. Parameter Store, a capability of
     * Amazon Web Services Systems Manager supports the following policy types:
     * </p>
     * <p>
     * Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify the
     * expiration date. You can update the expiration date and time by updating the policy. Updating the
     * <i>parameter</i> doesn't affect the expiration date and time. When the expiration time is reached, Parameter
     * Store deletes the parameter.
     * </p>
     * <p>
     * ExpirationNotification: This policy initiates an event in Amazon CloudWatch Events that notifies you about the
     * expiration. By using this policy, you can receive notification before or after the expiration time is reached, in
     * units of days or hours.
     * </p>
     * <p>
     * NoChangeNotification: This policy initiates a CloudWatch Events event if a parameter hasn't been modified for a
     * specified period of time. This policy type is useful when, for example, a secret needs to be changed within a
     * period of time, but it hasn't been changed.
     * </p>
     * <p>
     * All existing policies are preserved until you send new policies or an empty policy. For more information about
     * parameter policies, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
     * parameter policies</a>.
     * </p>
     * 
     * @return One or more policies to apply to a parameter. This operation takes a JSON array. Parameter Store, a
     *         capability of Amazon Web Services Systems Manager supports the following policy types:</p>
     *         <p>
     *         Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify
     *         the expiration date. You can update the expiration date and time by updating the policy. Updating the
     *         <i>parameter</i> doesn't affect the expiration date and time. When the expiration time is reached,
     *         Parameter Store deletes the parameter.
     *         </p>
     *         <p>
     *         ExpirationNotification: This policy initiates an event in Amazon CloudWatch Events that notifies you
     *         about the expiration. By using this policy, you can receive notification before or after the expiration
     *         time is reached, in units of days or hours.
     *         </p>
     *         <p>
     *         NoChangeNotification: This policy initiates a CloudWatch Events event if a parameter hasn't been modified
     *         for a specified period of time. This policy type is useful when, for example, a secret needs to be
     *         changed within a period of time, but it hasn't been changed.
     *         </p>
     *         <p>
     *         All existing policies are preserved until you send new policies or an empty policy. For more information
     *         about parameter policies, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html"
     *         >Assigning parameter policies</a>.
     */
    public final String policies() {
        return policies;
    }

    /**
     * <p>
     * The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon Machine
     * Image (AMI) IDs.
     * </p>
     * <p>
     * <b>The following data type values are supported.</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>text</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>aws:ec2:image</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>aws:ssm:integration</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Amazon Web Services
     * Systems Manager validates the parameter value is in the required format, such as
     * <code>ami-12345abcdeEXAMPLE</code>, and that the specified AMI is available in your Amazon Web Services account.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html">Native
     * parameter support for Amazon Machine Image (AMI) IDs</a> in the <i>Amazon Web Services Systems Manager User
     * Guide</i>.
     * </p>
     * 
     * @return The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon
     *         Machine Image (AMI) IDs.</p>
     *         <p>
     *         <b>The following data type values are supported.</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>text</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>aws:ec2:image</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>aws:ssm:integration</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Amazon Web
     *         Services Systems Manager validates the parameter value is in the required format, such as
     *         <code>ami-12345abcdeEXAMPLE</code>, and that the specified AMI is available in your Amazon Web Services
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html"
     *         >Native parameter support for Amazon Machine Image (AMI) IDs</a> in the <i>Amazon Web Services Systems
     *         Manager User Guide</i>.
     */
    public final String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(overwrite());
        hashCode = 31 * hashCode + Objects.hashCode(allowedPattern());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutParameterRequest)) {
            return false;
        }
        PutParameterRequest other = (PutParameterRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(value(), other.value()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(keyId(), other.keyId()) && Objects.equals(overwrite(), other.overwrite())
                && Objects.equals(allowedPattern(), other.allowedPattern()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(tierAsString(), other.tierAsString())
                && Objects.equals(policies(), other.policies()) && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutParameterRequest").add("Name", name()).add("Description", description())
                .add("Value", value() == null ? null : "*** Sensitive Data Redacted ***").add("Type", typeAsString())
                .add("KeyId", keyId()).add("Overwrite", overwrite()).add("AllowedPattern", allowedPattern())
                .add("Tags", hasTags() ? tags() : null).add("Tier", tierAsString()).add("Policies", policies())
                .add("DataType", dataType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "Overwrite":
            return Optional.ofNullable(clazz.cast(overwrite()));
        case "AllowedPattern":
            return Optional.ofNullable(clazz.cast(allowedPattern()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutParameterRequest, T> g) {
        return obj -> g.apply((PutParameterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutParameterRequest> {
        /**
         * <p>
         * The fully qualified name of the parameter that you want to add to the system. The fully qualified name
         * includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you must
         * include a leading forward slash character (/) when you create or reference a parameter. For example:
         * <code>/Dev/DBServer/MySQL/db-string13</code>
         * </p>
         * <p>
         * Naming Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Parameter names are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name must be unique within an Amazon Web Services Region
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name can't be prefixed with "<code>aws</code>" or "<code>ssm</code>" (case-insensitive).
         * </p>
         * </li>
         * <li>
         * <p>
         * Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-</code>
         * </p>
         * <p>
         * In addition, the slash character ( / ) is used to delineate hierarchies in parameter names. For example:
         * <code>/Dev/Production/East/Project-ABC/MyParameter</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * A parameter name can't include spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * Parameter hierarchies are limited to a maximum depth of fifteen levels.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For additional information about valid values for parameter names, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-paramstore-su-create.html">Creating
         * Systems Manager parameters</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * <note>
         * <p>
         * The maximum length constraint of 2048 characters listed below includes 1037 characters reserved for internal
         * use by Systems Manager. The maximum length for a parameter name that you create is 1011 characters. This
         * includes the characters in the ARN that precede the name you specify, such as
         * <code>arn:aws:ssm:us-east-2:111122223333:parameter/</code>.
         * </p>
         * </note>
         * 
         * @param name
         *        The fully qualified name of the parameter that you want to add to the system. The fully qualified name
         *        includes the complete hierarchy of the parameter path and name. For parameters in a hierarchy, you
         *        must include a leading forward slash character (/) when you create or reference a parameter. For
         *        example: <code>/Dev/DBServer/MySQL/db-string13</code> </p>
         *        <p>
         *        Naming Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Parameter names are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name must be unique within an Amazon Web Services Region
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name can't be prefixed with "<code>aws</code>" or "<code>ssm</code>" (case-insensitive).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Parameter names can include only the following symbols and letters: <code>a-zA-Z0-9_.-</code>
         *        </p>
         *        <p>
         *        In addition, the slash character ( / ) is used to delineate hierarchies in parameter names. For
         *        example: <code>/Dev/Production/East/Project-ABC/MyParameter</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A parameter name can't include spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Parameter hierarchies are limited to a maximum depth of fifteen levels.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For additional information about valid values for parameter names, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-paramstore-su-create.html"
         *        >Creating Systems Manager parameters</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        The maximum length constraint of 2048 characters listed below includes 1037 characters reserved for
         *        internal use by Systems Manager. The maximum length for a parameter name that you create is 1011
         *        characters. This includes the characters in the ARN that precede the name you specify, such as
         *        <code>arn:aws:ssm:us-east-2:111122223333:parameter/</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Information about the parameter that you want to add to the system. Optional but recommended.
         * </p>
         * <important>
         * <p>
         * Don't enter personally identifiable information in this field.
         * </p>
         * </important>
         * 
         * @param description
         *        Information about the parameter that you want to add to the system. Optional but recommended.</p>
         *        <important>
         *        <p>
         *        Don't enter personally identifiable information in this field.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The parameter value that you want to add to the system. Standard parameters have a value limit of 4 KB.
         * Advanced parameters have a value limit of 8 KB.
         * </p>
         * <note>
         * <p>
         * Parameters can't be referenced or nested in the values of other parameters. You can't include
         * <code>{{}}</code> or <code>{{ssm:<i>parameter-name</i>}}</code> in a parameter value.
         * </p>
         * </note>
         * 
         * @param value
         *        The parameter value that you want to add to the system. Standard parameters have a value limit of 4
         *        KB. Advanced parameters have a value limit of 8 KB.</p> <note>
         *        <p>
         *        Parameters can't be referenced or nested in the values of other parameters. You can't include
         *        <code>{{}}</code> or <code>{{ssm:<i>parameter-name</i>}}</code> in a parameter value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The type of parameter that you want to add to the system.
         * </p>
         * <note>
         * <p>
         * <code>SecureString</code> isn't currently supported for CloudFormation templates.
         * </p>
         * </note>
         * <p>
         * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
         * special character to escape items in the list. If you have a parameter value that requires a comma, then use
         * the <code>String</code> data type.
         * </p>
         * <important>
         * <p>
         * Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type when
         * creating a parameter.
         * </p>
         * </important>
         * 
         * @param type
         *        The type of parameter that you want to add to the system.</p> <note>
         *        <p>
         *        <code>SecureString</code> isn't currently supported for CloudFormation templates.
         *        </p>
         *        </note>
         *        <p>
         *        Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation
         *        or special character to escape items in the list. If you have a parameter value that requires a comma,
         *        then use the <code>String</code> data type.
         *        </p>
         *        <important>
         *        <p>
         *        Specifying a parameter type isn't required when updating a parameter. You must specify a parameter
         *        type when creating a parameter.
         *        </p>
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter that you want to add to the system.
         * </p>
         * <note>
         * <p>
         * <code>SecureString</code> isn't currently supported for CloudFormation templates.
         * </p>
         * </note>
         * <p>
         * Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation or
         * special character to escape items in the list. If you have a parameter value that requires a comma, then use
         * the <code>String</code> data type.
         * </p>
         * <important>
         * <p>
         * Specifying a parameter type isn't required when updating a parameter. You must specify a parameter type when
         * creating a parameter.
         * </p>
         * </important>
         * 
         * @param type
         *        The type of parameter that you want to add to the system.</p> <note>
         *        <p>
         *        <code>SecureString</code> isn't currently supported for CloudFormation templates.
         *        </p>
         *        </note>
         *        <p>
         *        Items in a <code>StringList</code> must be separated by a comma (,). You can't use other punctuation
         *        or special character to escape items in the list. If you have a parameter value that requires a comma,
         *        then use the <code>String</code> data type.
         *        </p>
         *        <important>
         *        <p>
         *        Specifying a parameter type isn't required when updating a parameter. You must specify a parameter
         *        type when creating a parameter.
         *        </p>
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The Key Management Service (KMS) ID that you want to use to encrypt a parameter. Either the default KMS key
         * automatically assigned to your Amazon Web Services account or a custom key. Required for parameters that use
         * the <code>SecureString</code> data type.
         * </p>
         * <p>
         * If you don't specify a key ID, the system uses the default key associated with your Amazon Web Services
         * account.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To use your default KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify the
         * <code>Key ID</code> when you create the parameter. The system automatically populates <code>Key ID</code>
         * with your default KMS key.
         * </p>
         * </li>
         * <li>
         * <p>
         * To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
         * parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyId
         *        The Key Management Service (KMS) ID that you want to use to encrypt a parameter. Either the default
         *        KMS key automatically assigned to your Amazon Web Services account or a custom key. Required for
         *        parameters that use the <code>SecureString</code> data type.</p>
         *        <p>
         *        If you don't specify a key ID, the system uses the default key associated with your Amazon Web
         *        Services account.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To use your default KMS key, choose the <code>SecureString</code> data type, and do <i>not</i> specify
         *        the <code>Key ID</code> when you create the parameter. The system automatically populates
         *        <code>Key ID</code> with your default KMS key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To use a custom KMS key, choose the <code>SecureString</code> data type with the <code>Key ID</code>
         *        parameter.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Overwrite an existing parameter. The default value is <code>false</code>.
         * </p>
         * 
         * @param overwrite
         *        Overwrite an existing parameter. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overwrite(Boolean overwrite);

        /**
         * <p>
         * A regular expression used to validate the parameter value. For example, for String types with values
         * restricted to numbers, you can specify the following: AllowedPattern=^\d+$
         * </p>
         * 
         * @param allowedPattern
         *        A regular expression used to validate the parameter value. For example, for String types with values
         *        restricted to numbers, you can specify the following: AllowedPattern=^\d+$
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedPattern(String allowedPattern);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
         *        parameter to identify the type of resource to which it applies, the environment, or the type of
         *        configuration data referenced by the parameter. In this case, you could specify the following
         *        key-value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=Resource,Value=S3bucket</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=ParameterType,Value=LicenseKey</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager
         *        parameter to identify the type of resource to which it applies, the environment, or the type of
         *        configuration data referenced by the parameter. In this case, you could specify the following
         *        key-value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=Resource,Value=S3bucket</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=ParameterType,Value=LicenseKey</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag a Systems Manager parameter to
         * identify the type of resource to which it applies, the environment, or the type of configuration data
         * referenced by the parameter. In this case, you could specify the following key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=Resource,Value=S3bucket</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=ParameterType,Value=LicenseKey</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing Systems Manager parameter, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the
         * need to create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The parameter tier to assign to a parameter.
         * </p>
         * <p>
         * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         * content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
         * 10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters are offered
         * at no additional cost.
         * </p>
         * <p>
         * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You
         * can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services account.
         * Advanced parameters incur a charge. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         * >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * <p>
         * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
         * parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result in
         * data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also
         * remove any policies attached to the parameter. Lastly, advanced parameters use a different form of encryption
         * than standard parameters.
         * </p>
         * <p>
         * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         * parameter, you must delete it and recreate it as a new standard parameter.
         * </p>
         * <p>
         * <b>Using the Default Tier Configuration</b>
         * </p>
         * <p>
         * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
         * specify a tier in the request, Parameter Store creates or updates the parameter according to that request.
         * However, if you don't specify a tier in a request, Parameter Store assigns the tier based on the current
         * Parameter Store default tier configuration.
         * </p>
         * <p>
         * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         * advanced-parameter tier, you can specify one of the following as the default:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
         * parameter is standard or advanced.
         * </p>
         * <p>
         * If the request doesn't include any options that require an advanced parameter, the parameter is created in
         * the standard-parameter tier. If one or more options requiring an advanced parameter are included in the
         * request, Parameter Store create a parameter in the advanced-parameter tier.
         * </p>
         * <p>
         * This approach helps control your parameter-related costs by always creating standard parameters unless an
         * advanced parameter is necessary.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Options that require an advanced parameter include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The content size of the parameter is more than 4 KB.
         * </p>
         * </li>
         * <li>
         * <p>
         * The parameter uses a parameter policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web
         * Services Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about configuring the default tier option, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
         * parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param tier
         *        The parameter tier to assign to a parameter.</p>
         *        <p>
         *        Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         *        content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum
         *        of 10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters
         *        are offered at no additional cost.
         *        </p>
         *        <p>
         *        Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
         *        You can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services
         *        account. Advanced parameters incur a charge. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         *        >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User
         *        Guide</i>.
         *        </p>
         *        <p>
         *        You can change a standard parameter to an advanced parameter any time. But you can't revert an
         *        advanced parameter to a standard parameter. Reverting an advanced parameter to a standard parameter
         *        would result in data loss because the system would truncate the size of the parameter from 8 KB to 4
         *        KB. Reverting would also remove any policies attached to the parameter. Lastly, advanced parameters
         *        use a different form of encryption than standard parameters.
         *        </p>
         *        <p>
         *        If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         *        parameter, you must delete it and recreate it as a new standard parameter.
         *        </p>
         *        <p>
         *        <b>Using the Default Tier Configuration</b>
         *        </p>
         *        <p>
         *        In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever
         *        you specify a tier in the request, Parameter Store creates or updates the parameter according to that
         *        request. However, if you don't specify a tier in a request, Parameter Store assigns the tier based on
         *        the current Parameter Store default tier configuration.
         *        </p>
         *        <p>
         *        The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         *        advanced-parameter tier, you can specify one of the following as the default:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if
         *        the parameter is standard or advanced.
         *        </p>
         *        <p>
         *        If the request doesn't include any options that require an advanced parameter, the parameter is
         *        created in the standard-parameter tier. If one or more options requiring an advanced parameter are
         *        included in the request, Parameter Store create a parameter in the advanced-parameter tier.
         *        </p>
         *        <p>
         *        This approach helps control your parameter-related costs by always creating standard parameters unless
         *        an advanced parameter is necessary.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Options that require an advanced parameter include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The content size of the parameter is more than 4 KB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The parameter uses a parameter policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon
         *        Web Services Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about configuring the default tier option, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
         *        default parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The parameter tier to assign to a parameter.
         * </p>
         * <p>
         * Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         * content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum of
         * 10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters are offered
         * at no additional cost.
         * </p>
         * <p>
         * Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies. You
         * can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services account.
         * Advanced parameters incur a charge. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         * >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * <p>
         * You can change a standard parameter to an advanced parameter any time. But you can't revert an advanced
         * parameter to a standard parameter. Reverting an advanced parameter to a standard parameter would result in
         * data loss because the system would truncate the size of the parameter from 8 KB to 4 KB. Reverting would also
         * remove any policies attached to the parameter. Lastly, advanced parameters use a different form of encryption
         * than standard parameters.
         * </p>
         * <p>
         * If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         * parameter, you must delete it and recreate it as a new standard parameter.
         * </p>
         * <p>
         * <b>Using the Default Tier Configuration</b>
         * </p>
         * <p>
         * In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever you
         * specify a tier in the request, Parameter Store creates or updates the parameter according to that request.
         * However, if you don't specify a tier in a request, Parameter Store assigns the tier based on the current
         * Parameter Store default tier configuration.
         * </p>
         * <p>
         * The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         * advanced-parameter tier, you can specify one of the following as the default:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if the
         * parameter is standard or advanced.
         * </p>
         * <p>
         * If the request doesn't include any options that require an advanced parameter, the parameter is created in
         * the standard-parameter tier. If one or more options requiring an advanced parameter are included in the
         * request, Parameter Store create a parameter in the advanced-parameter tier.
         * </p>
         * <p>
         * This approach helps control your parameter-related costs by always creating standard parameters unless an
         * advanced parameter is necessary.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Options that require an advanced parameter include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The content size of the parameter is more than 4 KB.
         * </p>
         * </li>
         * <li>
         * <p>
         * The parameter uses a parameter policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon Web
         * Services Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about configuring the default tier option, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a default
         * parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param tier
         *        The parameter tier to assign to a parameter.</p>
         *        <p>
         *        Parameter Store offers a standard tier and an advanced tier for parameters. Standard parameters have a
         *        content size limit of 4 KB and can't be configured to use parameter policies. You can create a maximum
         *        of 10,000 standard parameters for each Region in an Amazon Web Services account. Standard parameters
         *        are offered at no additional cost.
         *        </p>
         *        <p>
         *        Advanced parameters have a content size limit of 8 KB and can be configured to use parameter policies.
         *        You can create a maximum of 100,000 advanced parameters for each Region in an Amazon Web Services
         *        account. Advanced parameters incur a charge. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-advanced-parameters.html"
         *        >Standard and advanced parameter tiers</a> in the <i>Amazon Web Services Systems Manager User
         *        Guide</i>.
         *        </p>
         *        <p>
         *        You can change a standard parameter to an advanced parameter any time. But you can't revert an
         *        advanced parameter to a standard parameter. Reverting an advanced parameter to a standard parameter
         *        would result in data loss because the system would truncate the size of the parameter from 8 KB to 4
         *        KB. Reverting would also remove any policies attached to the parameter. Lastly, advanced parameters
         *        use a different form of encryption than standard parameters.
         *        </p>
         *        <p>
         *        If you no longer need an advanced parameter, or if you no longer want to incur charges for an advanced
         *        parameter, you must delete it and recreate it as a new standard parameter.
         *        </p>
         *        <p>
         *        <b>Using the Default Tier Configuration</b>
         *        </p>
         *        <p>
         *        In <code>PutParameter</code> requests, you can specify the tier to create the parameter in. Whenever
         *        you specify a tier in the request, Parameter Store creates or updates the parameter according to that
         *        request. However, if you don't specify a tier in a request, Parameter Store assigns the tier based on
         *        the current Parameter Store default tier configuration.
         *        </p>
         *        <p>
         *        The default tier when you begin using Parameter Store is the standard-parameter tier. If you use the
         *        advanced-parameter tier, you can specify one of the following as the default:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Advanced</b>: With this option, Parameter Store evaluates all requests as advanced parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Intelligent-Tiering</b>: With this option, Parameter Store evaluates each request to determine if
         *        the parameter is standard or advanced.
         *        </p>
         *        <p>
         *        If the request doesn't include any options that require an advanced parameter, the parameter is
         *        created in the standard-parameter tier. If one or more options requiring an advanced parameter are
         *        included in the request, Parameter Store create a parameter in the advanced-parameter tier.
         *        </p>
         *        <p>
         *        This approach helps control your parameter-related costs by always creating standard parameters unless
         *        an advanced parameter is necessary.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Options that require an advanced parameter include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The content size of the parameter is more than 4 KB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The parameter uses a parameter policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        More than 10,000 parameters already exist in your Amazon Web Services account in the current Amazon
         *        Web Services Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about configuring the default tier option, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-default-tier.html">Specifying a
         *        default parameter tier</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(ParameterTier tier);

        /**
         * <p>
         * One or more policies to apply to a parameter. This operation takes a JSON array. Parameter Store, a
         * capability of Amazon Web Services Systems Manager supports the following policy types:
         * </p>
         * <p>
         * Expiration: This policy deletes the parameter after it expires. When you create the policy, you specify the
         * expiration date. You can update the expiration date and time by updating the policy. Updating the
         * <i>parameter</i> doesn't affect the expiration date and time. When the expiration time is reached, Parameter
         * Store deletes the parameter.
         * </p>
         * <p>
         * ExpirationNotification: This policy initiates an event in Amazon CloudWatch Events that notifies you about
         * the expiration. By using this policy, you can receive notification before or after the expiration time is
         * reached, in units of days or hours.
         * </p>
         * <p>
         * NoChangeNotification: This policy initiates a CloudWatch Events event if a parameter hasn't been modified for
         * a specified period of time. This policy type is useful when, for example, a secret needs to be changed within
         * a period of time, but it hasn't been changed.
         * </p>
         * <p>
         * All existing policies are preserved until you send new policies or an empty policy. For more information
         * about parameter policies, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
         * parameter policies</a>.
         * </p>
         * 
         * @param policies
         *        One or more policies to apply to a parameter. This operation takes a JSON array. Parameter Store, a
         *        capability of Amazon Web Services Systems Manager supports the following policy types:</p>
         *        <p>
         *        Expiration: This policy deletes the parameter after it expires. When you create the policy, you
         *        specify the expiration date. You can update the expiration date and time by updating the policy.
         *        Updating the <i>parameter</i> doesn't affect the expiration date and time. When the expiration time is
         *        reached, Parameter Store deletes the parameter.
         *        </p>
         *        <p>
         *        ExpirationNotification: This policy initiates an event in Amazon CloudWatch Events that notifies you
         *        about the expiration. By using this policy, you can receive notification before or after the
         *        expiration time is reached, in units of days or hours.
         *        </p>
         *        <p>
         *        NoChangeNotification: This policy initiates a CloudWatch Events event if a parameter hasn't been
         *        modified for a specified period of time. This policy type is useful when, for example, a secret needs
         *        to be changed within a period of time, but it hasn't been changed.
         *        </p>
         *        <p>
         *        All existing policies are preserved until you send new policies or an empty policy. For more
         *        information about parameter policies, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html"
         *        >Assigning parameter policies</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(String policies);

        /**
         * <p>
         * The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon Machine
         * Image (AMI) IDs.
         * </p>
         * <p>
         * <b>The following data type values are supported.</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>text</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws:ec2:image</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>aws:ssm:integration</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Amazon Web Services
         * Systems Manager validates the parameter value is in the required format, such as
         * <code>ami-12345abcdeEXAMPLE</code>, and that the specified AMI is available in your Amazon Web Services
         * account. For more information, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html">Native
         * parameter support for Amazon Machine Image (AMI) IDs</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * 
         * @param dataType
         *        The data type for a <code>String</code> parameter. Supported data types include plain text and Amazon
         *        Machine Image (AMI) IDs.</p>
         *        <p>
         *        <b>The following data type values are supported.</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>text</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws:ec2:image</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>aws:ssm:integration</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you create a <code>String</code> parameter and specify <code>aws:ec2:image</code>, Amazon Web
         *        Services Systems Manager validates the parameter value is in the required format, such as
         *        <code>ami-12345abcdeEXAMPLE</code>, and that the specified AMI is available in your Amazon Web
         *        Services account. For more information, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-ec2-aliases.html"
         *        >Native parameter support for Amazon Machine Image (AMI) IDs</a> in the <i>Amazon Web Services Systems
         *        Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private String value;

        private String type;

        private String keyId;

        private Boolean overwrite;

        private String allowedPattern;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String tier;

        private String policies;

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(PutParameterRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            value(model.value);
            type(model.type);
            keyId(model.keyId);
            overwrite(model.overwrite);
            allowedPattern(model.allowedPattern);
            tags(model.tags);
            tier(model.tier);
            policies(model.policies);
            dataType(model.dataType);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        @Transient
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getKeyId() {
            return keyId;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        @Override
        @Transient
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final Boolean getOverwrite() {
            return overwrite;
        }

        public final void setOverwrite(Boolean overwrite) {
            this.overwrite = overwrite;
        }

        @Override
        @Transient
        public final Builder overwrite(Boolean overwrite) {
            this.overwrite = overwrite;
            return this;
        }

        public final String getAllowedPattern() {
            return allowedPattern;
        }

        public final void setAllowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
        }

        @Override
        @Transient
        public final Builder allowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        @Transient
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        @Transient
        public final Builder tier(ParameterTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final String getPolicies() {
            return policies;
        }

        public final void setPolicies(String policies) {
            this.policies = policies;
        }

        @Override
        @Transient
        public final Builder policies(String policies) {
            this.policies = policies;
            return this;
        }

        public final String getDataType() {
            return dataType;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        @Transient
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutParameterRequest build() {
            return new PutParameterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
