/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an Automation runbook used in a runbook workflow in Change Manager.
 * </p>
 * <note>
 * <p>
 * The Automation runbooks specified for the runbook workflow can't run until all required approvals for the change
 * request have been received.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Runbook implements SdkPojo, Serializable, ToCopyableBuilder<Runbook.Builder, Runbook> {
    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentName").getter(getter(Runbook::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(Runbook::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(Runbook::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> TARGET_PARAMETER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetParameterName").getter(getter(Runbook::targetParameterName))
            .setter(setter(Builder::targetParameterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetParameterName").build())
            .build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(Runbook::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxConcurrency").getter(getter(Runbook::maxConcurrency)).setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxErrors").getter(getter(Runbook::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<List<TargetLocation>> TARGET_LOCATIONS_FIELD = SdkField
            .<List<TargetLocation>> builder(MarshallingType.LIST)
            .memberName("TargetLocations")
            .getter(getter(Runbook::targetLocations))
            .setter(setter(Builder::targetLocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_NAME_FIELD,
            DOCUMENT_VERSION_FIELD, PARAMETERS_FIELD, TARGET_PARAMETER_NAME_FIELD, TARGETS_FIELD, MAX_CONCURRENCY_FIELD,
            MAX_ERRORS_FIELD, TARGET_LOCATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String documentName;

    private final String documentVersion;

    private final Map<String, List<String>> parameters;

    private final String targetParameterName;

    private final List<Target> targets;

    private final String maxConcurrency;

    private final String maxErrors;

    private final List<TargetLocation> targetLocations;

    private Runbook(BuilderImpl builder) {
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.parameters = builder.parameters;
        this.targetParameterName = builder.targetParameterName;
        this.targets = builder.targets;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.targetLocations = builder.targetLocations;
    }

    /**
     * <p>
     * The name of the Automation runbook used in a runbook workflow.
     * </p>
     * 
     * @return The name of the Automation runbook used in a runbook workflow.
     */
    public final String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The version of the Automation runbook used in a runbook workflow.
     * </p>
     * 
     * @return The version of the Automation runbook used in a runbook workflow.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value map of execution parameters, which were supplied when calling
     * <code>StartChangeRequestExecution</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The key-value map of execution parameters, which were supplied when calling
     *         <code>StartChangeRequestExecution</code>.
     */
    public final Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The name of the parameter used as the target resource for the rate-controlled runbook workflow. Required if you
     * specify <code>Targets</code>.
     * </p>
     * 
     * @return The name of the parameter used as the target resource for the rate-controlled runbook workflow. Required
     *         if you specify <code>Targets</code>.
     */
    public final String targetParameterName() {
        return targetParameterName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A key-value mapping to target resources that the runbook operation performs tasks on. Required if you specify
     * <code>TargetParameterName</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return A key-value mapping to target resources that the runbook operation performs tasks on. Required if you
     *         specify <code>TargetParameterName</code>.
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The <code>MaxConcurrency</code> value specified by the user when the operation started, indicating the maximum
     * number of resources that the runbook operation can run on at the same time.
     * </p>
     * 
     * @return The <code>MaxConcurrency</code> value specified by the user when the operation started, indicating the
     *         maximum number of resources that the runbook operation can run on at the same time.
     */
    public final String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The <code>MaxErrors</code> value specified by the user when the execution started, indicating the maximum number
     * of errors that can occur during the operation before the updates are stopped or rolled back.
     * </p>
     * 
     * @return The <code>MaxErrors</code> value specified by the user when the execution started, indicating the maximum
     *         number of errors that can occur during the operation before the updates are stopped or rolled back.
     */
    public final String maxErrors() {
        return maxErrors;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetLocations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetLocations() {
        return targetLocations != null && !(targetLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the current
     * Runbook operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetLocations} method.
     * </p>
     * 
     * @return Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the
     *         current Runbook operation.
     */
    public final List<TargetLocation> targetLocations() {
        return targetLocations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetParameterName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetLocations() ? targetLocations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Runbook)) {
            return false;
        }
        Runbook other = (Runbook) obj;
        return Objects.equals(documentName(), other.documentName()) && Objects.equals(documentVersion(), other.documentVersion())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(targetParameterName(), other.targetParameterName()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(maxConcurrency(), other.maxConcurrency())
                && Objects.equals(maxErrors(), other.maxErrors()) && hasTargetLocations() == other.hasTargetLocations()
                && Objects.equals(targetLocations(), other.targetLocations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Runbook").add("DocumentName", documentName()).add("DocumentVersion", documentVersion())
                .add("Parameters", hasParameters() ? parameters() : null).add("TargetParameterName", targetParameterName())
                .add("Targets", hasTargets() ? targets() : null).add("MaxConcurrency", maxConcurrency())
                .add("MaxErrors", maxErrors()).add("TargetLocations", hasTargetLocations() ? targetLocations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "TargetParameterName":
            return Optional.ofNullable(clazz.cast(targetParameterName()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "TargetLocations":
            return Optional.ofNullable(clazz.cast(targetLocations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Runbook, T> g) {
        return obj -> g.apply((Runbook) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Runbook> {
        /**
         * <p>
         * The name of the Automation runbook used in a runbook workflow.
         * </p>
         * 
         * @param documentName
         *        The name of the Automation runbook used in a runbook workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The version of the Automation runbook used in a runbook workflow.
         * </p>
         * 
         * @param documentVersion
         *        The version of the Automation runbook used in a runbook workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The key-value map of execution parameters, which were supplied when calling
         * <code>StartChangeRequestExecution</code>.
         * </p>
         * 
         * @param parameters
         *        The key-value map of execution parameters, which were supplied when calling
         *        <code>StartChangeRequestExecution</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The name of the parameter used as the target resource for the rate-controlled runbook workflow. Required if
         * you specify <code>Targets</code>.
         * </p>
         * 
         * @param targetParameterName
         *        The name of the parameter used as the target resource for the rate-controlled runbook workflow.
         *        Required if you specify <code>Targets</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetParameterName(String targetParameterName);

        /**
         * <p>
         * A key-value mapping to target resources that the runbook operation performs tasks on. Required if you specify
         * <code>TargetParameterName</code>.
         * </p>
         * 
         * @param targets
         *        A key-value mapping to target resources that the runbook operation performs tasks on. Required if you
         *        specify <code>TargetParameterName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * A key-value mapping to target resources that the runbook operation performs tasks on. Required if you specify
         * <code>TargetParameterName</code>.
         * </p>
         * 
         * @param targets
         *        A key-value mapping to target resources that the runbook operation performs tasks on. Required if you
         *        specify <code>TargetParameterName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * A key-value mapping to target resources that the runbook operation performs tasks on. Required if you specify
         * <code>TargetParameterName</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Target>.Builder} avoiding the need
         * to create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The <code>MaxConcurrency</code> value specified by the user when the operation started, indicating the
         * maximum number of resources that the runbook operation can run on at the same time.
         * </p>
         * 
         * @param maxConcurrency
         *        The <code>MaxConcurrency</code> value specified by the user when the operation started, indicating the
         *        maximum number of resources that the runbook operation can run on at the same time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The <code>MaxErrors</code> value specified by the user when the execution started, indicating the maximum
         * number of errors that can occur during the operation before the updates are stopped or rolled back.
         * </p>
         * 
         * @param maxErrors
         *        The <code>MaxErrors</code> value specified by the user when the execution started, indicating the
         *        maximum number of errors that can occur during the operation before the updates are stopped or rolled
         *        back.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the current
         * Runbook operation.
         * </p>
         * 
         * @param targetLocations
         *        Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the
         *        current Runbook operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocations(Collection<TargetLocation> targetLocations);

        /**
         * <p>
         * Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the current
         * Runbook operation.
         * </p>
         * 
         * @param targetLocations
         *        Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the
         *        current Runbook operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocations(TargetLocation... targetLocations);

        /**
         * <p>
         * Information about the Amazon Web Services Regions and Amazon Web Services accounts targeted by the current
         * Runbook operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<TargetLocation>.Builder} avoiding
         * the need to create one manually via {@link List<TargetLocation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TargetLocation>.Builder#build()} is called immediately and
         * its result is passed to {@link #targetLocations(List<TargetLocation>)}.
         * 
         * @param targetLocations
         *        a consumer that will call methods on {@link List<TargetLocation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetLocations(List<TargetLocation>)
         */
        Builder targetLocations(Consumer<TargetLocation.Builder>... targetLocations);
    }

    static final class BuilderImpl implements Builder {
        private String documentName;

        private String documentVersion;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String targetParameterName;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String maxConcurrency;

        private String maxErrors;

        private List<TargetLocation> targetLocations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Runbook model) {
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            parameters(model.parameters);
            targetParameterName(model.targetParameterName);
            targets(model.targets);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            targetLocations(model.targetLocations);
        }

        public final String getDocumentName() {
            return documentName;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        @Override
        @Transient
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        @Transient
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = AutomationParameterMapCopier.copy(parameters);
        }

        @Override
        @Transient
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = AutomationParameterMapCopier.copy(parameters);
            return this;
        }

        public final String getTargetParameterName() {
            return targetParameterName;
        }

        public final void setTargetParameterName(String targetParameterName) {
            this.targetParameterName = targetParameterName;
        }

        @Override
        @Transient
        public final Builder targetParameterName(String targetParameterName) {
            this.targetParameterName = targetParameterName;
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        @Transient
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        @Override
        @Transient
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        @Override
        @Transient
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final List<TargetLocation.Builder> getTargetLocations() {
            List<TargetLocation.Builder> result = TargetLocationsCopier.copyToBuilder(this.targetLocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargetLocations(Collection<TargetLocation.BuilderImpl> targetLocations) {
            this.targetLocations = TargetLocationsCopier.copyFromBuilder(targetLocations);
        }

        @Override
        @Transient
        public final Builder targetLocations(Collection<TargetLocation> targetLocations) {
            this.targetLocations = TargetLocationsCopier.copy(targetLocations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targetLocations(TargetLocation... targetLocations) {
            targetLocations(Arrays.asList(targetLocations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder targetLocations(Consumer<TargetLocation.Builder>... targetLocations) {
            targetLocations(Stream.of(targetLocations).map(c -> TargetLocation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Runbook build() {
            return new Runbook(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
