/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegisterTargetWithMaintenanceWindowRequest extends SsmRequest implements
        ToCopyableBuilder<RegisterTargetWithMaintenanceWindowRequest.Builder, RegisterTargetWithMaintenanceWindowRequest> {
    private static final SdkField<String> WINDOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WindowId").getter(getter(RegisterTargetWithMaintenanceWindowRequest::windowId))
            .setter(setter(Builder::windowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowId").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(RegisterTargetWithMaintenanceWindowRequest::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(RegisterTargetWithMaintenanceWindowRequest::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OWNER_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerInformation").getter(getter(RegisterTargetWithMaintenanceWindowRequest::ownerInformation))
            .setter(setter(Builder::ownerInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerInformation").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RegisterTargetWithMaintenanceWindowRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(RegisterTargetWithMaintenanceWindowRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(RegisterTargetWithMaintenanceWindowRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WINDOW_ID_FIELD,
            RESOURCE_TYPE_FIELD, TARGETS_FIELD, OWNER_INFORMATION_FIELD, NAME_FIELD, DESCRIPTION_FIELD, CLIENT_TOKEN_FIELD));

    private final String windowId;

    private final String resourceType;

    private final List<Target> targets;

    private final String ownerInformation;

    private final String name;

    private final String description;

    private final String clientToken;

    private RegisterTargetWithMaintenanceWindowRequest(BuilderImpl builder) {
        super(builder);
        this.windowId = builder.windowId;
        this.resourceType = builder.resourceType;
        this.targets = builder.targets;
        this.ownerInformation = builder.ownerInformation;
        this.name = builder.name;
        this.description = builder.description;
        this.clientToken = builder.clientToken;
    }

    /**
     * <p>
     * The ID of the maintenance window the target should be registered with.
     * </p>
     * 
     * @return The ID of the maintenance window the target should be registered with.
     */
    public final String windowId() {
        return windowId;
    }

    /**
     * <p>
     * The type of target being registered with the maintenance window.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link MaintenanceWindowResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of target being registered with the maintenance window.
     * @see MaintenanceWindowResourceType
     */
    public final MaintenanceWindowResourceType resourceType() {
        return MaintenanceWindowResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of target being registered with the maintenance window.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link MaintenanceWindowResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of target being registered with the maintenance window.
     * @see MaintenanceWindowResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets to register with the maintenance window. In other words, the managed nodes to run commands on when
     * the maintenance window runs.
     * </p>
     * <note>
     * <p>
     * If a single maintenance window task is registered with multiple targets, its task invocations occur sequentially
     * and not in parallel. If your task must run on multiple targets at the same time, register a task for each target
     * individually and assign each task the same priority level.
     * </p>
     * </note>
     * <p>
     * You can specify targets using managed node IDs, resource group names, or tags that have been applied to managed
     * nodes.
     * </p>
     * <p>
     * <b>Example 1</b>: Specify managed node IDs
     * </p>
     * <p>
     * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
     * </p>
     * <p>
     * <b>Example 2</b>: Use tag key-pairs applied to managed nodes
     * </p>
     * <p>
     * <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
     * </p>
     * <p>
     * <b>Example 3</b>: Use tag-keys applied to managed nodes
     * </p>
     * <p>
     * <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
     * </p>
     * <p>
     * <b>Example 4</b>: Use resource group names
     * </p>
     * <p>
     * <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
     * </p>
     * <p>
     * <b>Example 5</b>: Use filters for resource group types
     * </p>
     * <p>
     * <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
     * </p>
     * <note>
     * <p>
     * For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following format
     * </p>
     * <p>
     * <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
     * </p>
     * </note>
     * <p>
     * For more information about these examples formats, including the best use case for each one, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
     * >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems Manager User
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return The targets to register with the maintenance window. In other words, the managed nodes to run commands on
     *         when the maintenance window runs.</p> <note>
     *         <p>
     *         If a single maintenance window task is registered with multiple targets, its task invocations occur
     *         sequentially and not in parallel. If your task must run on multiple targets at the same time, register a
     *         task for each target individually and assign each task the same priority level.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify targets using managed node IDs, resource group names, or tags that have been applied to
     *         managed nodes.
     *         </p>
     *         <p>
     *         <b>Example 1</b>: Specify managed node IDs
     *         </p>
     *         <p>
     *         <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
     *         </p>
     *         <p>
     *         <b>Example 2</b>: Use tag key-pairs applied to managed nodes
     *         </p>
     *         <p>
     *         <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
     *         </p>
     *         <p>
     *         <b>Example 3</b>: Use tag-keys applied to managed nodes
     *         </p>
     *         <p>
     *         <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
     *         </p>
     *         <p>
     *         <b>Example 4</b>: Use resource group names
     *         </p>
     *         <p>
     *         <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
     *         </p>
     *         <p>
     *         <b>Example 5</b>: Use filters for resource group types
     *         </p>
     *         <p>
     *         <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
     *         </p>
     *         <note>
     *         <p>
     *         For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following format
     *         </p>
     *         <p>
     *         <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
     *         </p>
     *         </note>
     *         <p>
     *         For more information about these examples formats, including the best use case for each one, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
     *         >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems Manager
     *         User Guide</i>.
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * User-provided value that will be included in any Amazon CloudWatch Events events raised while running tasks for
     * these targets in this maintenance window.
     * </p>
     * 
     * @return User-provided value that will be included in any Amazon CloudWatch Events events raised while running
     *         tasks for these targets in this maintenance window.
     */
    public final String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * An optional name for the target.
     * </p>
     * 
     * @return An optional name for the target.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * An optional description for the target.
     * </p>
     * 
     * @return An optional description for the target.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * User-provided idempotency token.
     * </p>
     * 
     * @return User-provided idempotency token.
     */
    public final String clientToken() {
        return clientToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(windowId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegisterTargetWithMaintenanceWindowRequest)) {
            return false;
        }
        RegisterTargetWithMaintenanceWindowRequest other = (RegisterTargetWithMaintenanceWindowRequest) obj;
        return Objects.equals(windowId(), other.windowId())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(clientToken(), other.clientToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegisterTargetWithMaintenanceWindowRequest").add("WindowId", windowId())
                .add("ResourceType", resourceTypeAsString()).add("Targets", hasTargets() ? targets() : null)
                .add("OwnerInformation", ownerInformation() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Name", name()).add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientToken", clientToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowId":
            return Optional.ofNullable(clazz.cast(windowId()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "OwnerInformation":
            return Optional.ofNullable(clazz.cast(ownerInformation()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RegisterTargetWithMaintenanceWindowRequest, T> g) {
        return obj -> g.apply((RegisterTargetWithMaintenanceWindowRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, RegisterTargetWithMaintenanceWindowRequest> {
        /**
         * <p>
         * The ID of the maintenance window the target should be registered with.
         * </p>
         * 
         * @param windowId
         *        The ID of the maintenance window the target should be registered with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowId(String windowId);

        /**
         * <p>
         * The type of target being registered with the maintenance window.
         * </p>
         * 
         * @param resourceType
         *        The type of target being registered with the maintenance window.
         * @see MaintenanceWindowResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of target being registered with the maintenance window.
         * </p>
         * 
         * @param resourceType
         *        The type of target being registered with the maintenance window.
         * @see MaintenanceWindowResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowResourceType
         */
        Builder resourceType(MaintenanceWindowResourceType resourceType);

        /**
         * <p>
         * The targets to register with the maintenance window. In other words, the managed nodes to run commands on
         * when the maintenance window runs.
         * </p>
         * <note>
         * <p>
         * If a single maintenance window task is registered with multiple targets, its task invocations occur
         * sequentially and not in parallel. If your task must run on multiple targets at the same time, register a task
         * for each target individually and assign each task the same priority level.
         * </p>
         * </note>
         * <p>
         * You can specify targets using managed node IDs, resource group names, or tags that have been applied to
         * managed nodes.
         * </p>
         * <p>
         * <b>Example 1</b>: Specify managed node IDs
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
         * </p>
         * <p>
         * <b>Example 2</b>: Use tag key-pairs applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 3</b>: Use tag-keys applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 4</b>: Use resource group names
         * </p>
         * <p>
         * <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
         * </p>
         * <p>
         * <b>Example 5</b>: Use filters for resource group types
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
         * </p>
         * <note>
         * <p>
         * For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following format
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
         * </p>
         * </note>
         * <p>
         * For more information about these examples formats, including the best use case for each one, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
         * >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * 
         * @param targets
         *        The targets to register with the maintenance window. In other words, the managed nodes to run commands
         *        on when the maintenance window runs.</p> <note>
         *        <p>
         *        If a single maintenance window task is registered with multiple targets, its task invocations occur
         *        sequentially and not in parallel. If your task must run on multiple targets at the same time, register
         *        a task for each target individually and assign each task the same priority level.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify targets using managed node IDs, resource group names, or tags that have been applied
         *        to managed nodes.
         *        </p>
         *        <p>
         *        <b>Example 1</b>: Specify managed node IDs
         *        </p>
         *        <p>
         *        <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 2</b>: Use tag key-pairs applied to managed nodes
         *        </p>
         *        <p>
         *        <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 3</b>: Use tag-keys applied to managed nodes
         *        </p>
         *        <p>
         *        <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 4</b>: Use resource group names
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 5</b>: Use filters for resource group types
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
         *        </p>
         *        <note>
         *        <p>
         *        For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following
         *        format
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
         *        </p>
         *        </note>
         *        <p>
         *        For more information about these examples formats, including the best use case for each one, see <a
         *        href
         *        ="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
         *        >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems
         *        Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets to register with the maintenance window. In other words, the managed nodes to run commands on
         * when the maintenance window runs.
         * </p>
         * <note>
         * <p>
         * If a single maintenance window task is registered with multiple targets, its task invocations occur
         * sequentially and not in parallel. If your task must run on multiple targets at the same time, register a task
         * for each target individually and assign each task the same priority level.
         * </p>
         * </note>
         * <p>
         * You can specify targets using managed node IDs, resource group names, or tags that have been applied to
         * managed nodes.
         * </p>
         * <p>
         * <b>Example 1</b>: Specify managed node IDs
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
         * </p>
         * <p>
         * <b>Example 2</b>: Use tag key-pairs applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 3</b>: Use tag-keys applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 4</b>: Use resource group names
         * </p>
         * <p>
         * <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
         * </p>
         * <p>
         * <b>Example 5</b>: Use filters for resource group types
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
         * </p>
         * <note>
         * <p>
         * For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following format
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
         * </p>
         * </note>
         * <p>
         * For more information about these examples formats, including the best use case for each one, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
         * >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * 
         * @param targets
         *        The targets to register with the maintenance window. In other words, the managed nodes to run commands
         *        on when the maintenance window runs.</p> <note>
         *        <p>
         *        If a single maintenance window task is registered with multiple targets, its task invocations occur
         *        sequentially and not in parallel. If your task must run on multiple targets at the same time, register
         *        a task for each target individually and assign each task the same priority level.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify targets using managed node IDs, resource group names, or tags that have been applied
         *        to managed nodes.
         *        </p>
         *        <p>
         *        <b>Example 1</b>: Specify managed node IDs
         *        </p>
         *        <p>
         *        <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 2</b>: Use tag key-pairs applied to managed nodes
         *        </p>
         *        <p>
         *        <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 3</b>: Use tag-keys applied to managed nodes
         *        </p>
         *        <p>
         *        <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 4</b>: Use resource group names
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
         *        </p>
         *        <p>
         *        <b>Example 5</b>: Use filters for resource group types
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
         *        </p>
         *        <note>
         *        <p>
         *        For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following
         *        format
         *        </p>
         *        <p>
         *        <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
         *        </p>
         *        </note>
         *        <p>
         *        For more information about these examples formats, including the best use case for each one, see <a
         *        href
         *        ="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
         *        >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems
         *        Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets to register with the maintenance window. In other words, the managed nodes to run commands on
         * when the maintenance window runs.
         * </p>
         * <note>
         * <p>
         * If a single maintenance window task is registered with multiple targets, its task invocations occur
         * sequentially and not in parallel. If your task must run on multiple targets at the same time, register a task
         * for each target individually and assign each task the same priority level.
         * </p>
         * </note>
         * <p>
         * You can specify targets using managed node IDs, resource group names, or tags that have been applied to
         * managed nodes.
         * </p>
         * <p>
         * <b>Example 1</b>: Specify managed node IDs
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;,&lt;instance-id-3&gt;</code>
         * </p>
         * <p>
         * <b>Example 2</b>: Use tag key-pairs applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag:&lt;my-tag-key&gt;,Values=&lt;my-tag-value-1&gt;,&lt;my-tag-value-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 3</b>: Use tag-keys applied to managed nodes
         * </p>
         * <p>
         * <code>Key=tag-key,Values=&lt;my-tag-key-1&gt;,&lt;my-tag-key-2&gt;</code>
         * </p>
         * <p>
         * <b>Example 4</b>: Use resource group names
         * </p>
         * <p>
         * <code>Key=resource-groups:Name,Values=&lt;resource-group-name&gt;</code>
         * </p>
         * <p>
         * <b>Example 5</b>: Use filters for resource group types
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=&lt;resource-type-1&gt;,&lt;resource-type-2&gt;</code>
         * </p>
         * <note>
         * <p>
         * For <code>Key=resource-groups:ResourceTypeFilters</code>, specify resource types in the following format
         * </p>
         * <p>
         * <code>Key=resource-groups:ResourceTypeFilters,Values=AWS::EC2::INSTANCE,AWS::EC2::VPC</code>
         * </p>
         * </note>
         * <p>
         * For more information about these examples formats, including the best use case for each one, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-tutorial-targets-examples.html"
         * >Examples: Register targets with a maintenance window</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Target>.Builder} avoiding the need
         * to create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * User-provided value that will be included in any Amazon CloudWatch Events events raised while running tasks
         * for these targets in this maintenance window.
         * </p>
         * 
         * @param ownerInformation
         *        User-provided value that will be included in any Amazon CloudWatch Events events raised while running
         *        tasks for these targets in this maintenance window.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * An optional name for the target.
         * </p>
         * 
         * @param name
         *        An optional name for the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An optional description for the target.
         * </p>
         * 
         * @param description
         *        An optional description for the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * User-provided idempotency token.
         * </p>
         * 
         * @param clientToken
         *        User-provided idempotency token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String windowId;

        private String resourceType;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String ownerInformation;

        private String name;

        private String description;

        private String clientToken;

        private BuilderImpl() {
        }

        private BuilderImpl(RegisterTargetWithMaintenanceWindowRequest model) {
            super(model);
            windowId(model.windowId);
            resourceType(model.resourceType);
            targets(model.targets);
            ownerInformation(model.ownerInformation);
            name(model.name);
            description(model.description);
            clientToken(model.clientToken);
        }

        public final String getWindowId() {
            return windowId;
        }

        public final void setWindowId(String windowId) {
            this.windowId = windowId;
        }

        @Override
        public final Builder windowId(String windowId) {
            this.windowId = windowId;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(MaintenanceWindowResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RegisterTargetWithMaintenanceWindowRequest build() {
            return new RegisterTargetWithMaintenanceWindowRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
