/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the state of a patch on a particular managed node as it relates to the patch baseline used to patch
 * the node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchComplianceData implements SdkPojo, Serializable,
        ToCopyableBuilder<PatchComplianceData.Builder, PatchComplianceData> {
    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(PatchComplianceData::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> KB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KBId")
            .getter(getter(PatchComplianceData::kbId)).setter(setter(Builder::kbId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KBId").build()).build();

    private static final SdkField<String> CLASSIFICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Classification").getter(getter(PatchComplianceData::classification))
            .setter(setter(Builder::classification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Classification").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(PatchComplianceData::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(PatchComplianceData::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> INSTALLED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("InstalledTime").getter(getter(PatchComplianceData::installedTime))
            .setter(setter(Builder::installedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledTime").build()).build();

    private static final SdkField<String> CVE_IDS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("CVEIds")
            .getter(getter(PatchComplianceData::cveIds)).setter(setter(Builder::cveIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CVEIds").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TITLE_FIELD, KB_ID_FIELD,
            CLASSIFICATION_FIELD, SEVERITY_FIELD, STATE_FIELD, INSTALLED_TIME_FIELD, CVE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String title;

    private final String kbId;

    private final String classification;

    private final String severity;

    private final String state;

    private final Instant installedTime;

    private final String cveIds;

    private PatchComplianceData(BuilderImpl builder) {
        this.title = builder.title;
        this.kbId = builder.kbId;
        this.classification = builder.classification;
        this.severity = builder.severity;
        this.state = builder.state;
        this.installedTime = builder.installedTime;
        this.cveIds = builder.cveIds;
    }

    /**
     * <p>
     * The title of the patch.
     * </p>
     * 
     * @return The title of the patch.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The operating system-specific ID of the patch.
     * </p>
     * 
     * @return The operating system-specific ID of the patch.
     */
    public final String kbId() {
        return kbId;
    }

    /**
     * <p>
     * The classification of the patch, such as <code>SecurityUpdates</code>, <code>Updates</code>, and
     * <code>CriticalUpdates</code>.
     * </p>
     * 
     * @return The classification of the patch, such as <code>SecurityUpdates</code>, <code>Updates</code>, and
     *         <code>CriticalUpdates</code>.
     */
    public final String classification() {
        return classification;
    }

    /**
     * <p>
     * The severity of the patch such as <code>Critical</code>, <code>Important</code>, and <code>Moderate</code>.
     * </p>
     * 
     * @return The severity of the patch such as <code>Critical</code>, <code>Important</code>, and
     *         <code>Moderate</code>.
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * The state of the patch on the managed node, such as INSTALLED or FAILED.
     * </p>
     * <p>
     * For descriptions of each patch state, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
     * >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link PatchComplianceDataState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the patch on the managed node, such as INSTALLED or FAILED.</p>
     *         <p>
     *         For descriptions of each patch state, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
     *         >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see PatchComplianceDataState
     */
    public final PatchComplianceDataState state() {
        return PatchComplianceDataState.fromValue(state);
    }

    /**
     * <p>
     * The state of the patch on the managed node, such as INSTALLED or FAILED.
     * </p>
     * <p>
     * For descriptions of each patch state, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
     * >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link PatchComplianceDataState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the patch on the managed node, such as INSTALLED or FAILED.</p>
     *         <p>
     *         For descriptions of each patch state, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
     *         >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see PatchComplianceDataState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The date/time the patch was installed on the managed node. Not all operating systems provide this level of
     * information.
     * </p>
     * 
     * @return The date/time the patch was installed on the managed node. Not all operating systems provide this level
     *         of information.
     */
    public final Instant installedTime() {
        return installedTime;
    }

    /**
     * <p>
     * The IDs of one or more Common Vulnerabilities and Exposure (CVE) issues that are resolved by the patch.
     * </p>
     * <note>
     * <p>
     * Currently, CVE ID values are reported only for patches with a status of <code>Missing</code> or
     * <code>Failed</code>.
     * </p>
     * </note>
     * 
     * @return The IDs of one or more Common Vulnerabilities and Exposure (CVE) issues that are resolved by the
     *         patch.</p> <note>
     *         <p>
     *         Currently, CVE ID values are reported only for patches with a status of <code>Missing</code> or
     *         <code>Failed</code>.
     *         </p>
     */
    public final String cveIds() {
        return cveIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(kbId());
        hashCode = 31 * hashCode + Objects.hashCode(classification());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(installedTime());
        hashCode = 31 * hashCode + Objects.hashCode(cveIds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchComplianceData)) {
            return false;
        }
        PatchComplianceData other = (PatchComplianceData) obj;
        return Objects.equals(title(), other.title()) && Objects.equals(kbId(), other.kbId())
                && Objects.equals(classification(), other.classification()) && Objects.equals(severity(), other.severity())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(installedTime(), other.installedTime()) && Objects.equals(cveIds(), other.cveIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PatchComplianceData").add("Title", title()).add("KBId", kbId())
                .add("Classification", classification()).add("Severity", severity()).add("State", stateAsString())
                .add("InstalledTime", installedTime()).add("CVEIds", cveIds()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "KBId":
            return Optional.ofNullable(clazz.cast(kbId()));
        case "Classification":
            return Optional.ofNullable(clazz.cast(classification()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "InstalledTime":
            return Optional.ofNullable(clazz.cast(installedTime()));
        case "CVEIds":
            return Optional.ofNullable(clazz.cast(cveIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PatchComplianceData, T> g) {
        return obj -> g.apply((PatchComplianceData) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PatchComplianceData> {
        /**
         * <p>
         * The title of the patch.
         * </p>
         * 
         * @param title
         *        The title of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The operating system-specific ID of the patch.
         * </p>
         * 
         * @param kbId
         *        The operating system-specific ID of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kbId(String kbId);

        /**
         * <p>
         * The classification of the patch, such as <code>SecurityUpdates</code>, <code>Updates</code>, and
         * <code>CriticalUpdates</code>.
         * </p>
         * 
         * @param classification
         *        The classification of the patch, such as <code>SecurityUpdates</code>, <code>Updates</code>, and
         *        <code>CriticalUpdates</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classification(String classification);

        /**
         * <p>
         * The severity of the patch such as <code>Critical</code>, <code>Important</code>, and <code>Moderate</code>.
         * </p>
         * 
         * @param severity
         *        The severity of the patch such as <code>Critical</code>, <code>Important</code>, and
         *        <code>Moderate</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * The state of the patch on the managed node, such as INSTALLED or FAILED.
         * </p>
         * <p>
         * For descriptions of each patch state, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
         * >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param state
         *        The state of the patch on the managed node, such as INSTALLED or FAILED.</p>
         *        <p>
         *        For descriptions of each patch state, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
         *        >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see PatchComplianceDataState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceDataState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the patch on the managed node, such as INSTALLED or FAILED.
         * </p>
         * <p>
         * For descriptions of each patch state, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
         * >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param state
         *        The state of the patch on the managed node, such as INSTALLED or FAILED.</p>
         *        <p>
         *        For descriptions of each patch state, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/compliance-about.html#compliance-monitor-patch"
         *        >About patch compliance</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see PatchComplianceDataState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceDataState
         */
        Builder state(PatchComplianceDataState state);

        /**
         * <p>
         * The date/time the patch was installed on the managed node. Not all operating systems provide this level of
         * information.
         * </p>
         * 
         * @param installedTime
         *        The date/time the patch was installed on the managed node. Not all operating systems provide this
         *        level of information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedTime(Instant installedTime);

        /**
         * <p>
         * The IDs of one or more Common Vulnerabilities and Exposure (CVE) issues that are resolved by the patch.
         * </p>
         * <note>
         * <p>
         * Currently, CVE ID values are reported only for patches with a status of <code>Missing</code> or
         * <code>Failed</code>.
         * </p>
         * </note>
         * 
         * @param cveIds
         *        The IDs of one or more Common Vulnerabilities and Exposure (CVE) issues that are resolved by the
         *        patch.</p> <note>
         *        <p>
         *        Currently, CVE ID values are reported only for patches with a status of <code>Missing</code> or
         *        <code>Failed</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cveIds(String cveIds);
    }

    static final class BuilderImpl implements Builder {
        private String title;

        private String kbId;

        private String classification;

        private String severity;

        private String state;

        private Instant installedTime;

        private String cveIds;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchComplianceData model) {
            title(model.title);
            kbId(model.kbId);
            classification(model.classification);
            severity(model.severity);
            state(model.state);
            installedTime(model.installedTime);
            cveIds(model.cveIds);
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getKbId() {
            return kbId;
        }

        public final void setKbId(String kbId) {
            this.kbId = kbId;
        }

        @Override
        public final Builder kbId(String kbId) {
            this.kbId = kbId;
            return this;
        }

        public final String getClassification() {
            return classification;
        }

        public final void setClassification(String classification) {
            this.classification = classification;
        }

        @Override
        public final Builder classification(String classification) {
            this.classification = classification;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(PatchComplianceDataState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getInstalledTime() {
            return installedTime;
        }

        public final void setInstalledTime(Instant installedTime) {
            this.installedTime = installedTime;
        }

        @Override
        public final Builder installedTime(Instant installedTime) {
            this.installedTime = installedTime;
            return this;
        }

        public final String getCveIds() {
            return cveIds;
        }

        public final void setCveIds(String cveIds) {
            this.cveIds = cveIds;
        }

        @Override
        public final Builder cveIds(String cveIds) {
            this.cveIds = cveIds;
            return this;
        }

        @Override
        public PatchComplianceData build() {
            return new PatchComplianceData(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
