/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDocumentRequest extends SsmRequest implements
        ToCopyableBuilder<CreateDocumentRequest.Builder, CreateDocumentRequest> {
    private static final SdkField<String> CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Content")
            .getter(getter(CreateDocumentRequest::content)).setter(setter(Builder::content))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Content").build()).build();

    private static final SdkField<List<DocumentRequires>> REQUIRES_FIELD = SdkField
            .<List<DocumentRequires>> builder(MarshallingType.LIST)
            .memberName("Requires")
            .getter(getter(CreateDocumentRequest::requires))
            .setter(setter(Builder::requires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Requires").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentRequires> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentRequires::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttachmentsSource>> ATTACHMENTS_FIELD = SdkField
            .<List<AttachmentsSource>> builder(MarshallingType.LIST)
            .memberName("Attachments")
            .getter(getter(CreateDocumentRequest::attachments))
            .setter(setter(Builder::attachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attachments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttachmentsSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttachmentsSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateDocumentRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(CreateDocumentRequest::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionName").getter(getter(CreateDocumentRequest::versionName)).setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentType").getter(getter(CreateDocumentRequest::documentTypeAsString))
            .setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<String> DOCUMENT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentFormat").getter(getter(CreateDocumentRequest::documentFormatAsString))
            .setter(setter(Builder::documentFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentFormat").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetType").getter(getter(CreateDocumentRequest::targetType)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateDocumentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTENT_FIELD, REQUIRES_FIELD,
            ATTACHMENTS_FIELD, NAME_FIELD, DISPLAY_NAME_FIELD, VERSION_NAME_FIELD, DOCUMENT_TYPE_FIELD, DOCUMENT_FORMAT_FIELD,
            TARGET_TYPE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Content", CONTENT_FIELD);
                    put("Requires", REQUIRES_FIELD);
                    put("Attachments", ATTACHMENTS_FIELD);
                    put("Name", NAME_FIELD);
                    put("DisplayName", DISPLAY_NAME_FIELD);
                    put("VersionName", VERSION_NAME_FIELD);
                    put("DocumentType", DOCUMENT_TYPE_FIELD);
                    put("DocumentFormat", DOCUMENT_FORMAT_FIELD);
                    put("TargetType", TARGET_TYPE_FIELD);
                    put("Tags", TAGS_FIELD);
                }
            });

    private final String content;

    private final List<DocumentRequires> requires;

    private final List<AttachmentsSource> attachments;

    private final String name;

    private final String displayName;

    private final String versionName;

    private final String documentType;

    private final String documentFormat;

    private final String targetType;

    private final List<Tag> tags;

    private CreateDocumentRequest(BuilderImpl builder) {
        super(builder);
        this.content = builder.content;
        this.requires = builder.requires;
        this.attachments = builder.attachments;
        this.name = builder.name;
        this.displayName = builder.displayName;
        this.versionName = builder.versionName;
        this.documentType = builder.documentType;
        this.documentFormat = builder.documentFormat;
        this.targetType = builder.targetType;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The content for the new SSM document in JSON or YAML format. The content of the document must not exceed 64KB.
     * This quota also includes the content specified for input parameters at runtime. We recommend storing the contents
     * for your new document in an external JSON or YAML file and referencing the file in a command.
     * </p>
     * <p>
     * For examples, see the following topics in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-console">
     * Create an SSM document (console)</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-cli"
     * >Create an SSM document (command line)</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-api"
     * >Create an SSM document (API)</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The content for the new SSM document in JSON or YAML format. The content of the document must not exceed
     *         64KB. This quota also includes the content specified for input parameters at runtime. We recommend
     *         storing the contents for your new document in an external JSON or YAML file and referencing the file in a
     *         command.</p>
     *         <p>
     *         For examples, see the following topics in the <i>Amazon Web Services Systems Manager User Guide</i>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-console"
     *         >Create an SSM document (console)</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-cli"
     *         >Create an SSM document (command line)</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-api"
     *         >Create an SSM document (API)</a>
     *         </p>
     *         </li>
     */
    public final String content() {
        return content;
    }

    /**
     * For responses, this returns true if the service returned a value for the Requires property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRequires() {
        return requires != null && !(requires instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When a user
     * creates an AppConfig configuration in an SSM document, the user must also specify a required document for
     * validation purposes. In this case, an <code>ApplicationConfiguration</code> document requires an
     * <code>ApplicationConfigurationSchema</code> document for validation purposes. For more information, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is AppConfig?</a> in
     * the <i>AppConfig User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequires} method.
     * </p>
     * 
     * @return A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When a
     *         user creates an AppConfig configuration in an SSM document, the user must also specify a required
     *         document for validation purposes. In this case, an <code>ApplicationConfiguration</code> document
     *         requires an <code>ApplicationConfigurationSchema</code> document for validation purposes. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is
     *         AppConfig?</a> in the <i>AppConfig User Guide</i>.
     */
    public final List<DocumentRequires> requires() {
        return requires;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attachments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttachments() {
        return attachments != null && !(attachments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of key-value pairs that describe attachments to a version of a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachments} method.
     * </p>
     * 
     * @return A list of key-value pairs that describe attachments to a version of a document.
     */
    public final List<AttachmentsSource> attachments() {
        return attachments;
    }

    /**
     * <p>
     * A name for the SSM document.
     * </p>
     * <important>
     * <p>
     * You can't use the following strings as document name prefixes. These are reserved by Amazon Web Services for use
     * as document name prefixes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>aws</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>amazon</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>amzn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWSEC2</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWSConfigRemediation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWSSupport</code>
     * </p>
     * </li>
     * </ul>
     * </important>
     * 
     * @return A name for the SSM document.</p> <important>
     *         <p>
     *         You can't use the following strings as document name prefixes. These are reserved by Amazon Web Services
     *         for use as document name prefixes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>aws</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>amazon</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>amzn</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWSEC2</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWSConfigRemediation</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWSSupport</code>
     *         </p>
     *         </li>
     *         </ul>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * An optional field where you can specify a friendly name for the SSM document. This value can differ for each
     * version of the document. You can update this value at a later time using the <a>UpdateDocument</a> operation.
     * </p>
     * 
     * @return An optional field where you can specify a friendly name for the SSM document. This value can differ for
     *         each version of the document. You can update this value at a later time using the <a>UpdateDocument</a>
     *         operation.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * An optional field specifying the version of the artifact you are creating with the document. For example,
     * <code>Release12.1</code>. This value is unique across all versions of a document, and can't be changed.
     * </p>
     * 
     * @return An optional field specifying the version of the artifact you are creating with the document. For example,
     *         <code>Release12.1</code>. This value is unique across all versions of a document, and can't be changed.
     */
    public final String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The type of document to create.
     * </p>
     * <note>
     * <p>
     * The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for AppConfig.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document to create.</p> <note>
     *         <p>
     *         The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
     *         AppConfig.
     *         </p>
     * @see DocumentType
     */
    public final DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The type of document to create.
     * </p>
     * <note>
     * <p>
     * The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for AppConfig.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document to create.</p> <note>
     *         <p>
     *         The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
     *         AppConfig.
     *         </p>
     * @see DocumentType
     */
    public final String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the default
     * format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the
     *         default format.
     * @see DocumentFormat
     */
    public final DocumentFormat documentFormat() {
        return DocumentFormat.fromValue(documentFormat);
    }

    /**
     * <p>
     * Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the default
     * format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the
     *         default format.
     * @see DocumentFormat
     */
    public final String documentFormatAsString() {
        return documentFormat;
    }

    /**
     * <p>
     * Specify a target type to define the kinds of resources the document can run on. For example, to run a document on
     * EC2 instances, specify the following value: <code>/AWS::EC2::Instance</code>. If you specify a value of '/' the
     * document can run on all types of resources. If you don't specify a value, the document can't run on any
     * resources. For a list of valid resource types, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return Specify a target type to define the kinds of resources the document can run on. For example, to run a
     *         document on EC2 instances, specify the following value: <code>/AWS::EC2::Instance</code>. If you specify
     *         a value of '/' the document can run on all types of resources. If you don't specify a value, the document
     *         can't run on any resources. For a list of valid resource types, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     */
    public final String targetType() {
        return targetType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways, such
     * as by purpose, owner, or environment. For example, you might want to tag an SSM document to identify the types of
     * targets or the environment where it will run. In this case, you could specify the following key-value pairs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Key=OS,Value=Windows</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Key=Environment,Value=Production</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
     *         ways, such as by purpose, owner, or environment. For example, you might want to tag an SSM document to
     *         identify the types of targets or the environment where it will run. In this case, you could specify the
     *         following key-value pairs:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Key=OS,Value=Windows</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Key=Environment,Value=Production</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
     *         </p>
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(content());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequires() ? requires() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachments() ? attachments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(documentFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDocumentRequest)) {
            return false;
        }
        CreateDocumentRequest other = (CreateDocumentRequest) obj;
        return Objects.equals(content(), other.content()) && hasRequires() == other.hasRequires()
                && Objects.equals(requires(), other.requires()) && hasAttachments() == other.hasAttachments()
                && Objects.equals(attachments(), other.attachments()) && Objects.equals(name(), other.name())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(versionName(), other.versionName())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(documentFormatAsString(), other.documentFormatAsString())
                && Objects.equals(targetType(), other.targetType()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateDocumentRequest").add("Content", content())
                .add("Requires", hasRequires() ? requires() : null).add("Attachments", hasAttachments() ? attachments() : null)
                .add("Name", name()).add("DisplayName", displayName()).add("VersionName", versionName())
                .add("DocumentType", documentTypeAsString()).add("DocumentFormat", documentFormatAsString())
                .add("TargetType", targetType()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Content":
            return Optional.ofNullable(clazz.cast(content()));
        case "Requires":
            return Optional.ofNullable(clazz.cast(requires()));
        case "Attachments":
            return Optional.ofNullable(clazz.cast(attachments()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "DocumentFormat":
            return Optional.ofNullable(clazz.cast(documentFormatAsString()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateDocumentRequest, T> g) {
        return obj -> g.apply((CreateDocumentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateDocumentRequest> {
        /**
         * <p>
         * The content for the new SSM document in JSON or YAML format. The content of the document must not exceed
         * 64KB. This quota also includes the content specified for input parameters at runtime. We recommend storing
         * the contents for your new document in an external JSON or YAML file and referencing the file in a command.
         * </p>
         * <p>
         * For examples, see the following topics in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-console"
         * >Create an SSM document (console)</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-cli"
         * >Create an SSM document (command line)</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-api"
         * >Create an SSM document (API)</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param content
         *        The content for the new SSM document in JSON or YAML format. The content of the document must not
         *        exceed 64KB. This quota also includes the content specified for input parameters at runtime. We
         *        recommend storing the contents for your new document in an external JSON or YAML file and referencing
         *        the file in a command.</p>
         *        <p>
         *        For examples, see the following topics in the <i>Amazon Web Services Systems Manager User Guide</i>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-console"
         *        >Create an SSM document (console)</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-cli"
         *        >Create an SSM document (command line)</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/documents-using.html#create-ssm-document-api"
         *        >Create an SSM document (API)</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(String content);

        /**
         * <p>
         * A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When a user
         * creates an AppConfig configuration in an SSM document, the user must also specify a required document for
         * validation purposes. In this case, an <code>ApplicationConfiguration</code> document requires an
         * <code>ApplicationConfigurationSchema</code> document for validation purposes. For more information, see <a
         * href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is AppConfig?</a>
         * in the <i>AppConfig User Guide</i>.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When
         *        a user creates an AppConfig configuration in an SSM document, the user must also specify a required
         *        document for validation purposes. In this case, an <code>ApplicationConfiguration</code> document
         *        requires an <code>ApplicationConfigurationSchema</code> document for validation purposes. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is
         *        AppConfig?</a> in the <i>AppConfig User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(Collection<DocumentRequires> requires);

        /**
         * <p>
         * A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When a user
         * creates an AppConfig configuration in an SSM document, the user must also specify a required document for
         * validation purposes. In this case, an <code>ApplicationConfiguration</code> document requires an
         * <code>ApplicationConfigurationSchema</code> document for validation purposes. For more information, see <a
         * href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is AppConfig?</a>
         * in the <i>AppConfig User Guide</i>.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When
         *        a user creates an AppConfig configuration in an SSM document, the user must also specify a required
         *        document for validation purposes. In this case, an <code>ApplicationConfiguration</code> document
         *        requires an <code>ApplicationConfigurationSchema</code> document for validation purposes. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is
         *        AppConfig?</a> in the <i>AppConfig User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(DocumentRequires... requires);

        /**
         * <p>
         * A list of SSM documents required by a document. This parameter is used exclusively by AppConfig. When a user
         * creates an AppConfig configuration in an SSM document, the user must also specify a required document for
         * validation purposes. In this case, an <code>ApplicationConfiguration</code> document requires an
         * <code>ApplicationConfigurationSchema</code> document for validation purposes. For more information, see <a
         * href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is AppConfig?</a>
         * in the <i>AppConfig User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.DocumentRequires#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder#build()} is called immediately and
         * its result is passed to {@link #requires(List<DocumentRequires>)}.
         * 
         * @param requires
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requires(java.util.Collection<DocumentRequires>)
         */
        Builder requires(Consumer<DocumentRequires.Builder>... requires);

        /**
         * <p>
         * A list of key-value pairs that describe attachments to a version of a document.
         * </p>
         * 
         * @param attachments
         *        A list of key-value pairs that describe attachments to a version of a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Collection<AttachmentsSource> attachments);

        /**
         * <p>
         * A list of key-value pairs that describe attachments to a version of a document.
         * </p>
         * 
         * @param attachments
         *        A list of key-value pairs that describe attachments to a version of a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(AttachmentsSource... attachments);

        /**
         * <p>
         * A list of key-value pairs that describe attachments to a version of a document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.AttachmentsSource.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.AttachmentsSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.AttachmentsSource.Builder#build()} is called immediately and
         * its result is passed to {@link #attachments(List<AttachmentsSource>)}.
         * 
         * @param attachments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.AttachmentsSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachments(java.util.Collection<AttachmentsSource>)
         */
        Builder attachments(Consumer<AttachmentsSource.Builder>... attachments);

        /**
         * <p>
         * A name for the SSM document.
         * </p>
         * <important>
         * <p>
         * You can't use the following strings as document name prefixes. These are reserved by Amazon Web Services for
         * use as document name prefixes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>aws</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>amazon</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>amzn</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSEC2</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSConfigRemediation</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWSSupport</code>
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param name
         *        A name for the SSM document.</p> <important>
         *        <p>
         *        You can't use the following strings as document name prefixes. These are reserved by Amazon Web
         *        Services for use as document name prefixes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>aws</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>amazon</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>amzn</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSEC2</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSConfigRemediation</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWSSupport</code>
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An optional field where you can specify a friendly name for the SSM document. This value can differ for each
         * version of the document. You can update this value at a later time using the <a>UpdateDocument</a> operation.
         * </p>
         * 
         * @param displayName
         *        An optional field where you can specify a friendly name for the SSM document. This value can differ
         *        for each version of the document. You can update this value at a later time using the
         *        <a>UpdateDocument</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * An optional field specifying the version of the artifact you are creating with the document. For example,
         * <code>Release12.1</code>. This value is unique across all versions of a document, and can't be changed.
         * </p>
         * 
         * @param versionName
         *        An optional field specifying the version of the artifact you are creating with the document. For
         *        example, <code>Release12.1</code>. This value is unique across all versions of a document, and can't
         *        be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The type of document to create.
         * </p>
         * <note>
         * <p>
         * The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
         * AppConfig.
         * </p>
         * </note>
         * 
         * @param documentType
         *        The type of document to create.</p> <note>
         *        <p>
         *        The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
         *        AppConfig.
         *        </p>
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The type of document to create.
         * </p>
         * <note>
         * <p>
         * The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
         * AppConfig.
         * </p>
         * </note>
         * 
         * @param documentType
         *        The type of document to create.</p> <note>
         *        <p>
         *        The <code>DeploymentStrategy</code> document type is an internal-use-only document type reserved for
         *        AppConfig.
         *        </p>
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the
         * default format.
         * </p>
         * 
         * @param documentFormat
         *        Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is
         *        the default format.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(String documentFormat);

        /**
         * <p>
         * Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is the
         * default format.
         * </p>
         * 
         * @param documentFormat
         *        Specify the document format for the request. The document format can be JSON, YAML, or TEXT. JSON is
         *        the default format.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(DocumentFormat documentFormat);

        /**
         * <p>
         * Specify a target type to define the kinds of resources the document can run on. For example, to run a
         * document on EC2 instances, specify the following value: <code>/AWS::EC2::Instance</code>. If you specify a
         * value of '/' the document can run on all types of resources. If you don't specify a value, the document can't
         * run on any resources. For a list of valid resource types, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param targetType
         *        Specify a target type to define the kinds of resources the document can run on. For example, to run a
         *        document on EC2 instances, specify the following value: <code>/AWS::EC2::Instance</code>. If you
         *        specify a value of '/' the document can run on all types of resources. If you don't specify a value,
         *        the document can't run on any resources. For a list of valid resource types, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag an SSM document to identify the
         * types of targets or the environment where it will run. In this case, you could specify the following
         * key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=Environment,Value=Production</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag an SSM document to
         *        identify the types of targets or the environment where it will run. In this case, you could specify
         *        the following key-value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=Environment,Value=Production</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag an SSM document to identify the
         * types of targets or the environment where it will run. In this case, you could specify the following
         * key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=Environment,Value=Production</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different
         *        ways, such as by purpose, owner, or environment. For example, you might want to tag an SSM document to
         *        identify the types of targets or the environment where it will run. In this case, you could specify
         *        the following key-value pairs:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Key=OS,Value=Windows</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Key=Environment,Value=Production</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. Tags enable you to categorize a resource in different ways,
         * such as by purpose, owner, or environment. For example, you might want to tag an SSM document to identify the
         * types of targets or the environment where it will run. In this case, you could specify the following
         * key-value pairs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Key=OS,Value=Windows</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Key=Environment,Value=Production</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * To add tags to an existing SSM document, use the <a>AddTagsToResource</a> operation.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String content;

        private List<DocumentRequires> requires = DefaultSdkAutoConstructList.getInstance();

        private List<AttachmentsSource> attachments = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private String displayName;

        private String versionName;

        private String documentType;

        private String documentFormat;

        private String targetType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDocumentRequest model) {
            super(model);
            content(model.content);
            requires(model.requires);
            attachments(model.attachments);
            name(model.name);
            displayName(model.displayName);
            versionName(model.versionName);
            documentType(model.documentType);
            documentFormat(model.documentFormat);
            targetType(model.targetType);
            tags(model.tags);
        }

        public final String getContent() {
            return content;
        }

        public final void setContent(String content) {
            this.content = content;
        }

        @Override
        public final Builder content(String content) {
            this.content = content;
            return this;
        }

        public final List<DocumentRequires.Builder> getRequires() {
            List<DocumentRequires.Builder> result = DocumentRequiresListCopier.copyToBuilder(this.requires);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRequires(Collection<DocumentRequires.BuilderImpl> requires) {
            this.requires = DocumentRequiresListCopier.copyFromBuilder(requires);
        }

        @Override
        public final Builder requires(Collection<DocumentRequires> requires) {
            this.requires = DocumentRequiresListCopier.copy(requires);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(DocumentRequires... requires) {
            requires(Arrays.asList(requires));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(Consumer<DocumentRequires.Builder>... requires) {
            requires(Stream.of(requires).map(c -> DocumentRequires.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<AttachmentsSource.Builder> getAttachments() {
            List<AttachmentsSource.Builder> result = AttachmentsSourceListCopier.copyToBuilder(this.attachments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachments(Collection<AttachmentsSource.BuilderImpl> attachments) {
            this.attachments = AttachmentsSourceListCopier.copyFromBuilder(attachments);
        }

        @Override
        public final Builder attachments(Collection<AttachmentsSource> attachments) {
            this.attachments = AttachmentsSourceListCopier.copy(attachments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(AttachmentsSource... attachments) {
            attachments(Arrays.asList(attachments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Consumer<AttachmentsSource.Builder>... attachments) {
            attachments(Stream.of(attachments).map(c -> AttachmentsSource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getVersionName() {
            return versionName;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final String getDocumentType() {
            return documentType;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType == null ? null : documentType.toString());
            return this;
        }

        public final String getDocumentFormat() {
            return documentFormat;
        }

        public final void setDocumentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
        }

        @Override
        public final Builder documentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
            return this;
        }

        @Override
        public final Builder documentFormat(DocumentFormat documentFormat) {
            this.documentFormat(documentFormat == null ? null : documentFormat.toString());
            return this;
        }

        public final String getTargetType() {
            return targetType;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        @Override
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateDocumentRequest build() {
            return new CreateDocumentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
