/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The parameters for a <code>RUN_COMMAND</code> task type.
 * </p>
 * <p>
 * For information about specifying and updating task parameters, see <a>RegisterTaskWithMaintenanceWindow</a> and
 * <a>UpdateMaintenanceWindowTask</a>.
 * </p>
 * <note>
 * <p>
 * <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3) bucket to
 * contain logs, instead use the <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
 * <code>TaskInvocationParameters</code> structure. For information about how Amazon Web Services Systems Manager
 * handles these options for the supported maintenance window task types, see
 * <a>MaintenanceWindowTaskInvocationParameters</a>.
 * </p>
 * <p>
 * <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs, instead use
 * the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure. For information about how
 * Systems Manager handles these options for the supported maintenance window task types, see
 * <a>MaintenanceWindowTaskInvocationParameters</a>.
 * </p>
 * <p>
 * For <code>RUN_COMMAND</code> tasks, Systems Manager uses specified values for <code>TaskParameters</code> and
 * <code>LoggingInfo</code> only if no values are specified for <code>TaskInvocationParameters</code>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MaintenanceWindowRunCommandParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<MaintenanceWindowRunCommandParameters.Builder, MaintenanceWindowRunCommandParameters> {
    private static final SdkField<String> COMMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Comment")
            .getter(getter(MaintenanceWindowRunCommandParameters::comment)).setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment").build()).build();

    private static final SdkField<CloudWatchOutputConfig> CLOUD_WATCH_OUTPUT_CONFIG_FIELD = SdkField
            .<CloudWatchOutputConfig> builder(MarshallingType.SDK_POJO).memberName("CloudWatchOutputConfig")
            .getter(getter(MaintenanceWindowRunCommandParameters::cloudWatchOutputConfig))
            .setter(setter(Builder::cloudWatchOutputConfig)).constructor(CloudWatchOutputConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchOutputConfig").build())
            .build();

    private static final SdkField<String> DOCUMENT_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentHash").getter(getter(MaintenanceWindowRunCommandParameters::documentHash))
            .setter(setter(Builder::documentHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentHash").build()).build();

    private static final SdkField<String> DOCUMENT_HASH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentHashType").getter(getter(MaintenanceWindowRunCommandParameters::documentHashTypeAsString))
            .setter(setter(Builder::documentHashType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentHashType").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(MaintenanceWindowRunCommandParameters::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<NotificationConfig> NOTIFICATION_CONFIG_FIELD = SdkField
            .<NotificationConfig> builder(MarshallingType.SDK_POJO).memberName("NotificationConfig")
            .getter(getter(MaintenanceWindowRunCommandParameters::notificationConfig))
            .setter(setter(Builder::notificationConfig)).constructor(NotificationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationConfig").build())
            .build();

    private static final SdkField<String> OUTPUT_S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputS3BucketName").getter(getter(MaintenanceWindowRunCommandParameters::outputS3BucketName))
            .setter(setter(Builder::outputS3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3BucketName").build())
            .build();

    private static final SdkField<String> OUTPUT_S3_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputS3KeyPrefix").getter(getter(MaintenanceWindowRunCommandParameters::outputS3KeyPrefix))
            .setter(setter(Builder::outputS3KeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3KeyPrefix").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(MaintenanceWindowRunCommandParameters::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> SERVICE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceRoleArn").getter(getter(MaintenanceWindowRunCommandParameters::serviceRoleArn))
            .setter(setter(Builder::serviceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRoleArn").build()).build();

    private static final SdkField<Integer> TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeoutSeconds").getter(getter(MaintenanceWindowRunCommandParameters::timeoutSeconds))
            .setter(setter(Builder::timeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutSeconds").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMENT_FIELD,
            CLOUD_WATCH_OUTPUT_CONFIG_FIELD, DOCUMENT_HASH_FIELD, DOCUMENT_HASH_TYPE_FIELD, DOCUMENT_VERSION_FIELD,
            NOTIFICATION_CONFIG_FIELD, OUTPUT_S3_BUCKET_NAME_FIELD, OUTPUT_S3_KEY_PREFIX_FIELD, PARAMETERS_FIELD,
            SERVICE_ROLE_ARN_FIELD, TIMEOUT_SECONDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Comment", COMMENT_FIELD);
                    put("CloudWatchOutputConfig", CLOUD_WATCH_OUTPUT_CONFIG_FIELD);
                    put("DocumentHash", DOCUMENT_HASH_FIELD);
                    put("DocumentHashType", DOCUMENT_HASH_TYPE_FIELD);
                    put("DocumentVersion", DOCUMENT_VERSION_FIELD);
                    put("NotificationConfig", NOTIFICATION_CONFIG_FIELD);
                    put("OutputS3BucketName", OUTPUT_S3_BUCKET_NAME_FIELD);
                    put("OutputS3KeyPrefix", OUTPUT_S3_KEY_PREFIX_FIELD);
                    put("Parameters", PARAMETERS_FIELD);
                    put("ServiceRoleArn", SERVICE_ROLE_ARN_FIELD);
                    put("TimeoutSeconds", TIMEOUT_SECONDS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String comment;

    private final CloudWatchOutputConfig cloudWatchOutputConfig;

    private final String documentHash;

    private final String documentHashType;

    private final String documentVersion;

    private final NotificationConfig notificationConfig;

    private final String outputS3BucketName;

    private final String outputS3KeyPrefix;

    private final Map<String, List<String>> parameters;

    private final String serviceRoleArn;

    private final Integer timeoutSeconds;

    private MaintenanceWindowRunCommandParameters(BuilderImpl builder) {
        this.comment = builder.comment;
        this.cloudWatchOutputConfig = builder.cloudWatchOutputConfig;
        this.documentHash = builder.documentHash;
        this.documentHashType = builder.documentHashType;
        this.documentVersion = builder.documentVersion;
        this.notificationConfig = builder.notificationConfig;
        this.outputS3BucketName = builder.outputS3BucketName;
        this.outputS3KeyPrefix = builder.outputS3KeyPrefix;
        this.parameters = builder.parameters;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.timeoutSeconds = builder.timeoutSeconds;
    }

    /**
     * <p>
     * Information about the commands to run.
     * </p>
     * 
     * @return Information about the commands to run.
     */
    public final String comment() {
        return comment;
    }

    /**
     * Returns the value of the CloudWatchOutputConfig property for this object.
     * 
     * @return The value of the CloudWatchOutputConfig property for this object.
     */
    public final CloudWatchOutputConfig cloudWatchOutputConfig() {
        return cloudWatchOutputConfig;
    }

    /**
     * <p>
     * The SHA-256 or SHA-1 hash created by the system when the document was created. SHA-1 hashes have been deprecated.
     * </p>
     * 
     * @return The SHA-256 or SHA-1 hash created by the system when the document was created. SHA-1 hashes have been
     *         deprecated.
     */
    public final String documentHash() {
        return documentHash;
    }

    /**
     * <p>
     * SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentHashType}
     * will return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentHashTypeAsString}.
     * </p>
     * 
     * @return SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
     * @see DocumentHashType
     */
    public final DocumentHashType documentHashType() {
        return DocumentHashType.fromValue(documentHashType);
    }

    /**
     * <p>
     * SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentHashType}
     * will return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentHashTypeAsString}.
     * </p>
     * 
     * @return SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
     * @see DocumentHashType
     */
    public final String documentHashTypeAsString() {
        return documentHashType;
    }

    /**
     * <p>
     * The Amazon Web Services Systems Manager document (SSM document) version to use in the request. You can specify
     * <code>$DEFAULT</code>, <code>$LATEST</code>, or a specific version number. If you run commands by using the
     * Amazon Web Services CLI, then you must escape the first two options by using a backslash. If you specify a
     * version number, then you don't need to use the backslash. For example:
     * </p>
     * <p>
     * <code>--document-version "\$DEFAULT"</code>
     * </p>
     * <p>
     * <code>--document-version "\$LATEST"</code>
     * </p>
     * <p>
     * <code>--document-version "3"</code>
     * </p>
     * 
     * @return The Amazon Web Services Systems Manager document (SSM document) version to use in the request. You can
     *         specify <code>$DEFAULT</code>, <code>$LATEST</code>, or a specific version number. If you run commands by
     *         using the Amazon Web Services CLI, then you must escape the first two options by using a backslash. If
     *         you specify a version number, then you don't need to use the backslash. For example:</p>
     *         <p>
     *         <code>--document-version "\$DEFAULT"</code>
     *         </p>
     *         <p>
     *         <code>--document-version "\$LATEST"</code>
     *         </p>
     *         <p>
     *         <code>--document-version "3"</code>
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * Configurations for sending notifications about command status changes on a per-managed node basis.
     * </p>
     * 
     * @return Configurations for sending notifications about command status changes on a per-managed node basis.
     */
    public final NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    /**
     * <p>
     * The name of the Amazon Simple Storage Service (Amazon S3) bucket.
     * </p>
     * 
     * @return The name of the Amazon Simple Storage Service (Amazon S3) bucket.
     */
    public final String outputS3BucketName() {
        return outputS3BucketName;
    }

    /**
     * <p>
     * The S3 bucket subfolder.
     * </p>
     * 
     * @return The S3 bucket subfolder.
     */
    public final String outputS3KeyPrefix() {
        return outputS3KeyPrefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters for the <code>RUN_COMMAND</code> task execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The parameters for the <code>RUN_COMMAND</code> task execution.
     */
    public final Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume when
     * running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
     * service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in your
     * account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.
     * </p>
     * <p>
     * However, for an improved security posture, we strongly recommend creating a custom policy and custom service role
     * for running your maintenance window tasks. The policy can be crafted to provide only the permissions needed for
     * your particular maintenance window tasks. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html">Setting
     * up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume
     *         when running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
     *         service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in
     *         your account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.</p>
     *         <p>
     *         However, for an improved security posture, we strongly recommend creating a custom policy and custom
     *         service role for running your maintenance window tasks. The policy can be crafted to provide only the
     *         permissions needed for your particular maintenance window tasks. For more information, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
     *         >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
     */
    public final String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * If this time is reached and the command hasn't already started running, it doesn't run.
     * </p>
     * 
     * @return If this time is reached and the command hasn't already started running, it doesn't run.
     */
    public final Integer timeoutSeconds() {
        return timeoutSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchOutputConfig());
        hashCode = 31 * hashCode + Objects.hashCode(documentHash());
        hashCode = 31 * hashCode + Objects.hashCode(documentHashTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3KeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutSeconds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowRunCommandParameters)) {
            return false;
        }
        MaintenanceWindowRunCommandParameters other = (MaintenanceWindowRunCommandParameters) obj;
        return Objects.equals(comment(), other.comment())
                && Objects.equals(cloudWatchOutputConfig(), other.cloudWatchOutputConfig())
                && Objects.equals(documentHash(), other.documentHash())
                && Objects.equals(documentHashTypeAsString(), other.documentHashTypeAsString())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(notificationConfig(), other.notificationConfig())
                && Objects.equals(outputS3BucketName(), other.outputS3BucketName())
                && Objects.equals(outputS3KeyPrefix(), other.outputS3KeyPrefix()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(timeoutSeconds(), other.timeoutSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MaintenanceWindowRunCommandParameters").add("Comment", comment())
                .add("CloudWatchOutputConfig", cloudWatchOutputConfig()).add("DocumentHash", documentHash())
                .add("DocumentHashType", documentHashTypeAsString()).add("DocumentVersion", documentVersion())
                .add("NotificationConfig", notificationConfig()).add("OutputS3BucketName", outputS3BucketName())
                .add("OutputS3KeyPrefix", outputS3KeyPrefix())
                .add("Parameters", parameters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ServiceRoleArn", serviceRoleArn()).add("TimeoutSeconds", timeoutSeconds()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "CloudWatchOutputConfig":
            return Optional.ofNullable(clazz.cast(cloudWatchOutputConfig()));
        case "DocumentHash":
            return Optional.ofNullable(clazz.cast(documentHash()));
        case "DocumentHashType":
            return Optional.ofNullable(clazz.cast(documentHashTypeAsString()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "NotificationConfig":
            return Optional.ofNullable(clazz.cast(notificationConfig()));
        case "OutputS3BucketName":
            return Optional.ofNullable(clazz.cast(outputS3BucketName()));
        case "OutputS3KeyPrefix":
            return Optional.ofNullable(clazz.cast(outputS3KeyPrefix()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "ServiceRoleArn":
            return Optional.ofNullable(clazz.cast(serviceRoleArn()));
        case "TimeoutSeconds":
            return Optional.ofNullable(clazz.cast(timeoutSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MaintenanceWindowRunCommandParameters, T> g) {
        return obj -> g.apply((MaintenanceWindowRunCommandParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MaintenanceWindowRunCommandParameters> {
        /**
         * <p>
         * Information about the commands to run.
         * </p>
         * 
         * @param comment
         *        Information about the commands to run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * Sets the value of the CloudWatchOutputConfig property for this object.
         *
         * @param cloudWatchOutputConfig
         *        The new value for the CloudWatchOutputConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig);

        /**
         * Sets the value of the CloudWatchOutputConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link CloudWatchOutputConfig.Builder} avoiding
         * the need to create one manually via {@link CloudWatchOutputConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CloudWatchOutputConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchOutputConfig(CloudWatchOutputConfig)}.
         * 
         * @param cloudWatchOutputConfig
         *        a consumer that will call methods on {@link CloudWatchOutputConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchOutputConfig(CloudWatchOutputConfig)
         */
        default Builder cloudWatchOutputConfig(Consumer<CloudWatchOutputConfig.Builder> cloudWatchOutputConfig) {
            return cloudWatchOutputConfig(CloudWatchOutputConfig.builder().applyMutation(cloudWatchOutputConfig).build());
        }

        /**
         * <p>
         * The SHA-256 or SHA-1 hash created by the system when the document was created. SHA-1 hashes have been
         * deprecated.
         * </p>
         * 
         * @param documentHash
         *        The SHA-256 or SHA-1 hash created by the system when the document was created. SHA-1 hashes have been
         *        deprecated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentHash(String documentHash);

        /**
         * <p>
         * SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
         * </p>
         * 
         * @param documentHashType
         *        SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder documentHashType(String documentHashType);

        /**
         * <p>
         * SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
         * </p>
         * 
         * @param documentHashType
         *        SHA-256 or SHA-1. SHA-1 hashes have been deprecated.
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder documentHashType(DocumentHashType documentHashType);

        /**
         * <p>
         * The Amazon Web Services Systems Manager document (SSM document) version to use in the request. You can
         * specify <code>$DEFAULT</code>, <code>$LATEST</code>, or a specific version number. If you run commands by
         * using the Amazon Web Services CLI, then you must escape the first two options by using a backslash. If you
         * specify a version number, then you don't need to use the backslash. For example:
         * </p>
         * <p>
         * <code>--document-version "\$DEFAULT"</code>
         * </p>
         * <p>
         * <code>--document-version "\$LATEST"</code>
         * </p>
         * <p>
         * <code>--document-version "3"</code>
         * </p>
         * 
         * @param documentVersion
         *        The Amazon Web Services Systems Manager document (SSM document) version to use in the request. You can
         *        specify <code>$DEFAULT</code>, <code>$LATEST</code>, or a specific version number. If you run commands
         *        by using the Amazon Web Services CLI, then you must escape the first two options by using a backslash.
         *        If you specify a version number, then you don't need to use the backslash. For example:</p>
         *        <p>
         *        <code>--document-version "\$DEFAULT"</code>
         *        </p>
         *        <p>
         *        <code>--document-version "\$LATEST"</code>
         *        </p>
         *        <p>
         *        <code>--document-version "3"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per-managed node basis.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications about command status changes on a per-managed node basis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per-managed node basis.
         * </p>
         * This is a convenience method that creates an instance of the {@link NotificationConfig.Builder} avoiding the
         * need to create one manually via {@link NotificationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NotificationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #notificationConfig(NotificationConfig)}.
         * 
         * @param notificationConfig
         *        a consumer that will call methods on {@link NotificationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfig(NotificationConfig)
         */
        default Builder notificationConfig(Consumer<NotificationConfig.Builder> notificationConfig) {
            return notificationConfig(NotificationConfig.builder().applyMutation(notificationConfig).build());
        }

        /**
         * <p>
         * The name of the Amazon Simple Storage Service (Amazon S3) bucket.
         * </p>
         * 
         * @param outputS3BucketName
         *        The name of the Amazon Simple Storage Service (Amazon S3) bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3BucketName(String outputS3BucketName);

        /**
         * <p>
         * The S3 bucket subfolder.
         * </p>
         * 
         * @param outputS3KeyPrefix
         *        The S3 bucket subfolder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3KeyPrefix(String outputS3KeyPrefix);

        /**
         * <p>
         * The parameters for the <code>RUN_COMMAND</code> task execution.
         * </p>
         * 
         * @param parameters
         *        The parameters for the <code>RUN_COMMAND</code> task execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume when
         * running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
         * service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in your
         * account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.
         * </p>
         * <p>
         * However, for an improved security posture, we strongly recommend creating a custom policy and custom service
         * role for running your maintenance window tasks. The policy can be crafted to provide only the permissions
         * needed for your particular maintenance window tasks. For more information, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
         * >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param serviceRoleArn
         *        The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to
         *        assume when running a maintenance window task. If you do not specify a service role ARN, Systems
         *        Manager uses a service-linked role in your account. If no appropriate service-linked role for Systems
         *        Manager exists in your account, it is created when you run
         *        <code>RegisterTaskWithMaintenanceWindow</code>.</p>
         *        <p>
         *        However, for an improved security posture, we strongly recommend creating a custom policy and custom
         *        service role for running your maintenance window tasks. The policy can be crafted to provide only the
         *        permissions needed for your particular maintenance window tasks. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
         *        >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * If this time is reached and the command hasn't already started running, it doesn't run.
         * </p>
         * 
         * @param timeoutSeconds
         *        If this time is reached and the command hasn't already started running, it doesn't run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutSeconds(Integer timeoutSeconds);
    }

    static final class BuilderImpl implements Builder {
        private String comment;

        private CloudWatchOutputConfig cloudWatchOutputConfig;

        private String documentHash;

        private String documentHashType;

        private String documentVersion;

        private NotificationConfig notificationConfig;

        private String outputS3BucketName;

        private String outputS3KeyPrefix;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String serviceRoleArn;

        private Integer timeoutSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowRunCommandParameters model) {
            comment(model.comment);
            cloudWatchOutputConfig(model.cloudWatchOutputConfig);
            documentHash(model.documentHash);
            documentHashType(model.documentHashType);
            documentVersion(model.documentVersion);
            notificationConfig(model.notificationConfig);
            outputS3BucketName(model.outputS3BucketName);
            outputS3KeyPrefix(model.outputS3KeyPrefix);
            parameters(model.parameters);
            serviceRoleArn(model.serviceRoleArn);
            timeoutSeconds(model.timeoutSeconds);
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final CloudWatchOutputConfig.Builder getCloudWatchOutputConfig() {
            return cloudWatchOutputConfig != null ? cloudWatchOutputConfig.toBuilder() : null;
        }

        public final void setCloudWatchOutputConfig(CloudWatchOutputConfig.BuilderImpl cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig != null ? cloudWatchOutputConfig.build() : null;
        }

        @Override
        public final Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig;
            return this;
        }

        public final String getDocumentHash() {
            return documentHash;
        }

        public final void setDocumentHash(String documentHash) {
            this.documentHash = documentHash;
        }

        @Override
        public final Builder documentHash(String documentHash) {
            this.documentHash = documentHash;
            return this;
        }

        public final String getDocumentHashType() {
            return documentHashType;
        }

        public final void setDocumentHashType(String documentHashType) {
            this.documentHashType = documentHashType;
        }

        @Override
        public final Builder documentHashType(String documentHashType) {
            this.documentHashType = documentHashType;
            return this;
        }

        @Override
        public final Builder documentHashType(DocumentHashType documentHashType) {
            this.documentHashType(documentHashType == null ? null : documentHashType.toString());
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final NotificationConfig.Builder getNotificationConfig() {
            return notificationConfig != null ? notificationConfig.toBuilder() : null;
        }

        public final void setNotificationConfig(NotificationConfig.BuilderImpl notificationConfig) {
            this.notificationConfig = notificationConfig != null ? notificationConfig.build() : null;
        }

        @Override
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final String getOutputS3BucketName() {
            return outputS3BucketName;
        }

        public final void setOutputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
        }

        @Override
        public final Builder outputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
            return this;
        }

        public final String getOutputS3KeyPrefix() {
            return outputS3KeyPrefix;
        }

        public final void setOutputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
        }

        @Override
        public final Builder outputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final Integer getTimeoutSeconds() {
            return timeoutSeconds;
        }

        public final void setTimeoutSeconds(Integer timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
        }

        @Override
        public final Builder timeoutSeconds(Integer timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
            return this;
        }

        @Override
        public MaintenanceWindowRunCommandParameters build() {
            return new MaintenanceWindowRunCommandParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
