/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inventory item schema definition. Users can use this to compose inventory query filters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InventoryItemSchema implements SdkPojo, Serializable,
        ToCopyableBuilder<InventoryItemSchema.Builder, InventoryItemSchema> {
    private static final SdkField<String> TYPE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TypeName").getter(getter(InventoryItemSchema::typeName)).setter(setter(Builder::typeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TypeName").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(InventoryItemSchema::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<InventoryItemAttribute>> ATTRIBUTES_FIELD = SdkField
            .<List<InventoryItemAttribute>> builder(MarshallingType.LIST)
            .memberName("Attributes")
            .getter(getter(InventoryItemSchema::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InventoryItemAttribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(InventoryItemAttribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(InventoryItemSchema::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_NAME_FIELD,
            VERSION_FIELD, ATTRIBUTES_FIELD, DISPLAY_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("TypeName", TYPE_NAME_FIELD);
                    put("Version", VERSION_FIELD);
                    put("Attributes", ATTRIBUTES_FIELD);
                    put("DisplayName", DISPLAY_NAME_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String typeName;

    private final String version;

    private final List<InventoryItemAttribute> attributes;

    private final String displayName;

    private InventoryItemSchema(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.version = builder.version;
        this.attributes = builder.attributes;
        this.displayName = builder.displayName;
    }

    /**
     * <p>
     * The name of the inventory type. Default inventory item type names start with Amazon Web Services. Custom
     * inventory type names will start with Custom. Default inventory item types include the following:
     * <code>AWS:AWSComponent</code>, <code>AWS:Application</code>, <code>AWS:InstanceInformation</code>,
     * <code>AWS:Network</code>, and <code>AWS:WindowsUpdate</code>.
     * </p>
     * 
     * @return The name of the inventory type. Default inventory item type names start with Amazon Web Services. Custom
     *         inventory type names will start with Custom. Default inventory item types include the following:
     *         <code>AWS:AWSComponent</code>, <code>AWS:Application</code>, <code>AWS:InstanceInformation</code>,
     *         <code>AWS:Network</code>, and <code>AWS:WindowsUpdate</code>.
     */
    public final String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory item.
     * </p>
     * 
     * @return The schema version for the inventory item.
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The schema attributes for inventory. This contains data type and attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return The schema attributes for inventory. This contains data type and attribute name.
     */
    public final List<InventoryItemAttribute> attributes() {
        return attributes;
    }

    /**
     * <p>
     * The alias name of the inventory type. The alias name is used for display purposes.
     * </p>
     * 
     * @return The alias name of the inventory type. The alias name is used for display purposes.
     */
    public final String displayName() {
        return displayName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryItemSchema)) {
            return false;
        }
        InventoryItemSchema other = (InventoryItemSchema) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(version(), other.version())
                && hasAttributes() == other.hasAttributes() && Objects.equals(attributes(), other.attributes())
                && Objects.equals(displayName(), other.displayName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InventoryItemSchema").add("TypeName", typeName()).add("Version", version())
                .add("Attributes", hasAttributes() ? attributes() : null).add("DisplayName", displayName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.ofNullable(clazz.cast(typeName()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<InventoryItemSchema, T> g) {
        return obj -> g.apply((InventoryItemSchema) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InventoryItemSchema> {
        /**
         * <p>
         * The name of the inventory type. Default inventory item type names start with Amazon Web Services. Custom
         * inventory type names will start with Custom. Default inventory item types include the following:
         * <code>AWS:AWSComponent</code>, <code>AWS:Application</code>, <code>AWS:InstanceInformation</code>,
         * <code>AWS:Network</code>, and <code>AWS:WindowsUpdate</code>.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory type. Default inventory item type names start with Amazon Web Services.
         *        Custom inventory type names will start with Custom. Default inventory item types include the
         *        following: <code>AWS:AWSComponent</code>, <code>AWS:Application</code>,
         *        <code>AWS:InstanceInformation</code>, <code>AWS:Network</code>, and <code>AWS:WindowsUpdate</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory item.
         * </p>
         * 
         * @param version
         *        The schema version for the inventory item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<InventoryItemAttribute> attributes);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(InventoryItemAttribute... attributes);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.InventoryItemAttribute.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ssm.model.InventoryItemAttribute#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.InventoryItemAttribute.Builder#build()} is called
         * immediately and its result is passed to {@link #attributes(List<InventoryItemAttribute>)}.
         * 
         * @param attributes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.InventoryItemAttribute.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributes(java.util.Collection<InventoryItemAttribute>)
         */
        Builder attributes(Consumer<InventoryItemAttribute.Builder>... attributes);

        /**
         * <p>
         * The alias name of the inventory type. The alias name is used for display purposes.
         * </p>
         * 
         * @param displayName
         *        The alias name of the inventory type. The alias name is used for display purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String version;

        private List<InventoryItemAttribute> attributes = DefaultSdkAutoConstructList.getInstance();

        private String displayName;

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryItemSchema model) {
            typeName(model.typeName);
            version(model.version);
            attributes(model.attributes);
            displayName(model.displayName);
        }

        public final String getTypeName() {
            return typeName;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final List<InventoryItemAttribute.Builder> getAttributes() {
            List<InventoryItemAttribute.Builder> result = InventoryItemAttributeListCopier.copyToBuilder(this.attributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttributes(Collection<InventoryItemAttribute.BuilderImpl> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copyFromBuilder(attributes);
        }

        @Override
        public final Builder attributes(Collection<InventoryItemAttribute> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(InventoryItemAttribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(Consumer<InventoryItemAttribute.Builder>... attributes) {
            attributes(Stream.of(attributes).map(c -> InventoryItemAttribute.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        @Override
        public InventoryItemSchema build() {
            return new InventoryItemSchema(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
